import javax.swing.*;
import java.awt.*;
import java.awt.event.*;

/** An instance is a turtle in a panel within a JFrame.
 The turtle is a spot on the panel, facing a certain direction
 and holding a colored pen. You can move the turtle from place
 to place, drawing as the turtle moves. You can ask the
 turtle to pick the pen up so that moving doesn't draw. You can draw
 rectangles and circles of any size around the turtle.
 All instances of Turtle use the same JFrame and panel. */
public class Turtle {
    /** The frame on which the turtles move. */ 
    private static JFramePaint jframe;
    
    /** The panel in the JFrame. */
    private static JPanel panel; 
    
    /** The graphics associated with the panel. */
    private static Graphics graphics;
    
    /** The difference between frame and panel heights. */
    private int difference; 
    
    /** The panel on which the turtle moves size (width, height). */
    private static int width= 500;
    
    /** The panel on which the turtle moves has size (width, height). */
    private static int height= 500;
    
    /** The list of commands to execute when repainting are in
     data[0..iWrite-1] */
    private static final int ARRAY_INCREMENT=256;
    private static PaintCommand[] data= new PaintCommand[256];
    private static int iWrite= 0;
    
    /** = "window has been resized and not yet repainted" */
    private static boolean resizing= false;
    
    /** The turtle is at point (x, y). */
    private double x, y;
    
    /** The turtle points in direction angle: 0 <= angle < 360.
     0: east, 90:north, 180: west, 270 south. */
    private double angle; 
    
    /** = "the pen is down". */
    private boolean penDown= true;
    
    /** the pen color. */
    private Color turtleColor= Color.black;
    
    /** Constructor: a black turtle starting at the middle of the
     panel with angle 0 (looking east).
     East (right) is angle 0; north (up), 90; west (left), 180;
     South (down). 270. The pen is down.*/
    public Turtle() {
        this(width/2, height/2, 0);
    }
    
    /** Constructor: a black turtle starting at (x,y) with angle ang.
     East (right) is angle 0; north (up), 90; west (left), 180;
     South (down), 270. The pen is down. */
    public Turtle(double x, double y, double ang) {
        if (jframe == null) {
            panel= new JPanel();
            panel.setPreferredSize(new Dimension(width, height));
            panel.setBackground(Color.white);
            
            jframe= new JFramePaint("Turtle. After resizing, move mouse into window to redraw.");
            jframe.getContentPane().add(panel, BorderLayout.CENTER);
            jframe.pack();
            jframe.setVisible(true);
            difference= jframe.getHeight() - panel.getHeight();
            
            graphics= panel.getGraphics();
            clear();
            
            jframe.addComponentListener(jframe);
            
            jframe.addMouseListener(new MouseListener() {
                public void mouseClicked (MouseEvent me) {}
                
                /** The mouse entered the JFrame. If
                    it had been used to resize the JFrame, then its panel
                    as big as possible and repaint it. */
                public void mouseEntered (MouseEvent me) {
                    mouseReleased(me);
                }
                
                public void mousePressed (MouseEvent me) {}
                
                /** The mouse has been released inside the JFrame. If
                    it had been used to resize the JFrame, then its panel
                    as big as possible and repaint it. */
                public void mouseReleased (MouseEvent me) {
                    if(!resizing) {
                        return;
                    }
                    
                    // The JFrame has been resized. So make the panel as big
                    // as possible within the JFrame, whiten the panel, and
                    // repaint the panel.
                    setPanelSize();
                    Color save= graphics.getColor();
                    graphics.setColor(Color.white);
                    graphics.fillRect(0, 0, Turtle.width, Turtle.height);
                    graphics.setColor(save);
                    
                    jframe.repaint();  
                    resizing= false;
                    
                }
                
                public void mouseExited (MouseEvent me) {} 
            });
        }
        
        this.x= x;
        this.y= y;
        angle= ang;
    }
    
    
    /** Add command c to the list of commands to perform when painting */
    private void putData(PaintCommand c){
        if (iWrite >= data.length){
            // Create a bigger array data and copy the original data to it
            PaintCommand[] olddata= data;
            data= new PaintCommand[data.length + ARRAY_INCREMENT];
            for (int i= 0; i != iWrite; i= i+1) {
                data[i]= olddata[i];
            }
        }
        
        data[iWrite]= c;
        iWrite= iWrite + 1;
    }
    
    /** = number of commands saved (in data). */
    private static int getIWrite() {
        return iWrite;
    }
    
    /** = the array of saved commands (data). */
    private static PaintCommand[] getData() {
        return data;
    }
    
    /** = all the turtle commands carried out since the last
          clear(), one to a line. */
    public static String getCommands() {
        String res= "";
        for (int i= 0; i != iWrite; i= i+1) {
            if (i != 0) 
                res= res + "\n";
            res= res + data[i];
        }
        return res;
    }
    
    /** = the Color value for integer i. The values are as follows:
     1 cyan, 2 blue, 3 dark gray, 4 gray,
     5 green, 6 light gray, 7 magenta, 8 orange,
     9 pink, 10 red, 11 black, 12 yellow, and
     any other integer white.
     */
    public static Color intColor(int i) {
        if (i == 2) return Color.blue;
        if (i == 1) return Color.cyan;
        if (i == 3) return Color.darkGray;
        if (i == 4) return Color.gray;
        if (i == 5) return Color.green;
        if (i == 6) return Color.lightGray;
        if (i == 7) return Color.magenta;
        if (i == 8) return Color.orange;
        if (i == 9) return Color.pink;
        if (i == 10) return Color.red;
        if (i == 11) return Color.black;
        if (i == 12) return Color.yellow;
        return Color.white;
    }
    
    
    /** Set the panel size as big as possible in the current JFrame window. */
    public void setPanelSize() {
        Turtle.width= jframe.getWidth();
        Turtle.height= jframe.getHeight()-difference;
        
        Dimension d= new Dimension(Turtle.width, Turtle.height);
        panel.setPreferredSize(d);
        graphics= panel.getGraphics();
        
        jframe.pack();
        pause(100);
    }
    
    /** = x-coordinate of the turtle. */
    public double getX() {
        return x;   
    }
    
    /** = y-coordinate of the turtle. */
    public double getY() {
        return y;   
    }
    
    /** = angle of the turtle (in degrees). East (right) is angle 0;
     north (up), 90; west (left), 180; South (down), 270. */
    public double getAngle() {
        return angle;   
    }
    
    /** = width of panel. */
    public int getWidth() {
        return width;   
    }
    
    /** = height of the panel. */
    public int getHeight() {
        return height;   
    }
    
    /** = "the pen is down". */
    public boolean isPenDown() {
        return penDown;
    }
    
    
    /** Set the angle to ang degrees. East (right) is angle 0;
     north (up), 90; west (left), 180; South (down), 270.*/
    public void setAngle(double ang) {
        angle= mod(ang, 360);
    }
    
    /** Add ang degrees to the angle. */
    public void addAngle(double ang) {
        angle= mod (angle + ang, 360);
    }
    
    /** Lift pen. */
    public void putPenUp() {
        penDown= false;   
    }
    
    /** Put pen down. */
    public void putPenDown() {
        penDown= true;   
    }
    
    /** Set the color of the turtle to c. */
    public void setColor(Color c) {
        turtleColor= c;
    }
    
    /** = the turtle's current color. */
    public Color getColor() {
        return turtleColor;
    }
    
    /** Move the turtle to (x,y), without drawing,
     and face it at angle ang --who ever heard of turtles jumping? */
    public void jumpTo(double x, double y, double ang) {
        this.x= x;
        this.y= y;
        angle= ang;
    }
    
    /** Draw a circle of diameter d with center at the turtle's position. */
    public void drawCircle(double d) {
        int id= (int) Math.round(d);
        Color save= graphics.getColor();
        graphics.setColor(turtleColor);
        graphics.drawOval((int)Math.round(x-d/2),
                          (int)Math.round(y-d/2), id, id);
        graphics.setColor(save);
        
        putData(new PaintOvalCommand((int)Math.round(x-d/2),
                                     (int)Math.round(y-d/2), id, id, turtleColor));
    }
    
    /** Fill triangle with lower left point at the turtle position and side length s.
        Don't move the turtle.*/
    public void fillTriangle(double s) {
        
        int[] xpts= {(int)Math.round(x), (int)Math.round(x+s), (int)Math.round(x+s/2)};
        int[] ypts= {(int)Math.round(y), (int)Math.round(y), (int)(y - s*Math.sqrt(.75))};
        graphics.fillPolygon(xpts, ypts, 3);
        
        putData(new FillTriangleCommand(xpts, ypts, turtleColor));
    }
    
    /** Fill a circle of diameter d with center at the turtle's position. */
    public void fillCircle(double d) {
        int id= (int) Math.round(d);
        Color save= graphics.getColor();
        graphics.setColor(turtleColor);
        graphics.fillOval((int)Math.round(x-d/2),
                          (int)Math.round(y-d/2), id, id);
        graphics.setColor(save);
        
        putData(new FillOvalCommand((int)Math.round(x-d/2),
                                    (int)Math.round(y-d/2), id, id, turtleColor));
        
    }
    
    /** Draw a rectangle of width w, height h with center at the turtle's position. */
    public void drawRect(double w, double h) {
        Color save= graphics.getColor();
        graphics.setColor(turtleColor);
        graphics.drawRect((int)Math.round(x-w/2), (int)Math.round(y-h/2),
                          (int)Math.round(w), (int)Math.round(h));
        graphics.setColor(save);
        
        putData(new PaintRectCommand((int)Math.round(x-w/2), (int)Math.round(y-h/2),
                                     (int)Math.round(w), (int)Math.round(h),
                                     turtleColor));
    }
    
    /** Fill a rectangle of width w, height h with center at the turtle's position. */
    public void fillRect(double w, double h) {
        Color save= graphics.getColor();
        graphics.setColor(turtleColor);
        graphics.fillRect((int)Math.round(x-w/2), (int)Math.round(y-h/2),
                          (int)Math.round(w), (int)Math.round(h));
        graphics.setColor(save);
        
        putData(new FillRectCommand((int)Math.round(x-w/2), (int)Math.round(y-h/2),
                                    (int)Math.round(w), (int)Math.round(h),
                                    turtleColor));
    }
    
    /** Move the turtle d units in its current direction.
     If the pen is down, a line will be drawn; if the pen
     is up, it won't be drawn.*/
    public void moveAhead(double d) {
        double rAngle= (angle * Math.PI) / 180;
        double newX= x + Math.cos(rAngle) * d;
        double newY= y - Math.sin(rAngle) * d;
        if (penDown) {
            Color save= graphics.getColor();
            graphics.setColor(turtleColor);
            
            graphics.drawLine((int)Math.round(x), (int)Math.round(y),
                              (int)Math.round(newX), (int)Math.round(newY));
            graphics.setColor(save);
            
            putData(new PaintLineCommand((int)Math.round(x), (int)Math.round(y),
                                         (int)Math.round(newX), (int)Math.round(newY),
                                         turtleColor));
        }
        x= newX;
        y= newY;
    }
    
    /** Move the turtle d units in the opposite (backward) direction.
     If the pen is down, a line will be drawn; if the pen
     is up, it won't be drawn.*/
    public void moveBack(double d) {
        double save= angle;
        addAngle(180);
        moveAhead(d);
        angle= save;
    }
    
    /** = x mod y. Precondition: y != 0. */
    private double mod(double x, double y) {
        double ans= x % y;
        if (ans < 0) 
            ans= ans + y;
        return ans;
    }
    
    /** Clear the screen (make it all white). */
    public void clear() {
        Color save= graphics.getColor();
        graphics.setColor(Color.white);
        graphics.fillRect(0, 0, width, height);
        graphics.setColor(save);
        iWrite= 0;
        putData(new Clear());
    }
    
    /** Clear the screen (make it all white) but don't remove commands. */
    private void clearLeaveCommands() {
        Color save= graphics.getColor();
        graphics.setColor(Color.white);
        graphics.fillRect(0, 0, width, height);
        graphics.setColor(save);
    }
    
    /** Pause the program for millisec milliseconds. */
    public void pause(int millisec) {
        try { Thread.currentThread().sleep(millisec); }
        catch (InterruptedException e) { }
    }   
    
    
    /** An instance is a JFrame with a listener and a procedure to paint. */
    private class JFramePaint extends JFrame implements ComponentListener {
        
        /** Constructor: an instance with title t */
        public JFramePaint(String t) {
            super(t);
        }
        
        /** Using g, paint the commands in data[0..iWrite-1] */
        public void paint(Graphics g) {
            for (int i= 0; i != Turtle.getIWrite(); i= i+1) {
                Turtle.getData()[i].exec(g);
            }
        }
        
        /** The JFrame has become visible, so repaint it. */
        public void componentShown(ComponentEvent e) {
            repaint();
        }
        
        public void componentHidden(ComponentEvent e) {  
        }
        
        /** The resize button for the JFrame is being dragged, so set
            static variable resizing to true. */
        public void componentResized(ComponentEvent e) {
            //System.out.println("componentResized called");
            resizing= true;
        }
        
        public void componentMoved(ComponentEvent e) {   
        }
        
        
        
    }
    
    
    
    
    /** An instance of a subclass is a command to paint something. */
    private abstract class PaintCommand {
        /** (x1,y1) and (x2,y2) are usually the "beginning" 
         and "end" coordinates of what to paint */
        int x1, y1, x2, y2;
        
        Color c; // The color with which to paint;
        
        /** Execute this command to paint, using g*/
        public abstract void exec(Graphics g);
        
        /** = the command c with its parameters */
        public String toString(String c) {
            return c + "(" + x1 + ", " + y1 + ", " +
                + x2 + ", " + y2 + " ), color " + c;
        }
    }
    
    /** An instance is a command to paint a line */
    private class PaintLineCommand extends PaintCommand {
        
        /** Constructor: a command to draw a line from (x1, y1)
         to (x2, y2) with color c*/
        public PaintLineCommand(int x1, int y1, int x2, int y2, Color c) {
            this.x1= x1; this.y1= y1; this.x2= x2; this.y2= y2;
            this.c= c;
        }
        
        /** Execute this command to paint a line from (x1,y1)
         to (x2,y2) in color c, using g*/
        public void exec(Graphics g) {
            Color save= g.getColor();
            g.setColor(c);
            g.drawLine(x1, y1, x2, y2);
            g.setColor(save);
        }
        
        /** = a representation of this command */
        public String toString() {
            return toString("drawLine");
        }
    }
    
    /** An instance is a command to paint a Rectangle */
    private class PaintRectCommand extends PaintCommand {
        
        /** Constructor: a paint a rectangle in color c with
         upper-left corner (x1, y1) with size (x2, y2) */
        public PaintRectCommand(int x1, int y1, int x2, int y2, Color c) {
            this.x1= x1; this.y1= y1; this.x2= x2; this.y2= y2;
            this.c= c;
        }
        
        /** Execute this command to paint a rectangle in color c with
         upper-left corner (x1, y1) with size (x2, y2), using g*/
        public void exec(Graphics g) {
            Color save= g.getColor();
            g.setColor(c);
            g.drawRect(x1, y1, x2, y2);
            g.setColor(save);
        }
        
        /** = a representation of this command */
        public String toString() {
            return toString("drawRect");
        }
        
    }
    
    /** An instance is a command to fill a Rectangle */
    private class FillRectCommand extends PaintCommand {
        
        /** Constructor: fill a rectangle in color c with
         upper-left corner (x1, y1) with size (x2, y2) */
        public FillRectCommand(int x1, int y1, int x2, int y2, Color c) {
            this.x1= x1; this.y1= y1; this.x2= x2; this.y2= y2;
            this.c= c;
        }
        
        /** Execute this command to fill a rectangle in color c with
         upper-left corner (x1, y1) with size (x2, y2), using g*/
        public void exec(Graphics g) {
            Color save= g.getColor();
            g.setColor(c);
            g.fillRect(x1, y1, x2, y2);
            g.setColor(save);
        }
        
        /** = a representation of this command */
        public String toString() {
            return toString("fillRect");
        }
        
    }
    
    /** An instance is a command to clear the panel (paint it white) */
    private class Clear extends PaintCommand {
        
        /** Constructor: fill a rectangle in color c with
         upper-left corner (x1, y1) with size (x2, y2) */
        public Clear() {
        }
        
        /** Execute this command to fill a rectangle in color c with
         upper-left corner (x1, y1) with size (x2, y2), using g*/
        public void exec(Graphics g) {
            Color save= g.getColor();
            g.setColor(Color.white);
            g.fillRect(0, 0,Turtle.width, Turtle.height);
            g.setColor(save);
        }
        
        /** = a representation of this command */
        public String toString() {
            return "Clear";
        }
        
    }
    
    /** An instance is a command to fill a triangle */
    private class FillTriangleCommand extends PaintCommand {
        /**The vertices of the triangle are given by xpts and ypts */
        int[] xpts;
        int[] ypts;
        
        
        /** Constructor: a triangle in color c given by points in xpts and ypts.
           Precondition: the two arrays are of length 3.*/
        public FillTriangleCommand(int[] xpts, int[] ypts, Color c) {
            this.xpts= new int[3];
            this.xpts[0]= xpts[0];
            this.xpts[1]= xpts[1];
            this.xpts[2]= xpts[2];
            this.ypts= new int[3];
            this.ypts[0]= ypts[0];
            this.ypts[1]= ypts[1];
            this.ypts[2]= ypts[2];
            this.c= c;
        }
        
        /** Execute this command to fill a triangle given by xpts and
            ypts in color c, using g*/
        public void exec(Graphics g) {
            Color save= g.getColor();
            g.setColor(c);
            g.fillPolygon(xpts, ypts, 3);
            g.setColor(save);
        }
        
        /** = a representation of this command */
        public String toString() {
            return toString("fillTriangle");
        }
    }
    
    /** An instance is a command to paint an Oval */
    private class PaintOvalCommand extends PaintCommand {
        
        /** Constructor: a paint an Oval in color c with
         upper-left corner (x1, y1) with size (x2, y2) */
        public PaintOvalCommand(int x1, int y1, int x2, int y2, Color c) {
            this.x1= x1; this.y1= y1; this.x2= x2; this.y2= y2;
            this.c= c;
        }
        
        /** Execute this command to paint an Oval in color c with
         upper-left corner (x1, y1) with size (x2, y2), using g*/
        public void exec(Graphics g) {
            Color save= g.getColor();
            g.setColor(c);
            g.drawOval(x1, y1, x2, y2);
            g.setColor(save);
        }
        
        /** = a representation of this command */
        public String toString() {
            return toString("drawOval");
        }
        
    }
    
    /** An instance is a command to fill an Oval */
    private class FillOvalCommand extends PaintCommand {
        
        /** Constructor: fill an Oval in color c with
         upper-left corner (x1, y1) with size (x2, y2) */
        public FillOvalCommand(int x1, int y1, int x2, int y2, Color c) {
            this.x1= x1; this.y1= y1; this.x2= x2; this.y2= y2;
            this.c= c;
        }
        
        /** Execute this command to fill an Oval in color c with
         upper-left corner (x1, y1) with size (x2, y2), using g*/
        public void exec(Graphics g) {
            Color save= g.getColor();
            g.setColor(c);
            g.fillOval(x1, y1, x2, y2);
            g.setColor(save);
        }
        
        /** = a representation of this command */
        public String toString() {
            return toString("fillOval");
        }
        
    }
    
}

