import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

/** An instance of this class shows a color and its complement as well
 as sliders for the RGB, HSV, and CMYK color models. Changing
 any of the sliders changes the color (and the other sliders appropriately).*/
public class A4 implements ChangeListener {
    /** the JFrame that is the GUI. */
    private JFrame jf= null;
    
    /** Panels that contain the color and its complement, and
     labels that contain information in them. */
    JPanel image, compImage;
    JLabel imageLabel, compImageLabel;
    
    /** Below are sliders. In each constructor call, the first argument
     makes it a vertical slider. The next three arguments are the
     minimum value, maximum value, and initial value of the slider.
     
     /** RGB sliders. */
    JSlider RSlider= new JSlider(JSlider.VERTICAL, 0, 25500, 25500);
    JSlider GSlider= new JSlider(JSlider.VERTICAL, 0, 25500, 0);
    JSlider BSlider= new JSlider(JSlider.VERTICAL, 0, 25500, 0);
    
    /** HSV sliders. */
    JSlider HSlider= new JSlider(JSlider.VERTICAL, 0, 35999, 0);
    JSlider SSlider= new JSlider(JSlider.VERTICAL, 0, 100, 100);
    JSlider VSlider= new JSlider(JSlider.VERTICAL, 0, 100, 100);
    
    /** CMYK sliders. */
    JSlider CSlider= new JSlider(JSlider.VERTICAL, 0, 100, 0);
    JSlider MSlider= new JSlider(JSlider.VERTICAL, 0, 100, 100);
    JSlider YSlider= new JSlider(JSlider.VERTICAL, 0, 100, 100);
    JSlider KSlider= new JSlider(JSlider.VERTICAL, 0, 100, 0);
    
    /** Create and show a GUI that contains:
     (0) a color panel, with text to describe the color.
     (1) the complementary color panel, with text to describe it.
     (2) Sliders for RGB, HSV, and CMYK.
     Initially, the color is red. */
    public static void main(String[] args) {
        A4 a4= new A4();
    }
    
    /** Constructor: a GUI that contains:
     (0) a color panel, with text to describe the color.
     (1) the complementary color panel, with text to describe it.
     (2) Sliders for RGB, HSV, and CMYK.
     Initially, the color is red. */
    public A4() {
        jf= new JFrame("Color");
        jf.setSize(600, 200);
        jf.setResizable(false);
        
        // Create color red in three models.
        Color rgb= new Color(255, 0, 0);
        HSV hsv= A4Methods.RGBtoHSV(rgb);
        CMYK cmyk= A4Methods.RGBtoCMYK(rgb);
        
        // Create panel image to contain the color.
        image= new JPanel();
        image.setSize(200, 200);
        imageLabel= new JLabel("");
        imageLabel.setPreferredSize(new Dimension(200, 160));
        imageLabel.setVerticalAlignment(JLabel.TOP);
        image.add(imageLabel);
        
        // Create panel compImage to contain the complementary color.
        compImage= new JPanel();
        compImage.setSize(200, 200);
        compImageLabel= new JLabel("");
        compImageLabel.setPreferredSize(new Dimension(200, 160));
        compImageLabel.setVerticalAlignment(JLabel.TOP);
        compImage.add(compImageLabel);
        
        // Create each of the sliders and add them to Box sliders
        // with some space between the three sets of sliders.
        Box sliders= new Box(BoxLayout.X_AXIS);
        
        fixSlider(sliders, RSlider, 'R');
        fixSlider(sliders, GSlider, 'G');
        fixSlider(sliders, BSlider, 'B');
        
        sliders.add(Box.createRigidArea(new Dimension(20,0)));
        
        
        fixSlider(sliders, CSlider, 'C');
        fixSlider(sliders, MSlider, 'M');
        fixSlider(sliders, YSlider, 'Y');
        fixSlider(sliders, KSlider, 'K');
        
        sliders.add(Box.createRigidArea(new Dimension(20,0)));
        
        fixSlider(sliders, HSlider, 'H');
        fixSlider(sliders, SSlider, 'S');
        fixSlider(sliders, VSlider, 'V');
        
        
        
        // Add to Box GUI the color and complementary color panels and
        // the Box of sliders
        Box GUI= new Box(BoxLayout.X_AXIS);
        GUI.add(image);
        GUI.add(compImage);
        GUI.add(sliders);
        
        // Set the initial values of the sliders.
        setSliders(rgb, hsv, cmyk);
        
        // Set the color and complementary color panels
        setColorPanels(rgb, hsv, cmyk);
        
        jf.getContentPane().add(GUI);
        jf.pack();
        jf.setVisible(true);
    }
    
    /** Create a Box with slider s and c above it
     and add the Box to sliders.
     Add this object as a changeListener for s.*/
    public void fixSlider(Box sliders, JSlider s, char c) {
        Box b= new Box(BoxLayout.Y_AXIS);
        b.add(new JLabel("" + c));
        b.add(s);
        sliders.add(b);
        s.addChangeListener(this);
    }
    
    /** Process a movement in one of the sliders */
    public void stateChanged(ChangeEvent e) {
        Color rgb;
        HSV hsv;
        CMYK cmyk;
        
        // Remove listeners so that user changes during
        // execution of this method will have no effect.
        RSlider.removeChangeListener(this);
        GSlider.removeChangeListener(this);
        BSlider.removeChangeListener(this);
        HSlider.removeChangeListener(this);
        SSlider.removeChangeListener(this);
        VSlider.removeChangeListener(this);
        CSlider.removeChangeListener(this);
        MSlider.removeChangeListener(this);
        YSlider.removeChangeListener(this);
        KSlider.removeChangeListener(this);
        
        JSlider source= (JSlider)e.getSource();
        
        if (source.equals(HSlider) || source.equals(SSlider) ||
            source.equals(VSlider)) {
            double H= (double)HSlider.getValue() / 100.0;
            double S= (double)SSlider.getValue() / 100.0;
            double V= (double)VSlider.getValue() / 100.0;
            hsv= new HSV(H, S, V);
            rgb= A4Methods.HSVtoRGB(hsv);
            cmyk= A4Methods.RGBtoCMYK(rgb);
        }
        else if (source.equals(RSlider) || source.equals(GSlider) ||
                 source.equals(BSlider)) {
            int R= (int) Math.round(RSlider.getValue() / 100.0);
            int G= (int) Math.round(GSlider.getValue() / 100.0);
            int B= (int) Math.round(BSlider.getValue() / 100.0);
            rgb= new Color(R, G, B);
            hsv= A4Methods.RGBtoHSV(rgb);
            cmyk= A4Methods.RGBtoCMYK(rgb);
        }
        else if (source.equals(CSlider) || source.equals(MSlider) ||
                 source.equals(YSlider) || source.equals(KSlider)) {
            double C= CSlider.getValue() / 100.0;
            double M= MSlider.getValue() / 100.0;
            double Y= YSlider.getValue() / 100.0;
            double K= KSlider.getValue() / 100.0;
            cmyk= new CMYK(C, M, Y, K);
            rgb= A4Methods.CMYKtoRGB(cmyk);
            hsv= A4Methods.RGBtoHSV(rgb);
        }
        else return;  // event was not a change in a slider
        
        setSliders(rgb, hsv, cmyk);
        setColorPanels(rgb, hsv, cmyk);
        
        // Add the listeners for the sliders
        RSlider.addChangeListener(this);
        GSlider.addChangeListener(this);
        BSlider.addChangeListener(this);
        HSlider.addChangeListener(this);
        SSlider.addChangeListener(this);
        VSlider.addChangeListener(this);
        CSlider.addChangeListener(this);
        MSlider.addChangeListener(this);
        YSlider.addChangeListener(this);
        KSlider.addChangeListener(this);
    }
    
    /** Set the RGB, HSV, CMYK sliders based on rgb, hsv, and cmyk. */
    public void setSliders(Color rgb, HSV hsv, CMYK cmyk) {
        RSlider.setValue(rgb.getRed() * 100);
        GSlider.setValue(rgb.getGreen() * 100);
        BSlider.setValue(rgb.getBlue() * 100);
        
        HSlider.setValue((int)(hsv.h * 100.0));
        SSlider.setValue((int)(hsv.s * 100.0));
        VSlider.setValue((int)(hsv.v * 100.0));
        
        CSlider.setValue((int)(cmyk.c * 100.0));
        MSlider.setValue((int)(cmyk.m * 100.0));
        YSlider.setValue((int)(cmyk.y * 100.0));
        KSlider.setValue((int)(cmyk.k * 100.0));
    }
    
    /** Set background colors and the text in the color window and
     complementary color window, based on rgb, hsv, and cmyk. Put
     the rgb in the title.*/
    public void setColorPanels(Color rgb, HSV hsv, CMYK cmyk) {
        image.setBackground(rgb);
        Color compRgb= A4Methods.complementRGB(rgb);
        compImage.setBackground(compRgb);
        
        imageLabel.setForeground(compRgb);
        imageLabel.setText("<html>Color<br>RGB: " +
                           A4Methods.toString(rgb) + "<br>HSV: " +
                           A4Methods.toString(hsv) + "<br>CMYK: " +
                           A4Methods.toString(cmyk) + "<br><br>" +
                           "RGB slider ranges: 0..255.<br>" +
                           "H slider range: 0..360 degrees.<br>" + 
                           "Other slider ranges: 0..1." + "</html>" );
        
        compImageLabel.setForeground(rgb);
        HSV compHsv= A4Methods.RGBtoHSV(compRgb);
        CMYK compCmyk= A4Methods.RGBtoCMYK(compRgb);
        compImageLabel.setText("<html>Complementary Color<br>RGB: " +
                               A4Methods.toString(compRgb) +
                               "<br>HSV: " + 
                               A4Methods.toString(compHsv) + "<br>CMYK: " +
                               A4Methods.toString(compCmyk) + "</html>");
        jf.setTitle("Color RGB: " + A4Methods.toString(rgb));
    }
}