#include "cTronTest.h"
#include "Protocol Stack/cProtocolStack.h"
#include "Protocol Stack/cBufferManager.h"
#include "Endpoint/cEndpointFactory.h"
#include "Util\gError.h"
#include "Network Layer/cNetworkLayer.h"
#include <stdio.h>
#include <stdlib.h>
#include <winbase.h>
#include <winuser.h>
#include <commctrl.h>
#include "types.h"

extern TronTestHandles gHandle;
extern TronTestParam   gParam;
extern bool			   gui;

//#define START_INTERVAL 15
#define START_INTERVAL 10
#define REFRESH_INTERVAL 500

#ifndef ABOVE_NORMAL_PRIORITY_CLASS
#define ABOVE_NORMAL_PRIORITY_CLASS 32768
#endif

/*
 * cTrontest::ViewHandler()
 *
 * Purpose:	Receives a new view clears all of the node informations.
 * IN:		view		-> The new view.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The mStatus array is changed; mReady is set to true
 * Return:	true if success, else false.
 */
bool cTrontest::ViewHandler(cViewIterator* view)
{
	cEndpoint* ep;
	unsigned int i;
	char	buffer[10];
	DWORD	now = cProtocolStack::GetTime();
	int		size = view->GetNumMembers();

	_itoa(now, buffer, 10);
	SetWindowText(gHandle.hLastViewTime, buffer);
	_itoa(size, buffer, 10);
	SetWindowText(gHandle.hLastViewSize, buffer);

	mCS.Enter();

	if(mStatus)
	{
		for(i = 0; i < mNumNodes; i++)
		{
			mStatus[i].ep->Release();
		}
		delete mStatus;
	}

	mStatus = new NodeStatus[size];
	memset(mStatus, 0, sizeof(NodeStatus)*size);

	// Fill in the view.
	mNumNodes = size;
	i = 0;
	while(!view->Done())
	{
		ep = view->GetMember();
		if(ep->Equals(mMyEp))
		{
			mMyIndex = i;
		}
		mStatus[i].ep = ep;
		i++;
		ep->AddRef();
		view->GetNext();
	}

	if(mCurrentIndex > mNumNodes-1)
	{
		mCurrentIndex = mNumNodes-1;
		_WPrintNode();
		_UpdateSlider(mStatus[mCurrentIndex].sendRate);
	}
	mReady = true;

	mCS.Leave();
	return true;
}

/*
 * cTrontest::Deliver()
 *
 * Purpose:	Receives any incoming message.
 * Note:	Doesn't need a CS because runs in same thread as the View Handler.
 * IN:		sender			-> the message sender ep
 *			buffer			-> the message
 *			messageType		-> type of message, IGNORED
 * Return:	true if success, else false.
 */
bool cTrontest::Deliver(cEndpoint* sender, cMsgBuffer* buffer, int messageType)
{
	bool	 found = false;
	bool	 apply = false;
	char	 buf[10];
	TestMsg* msg;
	unsigned int i;
	int	  size;

	if(!mReady)
	{
		return true;
	}

	if(messageType == MSG_TYPE_NULL)
	{
		return true;
	}

	mNumReceived++;
	buffer->GetPayload((void **)&msg, &size);
	if(msg->type == TEST_MSG_NULL)
	{
		return true;	// ignore test messages.
	}
	if(!(msg->type & TEST_MSG_SINGLE))
	{
		apply = true;
	}
	else if(msg->data.values.hashCode == mMyEp->HashCode())
	{
		apply = true;
	}
	msg->type = msg->type & ~(TEST_MSG_SINGLE);
	switch(msg->type)
	{
		case TEST_MSG_STATUS:
			for(i = 0; i < mNumNodes; i++)
			{
				if(mStatus[i].ep->Equals(sender))
				{
					found = true;
					break;
				}
			}
			if(found)
			{
				mStatus[i].pbcastParam = msg->data.status.pbcastParam;
				mStatus[i].sendRate    = msg->data.status.sendRate;
				mStatus[i].recvRate		= msg->data.status.recvRate;
				mStatus[i].status	   = msg->data.status.status;
			}
			break;

		case TEST_MSG_SEND_RATE:
			if(apply)
			{
				if(msg->data.values.sendRate != mMyStatus.sendRate)
				{
					if(msg->data.values.sendRate == 0)
					{
						msg->data.values.sendRate = 1;
					}
					mMyStatus.sendInterval = 1000/msg->data.values.sendRate;
					mMyStatus.sendRate = msg->data.values.sendRate;
					if(!gui)
					{
						SetWindowText(gHandle.hSendRate, itoa(mMyStatus.sendRate, buf, 10));
					}
				}
			}
			break;

		case TEST_MSG_KILL:
			if(apply)
			{
				mMyStatus.status = STATUS_DEAD;
				mMyStatus.timeToNextEvent = cProtocolStack::GetTime() + 300;
			}
			break;

		case TEST_MSG_SLEEP:
			for(i = 0; i < mNumNodes; i++)
			{
				if(mStatus[i].ep->HashCode() == msg->data.values.hashCode)
				{
					mStatus[i].status = STATUS_SLEEP;
				}
			}
			if(apply)
			{
				//Sleep(mMyStatus.pbcastParam.mTimeToFailNode - 300);
				Sleep(3000);
				mStatus[i].status = STATUS_ALIVE;
			}
			break;

		default:
			if(gui)
			{
				MessageBox(NULL, "Unknown message received.", "Deliver Callback", MB_OK);
			}
	}
	return true;
}


/*
 * cTrontest::ErrorCallback()
 *
 * Purpose:	Receives any asynch errors.
 * Note:	Doesn't need a CS because runs in same thread as the View Handler.
 * IN:		param			-> Asynch error param, if there is one.
 *			type			-> Type of error.
 * Return:	true if success, else false.
 */
bool cTrontest::ErrorCallback(cObject* param, unsigned int type)
{
	DWORD	now = cProtocolStack::GetTime();

	switch(type)
	{
		case ASYNCH_ERROR_EP_FAIL:
			if(gui)
			{
				MessageBox(NULL, "Asynch Error: Endpoint fail.", "ASYNCH ERROR", MB_OK);
			}
			break;

		case ASYNCH_ERROR_MSG_GAP:
			mLastGapTime = cProtocolStack::GetTime();
			break;

		case ASYNCH_ERROR_SND_FAIL:
			MessageBox(NULL, "Asynch Error: Send fail.", "ASYNCH ERROR", MB_OK);
			break;

		default:
			MessageBox(NULL, "Asynch Error: Unknown.", "ASYNCH ERROR", MB_OK);
			break;
	}
	return true;
}

/*
 * cTrontest::Deliver()
 *
 * Purpose:	Main loop.  Updates display and sends pbcasts at requested interval.
 * Note:	Doesn't need to guard CS as long as _WPrintNode() does!
 * Return:	true if success, else false.
 */
void cTrontest::RunTest()
{
	DWORD	timeToRefresh = 0;
	DWORD   timeForFullUpdate = 0;
	mRun = true;	// Start test.
	mReady = false;
	memset(&mMyStatus, 0, sizeof(NodeStatus));
	memset(&mOldParam, 0, sizeof(PbcastParams));
	DWORD	mOldLastGapTime = mLastGapTime = 0;
	unsigned int oldNumReceived = 0;
	unsigned int oldNetRecvAmt = 0;

	// Slider init
	mSliderWait = false;
	mOldSliderVal = 0;

	if(!_ParseParameters())
	{
		gAbort("Unable to parse command line parameters.", __LINE__, __FILE__);
	}
	if(!_Setup())
	{
		gAbort("Unable to set up protocol stack.", __LINE__, __FILE__);
	}

	cGroup* temp;
	cIterator* iter;
	temp = mStack->GetLocalAddress();
	iter = temp->GetIterator();
	mMyEp = (cEndpoint *)iter->GetData();

	mMyStatus.sendInterval = 1000;	// One msg per second.

	// Start the stack
	mStack->Start();

	DWORD now, min;
	mMyStatus.timeToNextEvent = cProtocolStack::GetTime();
	while(mRun)
	{
		do
		{
			now = cProtocolStack::GetTime();
			min = __min(timeToRefresh, mMyStatus.timeToNextEvent);
			if(min > now)
			{
				Sleep( min-now );
			}

		} while(min > now);

		if(now > mMyStatus.timeToNextEvent)
		{

			switch(mMyStatus.status)
			{
				case STATUS_ALIVE:
					if(mMyStatus.sendInterval)
					{
						if(now > timeForFullUpdate)
						{
							PbcastParams*	param = mPbcastLayer->ReportStats();
							TestMsg	msg;
							msg.type = TEST_MSG_STATUS;
							memcpy(&msg.data.status.pbcastParam, param, sizeof(PbcastParams));
							msg.data.status.sendRate = mMyStatus.sendRate;
							msg.data.status.recvRate = mMyStatus.recvRate;
							msg.data.status.status	 = mMyStatus.status;
							mStack->Send(NULL, (char *)&msg, sizeof(msg));
							mMyStatus.timeToNextEvent += mMyStatus.sendInterval;
							timeForFullUpdate = now + REFRESH_INTERVAL;
						}
						else
						{
							char type = TEST_MSG_NULL;
							mStack->Send(NULL, (char *)&type, sizeof(type));
							mMyStatus.timeToNextEvent += mMyStatus.sendInterval;
						}
					}
					break;

				case STATUS_DEAD:
					Sleep(1000);	// Give pbcast a chance to disseminate death msg.
					SendNotifyMessage(gHandle.hMain, WM_COMMAND, WM_DESTROY, 0);
					mMyStatus.status = STATUS_DEFUNCT;
					break;
			}
		}
		if(now > timeToRefresh)
		{
			mMyStatus.recvRate = ((mNumReceived - oldNumReceived)*1000) / REFRESH_INTERVAL;
			oldNumReceived = mNumReceived;

			PbcastParams*	temp = mPbcastLayer->ReportStats();
			mNetRecvRate = ((temp->mPbcastsRecvd - oldNetRecvAmt)*1000) / REFRESH_INTERVAL;
			oldNetRecvAmt = temp->mPbcastsRecvd;

			char buffer[10];
			if(mOldLastGapTime != mLastGapTime)
			{
				_itoa(mLastGapTime, buffer, 10);
				SetWindowText(gHandle.hLastGapTime, buffer);
				mOldLastGapTime = mLastGapTime;
			}
			if(gui)
			{
				_itoa(now, buffer, 10);
				SetWindowText(gHandle.hTime, buffer);
				_WPrintNode();
			}
			timeToRefresh = now + REFRESH_INTERVAL;
		}
	}
	mStack->Stop();

	if(!_Shutdown())
	{
		gAbort("Unable to shut down properly.", __LINE__, __FILE__);
	}
}


/*
 * cTrontest::_ParseParameters()
 *
 * Purpose:	Parses the command line parameters.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	All of the *LayerParams should be set up.
 * Throws:	-
 * Return:	true if success, else false.
 */
bool cTrontest::_ParseParameters()
{
	mNetParam.ResetToDefault();
	mDebugParam.ResetToDefault();
	mPbcastParam.ResetToDefault();

	// Handle debug layer parameters
	double percentOut = gParam.mPercentOutDrop;
	double percentIn  = gParam.mPercentInDrop;
	u_long max = RAND_MAX;
	mDebugParam.mProbDropIncomingMsg = (unsigned int) ((double)max * percentIn);
	mDebugParam.mProbDropOutgoingMsg = (unsigned int) ((double)max * percentOut);

	// Handle Net Layer params.
	cEndpoint* mcastEp;
	mcastEp = cEndpointFactory::AllocIPEndpoint("224.0.0.69", mNetParam.mMulticastPort);
	if(!mcastEp)     { return false; }
	mNetParam.mMulticastGroup = (cIPEndpoint *)mcastEp;
	mNetParam.mNetworkOptions = cNetworkLayerParam::UNRELIABLE_BROADCAST | cNetworkLayerParam::UNRELIABLE_MULTICAST;

	// Handle pbcast layer params.
	mPbcastParam.mMsgQueueSize    = 10*mPbcastParam.mMsgQueueSize;
	mPbcastParam.mMsgQueueGrowInc = 10*mPbcastParam.mMsgQueueGrowInc;
	mPbcastParam.mMulticastEndpoint = mcastEp;
	cEndpoint* ep;
	cGroup*	   subnetGroup = new cGroup(gParam.mNumSubnets);
	if(!subnetGroup) { return false; }
	for(unsigned int i = 0; i < gParam.mNumSubnets; i++)
	{
		ep = cEndpointFactory::AllocIPEndpoint(inet_ntoa(gParam.mSubnetList[i].sin_addr), mNetParam.mMulticastPort);
		if(!ep) { return false; }
		subnetGroup->AddEndpoint(ep);
	}
	mPbcastParam.mSubnetList = subnetGroup;
	return true;
}

/*
 * cTrontest::_Setup()
 *
 * Purpose:	Sets up all of the protocol stacks.  Assumes that the *LayerParams are set up.
 * IN:		-
 * OUT:		-
 * Cond:	All of the *LayerParams should be set up.
 * PostCnd:	-
 * Throws:	-
 * Return:	true if success, else false.
 */
bool cTrontest::_Setup()
{
	cHandle				handle;
	cHandle				viewHandle;
	cHandle				errorHandle;

	mCurrentIndex	= 0;
	mNumNodes		= 0;

	// Alloc all of the protocol stacks, layers, etc.
	mStack		  = new cProtocolStack("");
	mNetLayer	  = new cNetworkLayer();
	mDebugLayer   = new cDebugLayer();
	mPbcastLayer  = new cPbcastLayer();
	mStatus		  = NULL;
	if(!mStack || !mNetLayer || !mDebugLayer || !mPbcastLayer) { return false; }

	// Build the stack.
	if(!(mStack->AddLayer(mNetLayer,	&mNetParam)) ||
//	   !(mStack->AddLayer(mDebugLayer,  &mDebugParam)) ||
	   !(mStack->AddLayer(mPbcastLayer, &mPbcastParam)) )
	{
		return false;
	}
	if(!(mStack->RegisterErrorCallback(&errorHandle, this)) ||
	   !(mStack->RegisterDeliverCallback(&handle, this))    ||
	   !(mStack->RegisterViewCallback(&viewHandle, this)) )
	{
		return false;
	}
	return true;
}

/*
 * cTrontest::_Shutdown()
 *
 * Purpose:	Shuts down the protocol stack and frees everything.
 * IN:		-
 * OUT:		-
 * Cond:	All of the protocol stacks should be set up.
 * PostCnd:	-
 * Throws:	-
 * Return:	true if success, else false.
 */
bool cTrontest::_Shutdown()
{
	// Free up extra endpoints.
	mPbcastParam.mSubnetList->Clear();
	delete mPbcastParam.mSubnetList;
	mStack->Cleanup();
	delete mStack;
	delete mNetLayer;
	delete mDebugLayer;
	delete mPbcastLayer;
	if(mStatus)
	{
		delete mStatus;
	}
	return true;
}

void cTrontest::_WPrintNode(unsigned int index)
{
	if(mCurrentIndex == index)
	{
		_WPrintNode();
	}
}

/*
 * cTrontest::_WPrintNode()
 *
 * Purpose:	Updates the current node display.
 * Note:	Must guard the CS!
 * Return:	true if success, else false.
 */
void cTrontest::_WPrintNode()
{
	if(!mReady)
	{
		return;
	}
	mCS.Enter();

	unsigned char status  = mStatus[mCurrentIndex].status;
	PbcastParams	param = mStatus[mCurrentIndex].pbcastParam;
	long value	          = mStatus[mCurrentIndex].sendRate;
	unsigned int recvRate;

	if(mCurrentIndex == mMyIndex)
	{
		recvRate = mMyStatus.recvRate;
	}
	else
	{
		recvRate = mStatus[mCurrentIndex].recvRate;
	}
	char	buffer[10];

	mCS.Leave();

	if(mCurrentIndex == mMyIndex)
	{
		SendNotifyMessage(gHandle.hLocalNodeCheck, BM_SETCHECK, (WPARAM)BST_CHECKED, 0);
	}
	else
	{
		SendNotifyMessage(gHandle.hLocalNodeCheck, BM_SETCHECK, (WPARAM)BST_UNCHECKED, 0);
	}

	if(status == STATUS_ALIVE)
	{
		SetWindowText(gHandle.hAlive, "alive");
	}
	else if(status == STATUS_DEAD)
	{
		SetWindowText(gHandle.hAlive, "dead");
	}
	else
	{
		SetWindowText(gHandle.hAlive, "sleeping");
	}

	_itoa(mCurrentIndex, buffer, 10);
	SetWindowText(gHandle.hCurrentNode, buffer);

	_itoa(recvRate, buffer, 10);
	SetWindowText(gHandle.hRecvRate, buffer);

	_itoa(mNetRecvRate, buffer, 10);
	SetWindowText(gHandle.hNetRecvRate, buffer);

	if(param.mViewSize != mOldParam.mViewSize)
	{
		_itoa(param.mViewSize, buffer, 10);
		SetWindowText(gHandle.hViewSize, buffer);
	}

	if(param.mPbcastsRecvd != mOldParam.mPbcastsRecvd)
	{
		_itoa(param.mPbcastsRecvd, buffer, 10);
		SetWindowText(gHandle.hPbcastIn, buffer);
	}
	if(param.mSubnetListSize != mOldParam.mSubnetListSize)
	{
		_itoa(param.mSubnetListSize, buffer, 10);
		SetWindowText(gHandle.hSubnetSize, buffer);
	}

	if(param.mGossipInterval != mOldParam.mGossipInterval)
	{
		_itoa(param.mGossipInterval, buffer, 10);
		SetWindowText(gHandle.hGossipInterval, buffer);
	}

	if(param.mFanout != mOldParam.mFanout)
	{
		_itoa(param.mFanout, buffer, 10);
		SetWindowText(gHandle.hFanout, buffer);
	}

	if(param.mTimeToFailNode != mOldParam.mTimeToFailNode)
	{
		_itoa(param.mTimeToFailNode, buffer, 10);
		SetWindowText(gHandle.hTimeToFail, buffer);
	}

	if(param.mNumRetansReq != mOldParam.mNumRetansReq)
	{
		_itoa(param.mNumRetansReq, buffer, 10);
		SetWindowText(gHandle.hRetranReq, buffer);
	}

	if(param.mNumRetransRecvd != mOldParam.mNumRetransRecvd)
	{
		_itoa(param.mNumRetransRecvd, buffer, 10);
		SetWindowText(gHandle.hRetranRec, buffer);
	}

	if(param.mFullGossipsSent != mOldParam.mFullGossipsSent)
	{
		_itoa(param.mFullGossipsSent, buffer, 10);
		SetWindowText(gHandle.hFullGosOut, buffer);
	}

	if(param.mMinGossipsSent != mOldParam.mMinGossipsSent)
	{
		_itoa(param.mMinGossipsSent, buffer, 10);
		SetWindowText(gHandle.hMinGosOut, buffer);
	}

	if(param.mFullGossipsReceived != mOldParam.mFullGossipsReceived)
	{
		_itoa(param.mFullGossipsReceived, buffer, 10);
		SetWindowText(gHandle.hFullGosIn, buffer);
	}

	if(param.mMinGossipsReceived != mOldParam.mMinGossipsReceived)
	{
		_itoa(param.mMinGossipsReceived, buffer, 10);
		SetWindowText(gHandle.hMinGosIn, buffer);
	}

	if(param.mPbcastsSent != mOldParam.mPbcastsSent)
	{
		_itoa(param.mPbcastsSent, buffer, 10);
		SetWindowText(gHandle.hPbcastOut, buffer);
	}
	mOldParam = param;

	_UpdateSlider(value);

}

// Update the send slider control.
void cTrontest::_UpdateSlider(long value)
{
	char buffer[10];
	if(!mReady)
	{
		return;
	}
	if(value != mOldSliderVal)
	{
		if(!mSliderWait)
		{
			if(value <= 0) { value = 1; }
			SendNotifyMessage(gHandle.hSlider, TBM_SETPOS, (WPARAM)(BOOL)TRUE, (LPARAM) (LONG) value);
			SetWindowText(gHandle.hSendRate, itoa(value, buffer, 10));
			mOldSliderVal = value;
		}
	}
}

// Node Change
void cTrontest::NodeUp()
{
	if(!mReady)
	{
		return;
	}
	mCS.Enter();
	if(mCurrentIndex != mNumNodes-1)
	{
		mCurrentIndex++;
	}
	mCS.Leave();
}
void cTrontest::NodeDown()
{
	if(!mReady)
	{
		return;
	}
	mCS.Enter();
	if(mCurrentIndex != 0)
	{
		mCurrentIndex--;
	}
	mCS.Leave();
}

void cTrontest::Kill()
{
	if(!mReady)
	{
		return;
	}
	mCS.Enter();
	TestMsg	msg;
	msg.type = TEST_MSG_KILL | TEST_MSG_SINGLE;
	msg.data.values.hashCode = mStatus[mCurrentIndex].ep->HashCode();
	mStack->Send(NULL, (char *)&msg, sizeof(msg));
	mCS.Leave();
}

void cTrontest::KillEmAll()
{
	if(!mReady)
	{
		return;
	}
	mCS.Enter();
	TestMsg	msg;
	msg.type = TEST_MSG_KILL;
	mStack->Send(NULL, (char *)&msg, sizeof(msg));
	mCS.Leave();
}

void cTrontest::MakeSleep()
{
	if(!mReady)
	{
		return;
	}
	mCS.Enter();
	TestMsg	msg;
	msg.type = TEST_MSG_SLEEP | TEST_MSG_SINGLE;
	msg.data.values.hashCode = mStatus[mCurrentIndex].ep->HashCode();
	mStack->Send(NULL, (char *)&msg, sizeof(msg));
	mCS.Leave();
}

void cTrontest::UpdateSendRate(unsigned int rate)
{
	if(!mReady)
	{
		return;
	}
	mCS.Enter();
	if(rate != mStatus[mCurrentIndex].sendRate)
	{
		TestMsg	msg;
		msg.type = TEST_MSG_SEND_RATE | TEST_MSG_SINGLE;
		msg.data.values.sendRate = rate;
		msg.data.values.hashCode = mStatus[mCurrentIndex].ep->HashCode();
		mStack->Send(NULL, (char *)&msg, sizeof(msg));
	}
	mCS.Leave();
}

void cTrontest::UpdateMySendRate(unsigned int rate)
{
	char buffer[10];
	if(rate != mMyStatus.sendRate)
	{
		if(rate == 0)
		{
			rate = 1;
		}
		mMyStatus.sendInterval = 1000/rate;
		mMyStatus.sendRate = rate;
		SetWindowText(gHandle.hSendRate, itoa(rate, buffer, 10));
	}
}

void cTrontest::SetForAll(unsigned int rate)
{
	TestMsg msg;
	msg.type = TEST_MSG_SEND_RATE;
	msg.data.values.sendRate = rate;
	mStack->Send(NULL, (char *)&msg, sizeof(msg));
}

void cTrontest::SetForAll()
{
	if(!mReady)
	{
		return;
	}
	mCS.Enter();
	TestMsg	msg;
	msg.type = TEST_MSG_SEND_RATE;
	msg.data.values.sendRate = mStatus[mCurrentIndex].sendRate;
	mStack->Send(NULL, (char *)&msg, sizeof(msg));
	mCS.Leave();
}