#include "cTronSim.h"
#include "Protocol Stack/cProtocolStack.h"
#include "Protocol Stack/cBufferManager.h"
#include "Endpoint/cEndpointFactory.h"
#include "Util\gError.h"
#include "SimNet Layer/cSimEngine.h"
#include <stdio.h>
#include <stdlib.h>
#include <winbase.h>
#include <winuser.h>
#include <commctrl.h>
#include "types.h"

extern TronSimHandles gHandle;
extern TronSimParam	  gParam;

#define START_INTERVAL 12
#define REFRESH_INTERVAL 500

#ifndef ABOVE_NORMAL_PRIORITY_CLASS
#define ABOVE_NORMAL_PRIORITY_CLASS 32768
#endif

// Receive view changes.
bool cTronSim::ViewHandler(cViewIterator* view)
{
	char	buffer[10];
	DWORD	now = cProtocolStack::GetTime();
	int		size = view->GetNumMembers();

	_itoa(now, buffer, 10);
	SetWindowText(gHandle.hLastViewTime, buffer);
	_itoa(size, buffer, 10);
	SetWindowText(gHandle.hLastViewSize, buffer);

	return true;

	cerr << "****NEW VIEW****" << endl;
	while(!view->Done())
	{
		view->GetMember()->Print(cerr);
		view->GetNext();
	}
	cerr << "****************" << endl;
	return true;
}

bool cTronSim::Deliver(cEndpoint* sender, cMsgBuffer* buffer, int messageType)
{
	unsigned long seqNum;
	int	  size;
	void* data;

	if(messageType == MSG_TYPE_NULL)
	{
		cerr << "WARNING! Received interruption in broadcast stream." << endl;
		return true;
	}

	mNumReceived++;
	if(mNumReceived == mNumLiveNodes)
	{
		buffer->GetPayload(&data, &size);
		memcpy(&seqNum, data, sizeof(unsigned long));
		if(seqNum == 0)
		{
			cerr << "Received KILL msg.  Shutting down." << endl;
			exit(1);
		}

		cerr << "SUCCESSFULLY delivered msg: SeqNum=" << seqNum << endl;
	}
	return true;
}

bool cTronSim::ErrorCallback(cObject* param, unsigned int type)
{
	char	buffer[10];
	DWORD	now = cProtocolStack::GetTime();

	switch(type)
	{
		case ASYNCH_ERROR_EP_FAIL:
			MessageBox(NULL, "Asynch Error: Endpoint fail.", "ASYNCH ERROR", MB_OK);
			break;

		case ASYNCH_ERROR_MSG_GAP:
			_itoa(now, buffer, 10);
			SetWindowText(gHandle.hLastGapTime, buffer);
			break;

		case ASYNCH_ERROR_SND_FAIL:
			MessageBox(NULL, "Asynch Error: Send fail.", "ASYNCH ERROR", MB_OK);
			break;

		default:
			MessageBox(NULL, "Asynch Error: Unknown.", "ASYNCH ERROR", MB_OK);
			break;
	}
	return true;
}

void cTronSim::RunTest(unsigned int argc, char** argv)
{
	mRun = true;	// Start test.
	mCurrentSubnet = 0;
	mCurrentAddress = 0;
	mFullViews		= 0;
	DWORD	nextStart = 0;
	DWORD	nextRefresh = 0;
	unsigned int currentStart;
	char	buf[sizeof(unsigned int)];

	memset(&mOldParam, 0, sizeof(PbcastParams));
	cProtocolStack::TimeStep(1);
	if(!_ParseParameters(argc, argv))
	{
		gAbort("Unable to parse command line parameters.", __LINE__, __FILE__);
	}
	if(!_Setup())
	{
		gAbort("Unable to set up protocol stacks.", __LINE__, __FILE__);
	}

	_WPrintNode();

	DWORD	now;
	DWORD	timeRegulate = GetTickCount();
	mNumNodes = mSimParam.mNumSubnets*mSimParam.mNumAddressesPerSubnet;
	mNumLiveNodes = mNumNodes;
	currentStart = 0;
	while(mRun)
	{
		now = cProtocolStack::GetTime();
		if((currentStart < mNumNodes) && (now > nextStart))
		{
			mStatus[currentStart].status = STATUS_ALIVE;
			currentStart++;
			nextStart = now + START_INTERVAL;
		}
		if((nextRefresh < now))
		{
			char buffer[10];
			_itoa(now, buffer, 10);
			SetWindowText(gHandle.hTime, buffer);
			_WPrintNode();
			nextRefresh = now + REFRESH_INTERVAL;
		}

		// Schedule the stacks.
		for(unsigned int i = 0; i < mNumNodes; i++)
		{
			if(mStatus[i].status == STATUS_SLEEP)
			{
				if(now > mStatus[i].timeToNextEvent)
				{
					mStatus[i].status = STATUS_ALIVE;
					_WPrintNode(i);
					mStatus[i].timeToNextEvent = now + mStatus[i].sendInterval;
				}
			}
			if(mStatus[i].status == STATUS_ALIVE)
			{
				if((mStatus[i].sendInterval) && (now > mStatus[i].timeToNextEvent))
				{
					mStatus[i].timeToNextEvent = now + mStatus[i].sendInterval;
					mStatus[i].seqNum++;
					memcpy(buf, &mStatus[i].seqNum, sizeof(buf));
					mStacks[i]->Send(NULL, &buf[0], sizeof(buf), MSG_TYPE_STANDARD);
				}
				mStacks[i]->Schedule();
			}
		}

		// Schedule the engine
		mEngine->Schedule();

		// Keep time from moving too fast!!
		do
		{
			now = GetTickCount();
		}
		while(now <= timeRegulate);
		cProtocolStack::TimeStep(2);
		timeRegulate = now;
	}

	if(!_Shutdown())
	{
		gAbort("Unable to shut down properly.", __LINE__, __FILE__);
	}
}


/*
 * cTronSim::_ParseParameters()
 *
 * Purpose:	Parses the command line parameters.
 * IN:		argc, argv		-> Standard argument count, args.
 * OUT:		-
 * Cond:	-
 * PostCnd:	All of the *LayerParams should be set up.
 * Throws:	-
 * Return:	true if success, else false.
 */
bool cTronSim::_ParseParameters(int argc, char* argv[])
{
	mSimParam.ResetToDefault();
	mDebugParam.ResetToDefault();
	mPbcastParam.ResetToDefault();

	if(argc < 5)
	{
		cerr << "Run As:  test  <percent-outgoing-drop> <percent-incoming-drop> <num_subnets> <num_ep_per_subnet>" << endl;
		exit(1);
	}

	// Handle debug layer parameters
	double percentOut = gParam.mPercentOutDrop;
	double percentIn  = gParam.mPercentInDrop;
	u_long max = RAND_MAX;
	mDebugParam.mProbDropIncomingMsg = (unsigned int) ((double)max * percentIn);
	mDebugParam.mProbDropOutgoingMsg = (unsigned int) ((double)max * percentOut);

	// Handle SimNet Layer params.
	mSimParam.mNumSubnets = gParam.mNumSubnets;
	mSimParam.mNumAddressesPerSubnet = gParam.mNodesPerSubnet;

	// Handle pbcast layer params.
	cEndpoint* ep;
	cGroup*	   subnetGroup = new cGroup(mSimParam.mNumSubnets);
	if(!subnetGroup) { return false; }
	
	ep = cEndpointFactory::AllocSimEndpoint(0, 0);
	if(!ep) { return false; }
	mPbcastParam.mMulticastEndpoint = ep;
	for(unsigned int i = 0; i < mSimParam.mNumSubnets; i++)
	{
		ep = cEndpointFactory::AllocSimEndpoint(i, 0);
		if(!ep) { return false; }
		subnetGroup->AddEndpoint(ep);
	}
	mPbcastParam.mSubnetList		  = subnetGroup;
	mPbcastParam.mMsgQueueSize		  = 1024;
	mPbcastParam.mMsgQueueGrowInc	  = 64;
	return true;
}

/*
 * cTronSim::_Setup()
 *
 * Purpose:	Sets up all of the protocol stacks.  Assumes that the *LayerParams are set up.
 * IN:		-
 * OUT:		-
 * Cond:	All of the *LayerParams should be set up.
 * PostCnd:	-
 * Throws:	-
 * Return:	true if success, else false.
 */
bool cTronSim::_Setup()
{
	cHandle				handle;
	cHandle				viewHandle;
	cHandle				errorHandle;
	unsigned int numNodes = mSimParam.mNumSubnets*mSimParam.mNumAddressesPerSubnet;
	cProtocolStack* stack;
	cSimNetLayer*   simnet;
	cDebugLayer*	debug;
	cPbcastLayer*	pbcast;

	mCurrentIndex = 0;

	// Attempt to increase the priority
//	if(!SetPriorityClass(GetCurrentProcess(), ABOVE_NORMAL_PRIORITY_CLASS))
//	{
//		gAbort("Unable to raise priority class.", __LINE__, __FILE__);
//	}

	// Create the simulation engine.
	mEngine = new cSimEngine(&mSimParam);
	if(!mEngine)
	{
		gAbort("Unable to create engine.", __LINE__, __FILE__);
	}
	mSimParam.mEngine = mEngine;

	// Alloc all of the protocol stacks, layers, etc.
	mStacks = new cProtocolStack*[numNodes];
	mSimNetLayers = new cSimNetLayer*[numNodes];
	mDebugLayers  = new cDebugLayer*[numNodes];
	mPbcastLayers = new cPbcastLayer*[numNodes];
	mStatus		  = new NodeStatus[numNodes];
	if(!mStacks || !mSimNetLayers || !mDebugLayers || !mPbcastLayers || !mStatus) { return false; }
	for(unsigned int i = 0; i < numNodes; i++)
	{
		stack  = new cProtocolStack("");
		simnet = new cSimNetLayer();
		debug  = new cDebugLayer();
		pbcast = new cPbcastLayer();
		if(!stack || !simnet || !debug || !pbcast) { return false; }
		mStacks[i]		  = stack;
		mSimNetLayers[i]  = simnet;
		mDebugLayers[i]   = debug;
		mPbcastLayers[i]  = pbcast;
		memset(&mStatus[i], 0, sizeof(NodeStatus));
	}

	// Build all of the stacks.
	for(i = 0; i < numNodes; i++)
	{
		if(!(mStacks[i]->AddLayer(mSimNetLayers[i], &mSimParam)) ||
		   !(mStacks[i]->AddLayer(mDebugLayers[i], &mDebugParam)) ||
		   !(mStacks[i]->AddLayer(mPbcastLayers[i], &mPbcastParam)) )
		{
			return false;
		}
		if(!(mStacks[i]->RegisterErrorCallback(&errorHandle, this)) ||
		   !(mStacks[i]->RegisterDeliverCallback(&handle, this))    ||
		   !(mStacks[i]->RegisterViewCallback(&viewHandle, this)) )
		{
			return false;
		}
	}
	return true;
}

/*
 * cTronSim::_Shutdown()
 *
 * Purpose:	Shuts down all of the protocol stacks and frees everything new'd
 * IN:		-
 * OUT:		-
 * Cond:	All of the protocol stacks should be set up.
 * PostCnd:	-
 * Throws:	-
 * Return:	true if success, else false.
 */
bool cTronSim::_Shutdown()
{
	unsigned int numNodes = mSimParam.mNumSubnets*mSimParam.mNumAddressesPerSubnet;

	// Free up extra endpoints.
	mPbcastParam.mSubnetList->Clear();
	delete mPbcastParam.mSubnetList;

	for(unsigned int i = 0; i < numNodes; i++)
	{
		mStacks[i]->Cleanup();
		delete mStacks[i];
		delete mSimNetLayers[i];
		delete mDebugLayers[i];
		delete mPbcastLayers[i];
	}
	delete[] mStacks;
	delete[] mSimNetLayers;
	delete[] mDebugLayers;
	delete[] mPbcastLayers;
	delete	 mStatus;
	return true;
}

void cTronSim::_WPrintNode(unsigned int index)
{
	if(mCurrentIndex == index)
	{
		_WPrintNode();
	}
}

void cTronSim::_WPrintNode()
{
	int index = mCurrentIndex;
	PbcastParams* param = mPbcastLayers[index]->ReportStats();
	char	buffer[10];

	if(mStatus[index].status == STATUS_ALIVE)
	{
		SetWindowText(gHandle.hAlive, "alive");
	}
	else if(mStatus[index].status == STATUS_DEAD)
	{
		SetWindowText(gHandle.hAlive, "dead");
	}
	else
	{
		SetWindowText(gHandle.hAlive, "sleeping");
	}
	_itoa(mCurrentSubnet, buffer, 10);
	SetWindowText(gHandle.hCurrentSubnet, buffer);

	_itoa(mCurrentAddress, buffer, 10);
	SetWindowText(gHandle.hCurrentAddress, buffer);

	if(param->mViewSize != mOldParam.mViewSize)
	{
		_itoa(param->mViewSize, buffer, 10);
		SetWindowText(gHandle.hViewSize, buffer);
	}

	if(param->mPbcastsRecvd != mOldParam.mPbcastsRecvd)
	{
		_itoa(param->mPbcastsRecvd, buffer, 10);
		SetWindowText(gHandle.hPbcastIn, buffer);
	}
	if(param->mSubnetListSize != mOldParam.mSubnetListSize)
	{
		_itoa(param->mSubnetListSize, buffer, 10);
		SetWindowText(gHandle.hSubnetSize, buffer);
	}
	if(param->mGossipInterval != mOldParam.mGossipInterval)
	{
		_itoa(param->mGossipInterval, buffer, 10);
		SetWindowText(gHandle.hGossipInterval, buffer);
	}

	if(param->mFanout != mOldParam.mFanout)
	{
		_itoa(param->mFanout, buffer, 10);
		SetWindowText(gHandle.hFanout, buffer);
	}

	if(param->mTimeToFailNode != mOldParam.mTimeToFailNode)
	{
		_itoa(param->mTimeToFailNode, buffer, 10);
		SetWindowText(gHandle.hTimeToFail, buffer);
	}

	if(param->mNumRetansReq != mOldParam.mNumRetansReq)
	{
		_itoa(param->mNumRetansReq, buffer, 10);
		SetWindowText(gHandle.hRetranReq, buffer);
	}

	if(param->mNumRetransRecvd != mOldParam.mNumRetransRecvd)
	{
		_itoa(param->mNumRetransRecvd, buffer, 10);
		SetWindowText(gHandle.hRetranRec, buffer);
	}

	if(param->mFullGossipsSent != mOldParam.mFullGossipsSent)
	{
		_itoa(param->mFullGossipsSent, buffer, 10);
		SetWindowText(gHandle.hFullGosOut, buffer);
	}

	if(param->mMinGossipsSent != mOldParam.mMinGossipsSent)
	{
		_itoa(param->mMinGossipsSent, buffer, 10);
		SetWindowText(gHandle.hMinGosOut, buffer);
	}

	if(param->mFullGossipsReceived != mOldParam.mFullGossipsReceived)
	{
		_itoa(param->mFullGossipsReceived, buffer, 10);
		SetWindowText(gHandle.hFullGosIn, buffer);
	}

	if(param->mMinGossipsReceived != mOldParam.mMinGossipsReceived)
	{
		_itoa(param->mMinGossipsReceived, buffer, 10);
		SetWindowText(gHandle.hMinGosIn, buffer);
	}

	if(param->mPbcastsSent != mOldParam.mPbcastsSent)
	{
		_itoa(param->mPbcastsSent, buffer, 10);
		SetWindowText(gHandle.hPbcastOut, buffer);
	}
	mOldParam = *param;
}

void cTronSim::_ComputeCurrentIndex()
{
	mCurrentIndex = mCurrentAddress + (mCurrentSubnet*mSimParam.mNumAddressesPerSubnet);
	SendMessage(gHandle.hSlider, TBM_SETPOS, (WPARAM)(BOOL)TRUE, (LPARAM) (LONG) mStatus[mCurrentIndex].sendRate);
}

// Node Change
void cTronSim::AddressLeft()
{
	if(mCurrentAddress != 0)
	{
		mCurrentAddress--;
		_ComputeCurrentIndex();
		_WPrintNode();
	}
}
void cTronSim::AddressRight()
{
	if(mCurrentAddress != mSimParam.mNumAddressesPerSubnet-1)
	{
		mCurrentAddress++;
		_ComputeCurrentIndex();
		_WPrintNode();
	}
}
void cTronSim::SubnetUp()
{
	if(mCurrentSubnet != mSimParam.mNumSubnets-1)
	{
		mCurrentSubnet++;
		_ComputeCurrentIndex();
		_WPrintNode();
	}
}
void cTronSim::SubnetDown()
{
	if(mCurrentSubnet != 0)
	{
		mCurrentSubnet--;
		_ComputeCurrentIndex();
		_WPrintNode();
	}
}

void cTronSim::Kill()
{
	if(mStatus[mCurrentIndex].status == STATUS_DEAD)
	{
		mStatus[mCurrentIndex].status = STATUS_ALIVE;
		mNumLiveNodes++;
	}
	else if(mStatus[mCurrentIndex].status == STATUS_ALIVE)
	{
		mStatus[mCurrentIndex].status = STATUS_DEAD;
		mNumLiveNodes--;
	}
	_WPrintNode();
}

void cTronSim::Sleep()
{
	PbcastParams* param = mPbcastLayers[mCurrentIndex]->ReportStats();
	mStatus[mCurrentIndex].timeToNextEvent = cProtocolStack::GetTime() + param->mTimeToFailNode - 300;
	mStatus[mCurrentIndex].status = STATUS_SLEEP;
	_WPrintNode();
}

void cTronSim::UpdateSendRate(unsigned int rate)
{
	if(rate != mStatus[mCurrentIndex].sendRate)
	{
		if(rate)
		{
			mStatus[mCurrentIndex].sendInterval = 1000/rate;
		}
		else
		{
			mStatus[mCurrentIndex].sendInterval = 0;
		}
		mStatus[mCurrentIndex].sendRate		= rate;
	}
}

void cTronSim::SetForAll()
{
	for(unsigned int i = 0; i < mNumNodes; i++)
	{
		mStatus[i].sendRate     = mStatus[mCurrentIndex].sendRate;
		mStatus[i].sendInterval = mStatus[mCurrentIndex].sendInterval;
	}
}