/*
 * cHashTable.cpp
 * 
 * Class implementatino for generic hash table.
 *
 * Authors: ted
 */
#include "cHashTable.h"
#include <assert.h>

#ifndef NULL
	#define NULL 0
#endif

cHashTableIterator::cHashTableIterator(cHashTable* table)
{
	mTable   = table;
	mCurrent = mTable->mTable[0];
	mIndex   = 0;
	GetNext();
}

bool		cHashTableIterator::Done()
{
	return (mIndex == mTable->mTableSize);
}

cObject*	cHashTableIterator::GetData()
{
	return mCurrent->data;
}

bool		cHashTableIterator::GetNext()
{
	mCurrent = mCurrent->next;
	while(mCurrent == mTable->mTable[mIndex])
	{
		mIndex++;
		if(Done())
		{
			return false;
		}
		mCurrent = mTable->mTable[mIndex]->next;
	}
	return true;
}

bool cHashTableIterator::DeleteCurrent()
{
	Node* node;
	if(Done())
	{
		return false;
	}
	else
	{
		node = mCurrent;
		GetNext();
		return mTable->Delete(node);
	}
}


/*
 * cHashTable::cHashTable()
 *
 * Purpose:	Create a new hash table.
 * IN:		size	-> The starting size of the hash table.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The hash table is set up.
 * Return:	-
 */
cHashTable::cHashTable(int size)
{
	mNodeBlock		= NULL;
	mTable			= NULL;
	mSentinelArray	= NULL;
	mNumElements    = 0;
	mTableSize = size;
	
	/*
	 * Alloc all of internal nodes.  mNodeBlock is just used to 
	 * allocate them in order so they can be de-alloc'd easier.
	 */
	mNodeBlock = new Node[mTableSize];

	//Alloc the hash table array (mTableSize buckets).
	mTable = new Node*[mTableSize];

	//Alloc array to hold sentinels (+1 extra for the free list.
	mSentinelArray = new Node[mTableSize+1];

	if (!mNodeBlock || !mTable || !mSentinelArray)
	{  
		throw cHashException("<cHashTable::cHashTable> Unable to alloc mem for data structures.");
	}

	//Place sentinels in hash table and freelist.
	mFreeList = &mSentinelArray[mTableSize];	//Last one.
	_ResetNode(mFreeList);

	for(unsigned int i = 0; i < mTableSize; i++)
	{
		mTable[i] = &mSentinelArray[i];
		_ResetNode(mTable[i]);
	}

	//Add nodes to free list.
	for(i = 0; i < mTableSize; i++)
	{
		_PushFreeList(&mNodeBlock[i]);		
	}
}

/*
 * cHashTable::~cHashTable()
 *
 * Purpose:	Destroy the hash table.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The hash table is cleaned up.
 * Return:	-
 */
cHashTable::~cHashTable()
{
	if(mNodeBlock)
	{
		delete[] mNodeBlock;
	}
	if(mTable)
	{
		delete[] mTable;
	}
	if(mSentinelArray)
	{
		delete[] mSentinelArray;
	}
}

/*
 * cHashTable::Insert()
 *
 * Purpose:	Insers the data into the hash table hashing on key.
 * IN:		node	-> Pointer to the node to be added to the free list.
 * OUT:		-
 * Cond:	There is a node in the free list.
 * PostCnd:	The free list shrinks by 1.
 * Return:	NULL if fail, else a pointer to the Node that holds data.
 */ 
Node* cHashTable::Insert(cHashObject* key, cHashObject* hObj)
{
	Node *node;
	Node *sentinel;
	unsigned int hashVal;

	assert(key);
	assert(hObj);

	mCS.Enter();

	node = _PopFreeList();
	if(node == NULL)
	{
		mCS.Leave();
		return NULL;
	}

	hashVal = _Hash(key);

	//Insert node into hash table.
	sentinel = mTable[hashVal];
	node->next = sentinel->next;
	(sentinel->next)->prev = node;
	node->prev = sentinel;
	sentinel->next = node;

	// Increase ref count so elements do not disappear.
	hObj->AddRef();
	key->AddRef();
	node->data = (cObject *)hObj;
	node->key  = key;

	mNumElements++;
	
	mCS.Leave();
	return node;
}

/*
 * cHashTable::Find()
 *
 * Purpose:	Finds the given node based on the key.
 * IN:		key	-> Unique search key for the node.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	NULL if fail, otherwise a pointer to the data.
 */
Node*	cHashTable::Find(cHashObject* key, cHashObject** data)
{
	int hashValue;
	Node *listIterator;

	mCS.Enter();

	hashValue = _Hash(key);
	listIterator = mTable[hashValue]->next;

	//Iterate through hash table bucket to see if the node exists.
	while (listIterator != mTable[hashValue]) 
	{
		if (key->Equals(listIterator->key))
		{
			mCS.Leave();
			*data = (cHashObject *)listIterator->data;
			return listIterator;
		}
		listIterator = listIterator->next;
	}

	mCS.Leave();
	return NULL;
}

/*
 * cHashTable::Delete()
 *
 * Purpose:	Removes an element based on the key <<ATOMIC>>
 * IN:		key		-> The key that is associated w/ the element.
 * OUT:		-
 * Cond:	The node is from the hash table.
 * PostCnd:	The node is added to the free list and is INVALID!.
 * Return:	true if success, false otherwise.
 */
bool cHashTable::Delete(cHashObject* key)
{
	Node		*node;
	cHashObject	*data;
	bool	 retVal = false;

	mCS.Enter();
	
	node = Find(key, &data);
	if(node)
	{
		retVal = Delete(node);
	}

	mCS.Leave();
	return retVal;
}


/*
 * cHashTable::Delete()
 *
 * Purpose:	Removes the given node from the hash table.
 * IN:		node	-> Pointer to the node to be removed.
 * OUT:		-
 * Cond:	The node is from the hash table.
 * PostCnd:	The node is added to the free list and is INVALID!.
 * Return:	true if success, false otherwise.
 */
bool cHashTable::Delete(Node *node)
{
	assert(node != NULL);

	mCS.Enter();

	//Remove from hash table.
	(node->prev)->next = node->next;
	(node->next)->prev = node->prev;

	// Free the node elements
	((cHashObject *)node->data)->Release();
	node->key->Release();

	//Add to free list
	_PushFreeList(node);

	mNumElements--;

	mCS.Leave();
	return true;
}


/*
 * cHashTable::_PushFreeList()
 *
 * Purpose:	Adds the given node to the free list.
 * IN:		node	-> Pointer to the node to be added to the free list.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The node is added to the free list.
 * Return:	-
 */
void cHashTable::_PushFreeList(Node *node)
{
	node->next = mFreeList->next;
	(mFreeList->next)->prev = node;
	node->prev = mFreeList;
	mFreeList->next = node;
}

/*
 * cHashTable::_PopFreeList()
 *
 * Purpose:	Gets a node of the free list if there is one.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The node is added to the free list.
 * Return:	NULL if the free list is empty, else a free node pointer.
 */
Node* cHashTable::_PopFreeList()
{
	Node *node;

	node = mFreeList->next;
	if (node == mFreeList) {
		return NULL; //empty list
	} else {
		(node->prev)->next = node->next;
		(node->next)->prev = node->prev;
		return node;
	}
}

/*
 * cHashTable::_Hash()
 *
 * Purpose:	Computes the hash value for the given input.
 * IN:		key		-> A key to hash on.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	The hash value of the given key.
 */
int	cHashTable::_Hash(cHashObject* key)
{
	return (key->HashCode() % mTableSize);
}

/*
 * cHashTable::_ResetNode()
 *
 * Purpose:	Makes the node double point to itself.
 * IN:		node	-> The node to reset.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The node's next and prev point to itself.
 * Return:	-
 */
void cHashTable::_ResetNode(Node *node)
{
	node->prev = node;
	node->next = node;
}