/*
 * cFifo.cpp
 * 
 * Class implementatino for generic Fifo queue table.
 *
 * Authors:  Vicky & Ted
 */
#include "cFifo.h"
#include <assert.h>

#ifndef NULL
	#define NULL 0
#endif

cObject* cFifoIterator::GetData() { 
	return (cObject *)mNode->data; 
}

bool cFifoIterator::DeleteCurrent()	
{
	Node* node;
	if(Done())
	{
		return false;
	}
	else
	{
		node = mNode;
		GetNext();
		return mFifo->Delete(node);
	}
}

/*
 * cFifo::cFifo()
 *
 * Purpose:	Create a new Fifo queue.
 * IN:		size	-> The starting size of the Fifo queue.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is set up.
 * Throws:	cFifoException
 * Return:	-
 */
cFifo::cFifo(int size)
{
	Node *temp		= NULL;
	mSentinel		= NULL;
	mSize			= size;
	mNumElements    = 0;

	// Alloc sentinel
	mSentinel = new Node;
	if(!mSentinel)
	{  
		throw cFifoException("<cFifo::cFifo>: Unable to alloc the sentinel node.");
	}
	_ResetNode(mSentinel);

	// Alloc free list sentinel.
	mFreeList = new Node;
	if(!mFreeList)
	{  
		throw cFifoException("<cFifo::cFifo>: Unable to alloc the free list.");
	}
	_ResetNode(mFreeList);

	// Alloc nodes.
	for(int i = 0; i < mSize; i++)
	{
		temp = new Node;
		if(!temp)
		{
			throw cFifoException("<cFifo::cFifo>: Unable to alloc enough nodes.");
		}
		_PushFreeList(temp);
	}
	mBack  = mSentinel;
}

/*
 * cFifo::~cFifo()
 *
 * Purpose:	Destroy the queue.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The queue is cleaned up.
 * Return:	-
 */
cFifo::~cFifo()
{
	cObject *data;
	Node	*node;

	// Pop all of the nodes to the free list.
	while(Pop(&data));

	mCS.Enter();

	node = _PopFreeList();
	while(node != NULL)
	{
		delete node;
		node = _PopFreeList();
	}
	if(mSentinel)
	{
		delete mSentinel;
	}
	if(mFreeList)
	{
		delete mFreeList;
	}

	mCS.Leave();
}

/*
 * cFifo::Push()
 *
 * Purpose:	Inserts the data into the front of the queue.
 * IN:		data	-> A pointer to the data object to be inserted.
 * OUT:		-
 * Cond:	There is a node in the free list.
 * PostCnd:	The free list shrinks by 1.
 * Return:	NULL if fail, else a pointer to the Node that holds data.
 */ 
Node* cFifo::Push(cObject* data)
{
	Node *node; 

	mCS.Enter();

	node = _PopFreeList();
	if(node == NULL)
	{
		mCS.Leave();
		return NULL;
	}

	//Add the data
	node->data = data;

	//Insert node into queue.
	node->next  = mBack->next;
	mBack->next = node;
	node->prev  = mBack;
	mBack		= node;
	mNumElements++;

	mCS.Leave();
	return node;
}

/*
 * cFifo::Pop()
 *
 * Purpose:	Removes the next node in fifo order.
 * IN:		-
 * OUT:		data	-> The data that was contained in the front node.
 * Cond:	There is a node in the queue.
 * PostCnd:	The free list grows by 1.
 * Return:	true if success, else fail.
 */ 
bool cFifo::Pop(cObject** data)
{
	Node *node;

	mCS.Enter();

	node = mSentinel->next;

	assert(node != NULL);

	if (node == mSentinel) {
		mCS.Leave();
		return false; //empty list
	} else {
		// Special case: Node is the only one in the list.
		if(node == mBack)
		{
			mBack = mSentinel;
		}

		assert(node->prev != NULL);
		assert(node->next != NULL);

		(node->prev)->next = node->next;
		(node->next)->prev = node->prev;

		// Set the data.
		*data = node->data;

		// Add to free list.
		_PushFreeList(node);

		mNumElements--;

		mCS.Leave();
		return true;
	}
}

/*
 * cFifo::Peek()
 *
 * Purpose:	Sets pointer to point to data from the next node that would be popped.
 * IN:		-
 * OUT:		data	-> The data that was contained in the front node.
 * Cond:	There is a node in the queue.
 * PostCnd:	-
 * Return:	true if success, else fail.
 */ 
Node* cFifo::Peek(cObject** data)
{
	Node *node;

	mCS.Enter();

	node = mSentinel->next;
	if (node == mSentinel) 
	{
		node = NULL;
	} 
	else 
	{
		*data = node->data;
	}
	mCS.Leave();
	return node;
}

/*
 * cFifo::Delete()
 *
 * Purpose:	Removes the given node from the queue.
 * IN:		node	-> Pointer to the node to be removed.
 * OUT:		-
 * Cond:	The node is from the queue.
 * PostCnd:	The node is added to the free list and is INVALID!.
 * Return:	true if success, false otherwise.
 */
bool cFifo::Delete(Node *node)
{
	assert(node != NULL);

	mCS.Enter();

	// If this is the last node, then special case.
	if(mBack == node)
	{
		mBack = node->prev;
	}

	//Remove from queue.
	assert(node->prev != NULL);
	assert(node->next != NULL);
	(node->prev)->next = node->next;
	(node->next)->prev = node->prev;

	//Add to free list
	_PushFreeList(node);

	mNumElements--;
	mCS.Leave();
	return true;
}

/*
 * cFifo::_PushFreeList()
 *
 * Purpose:	Adds the given node to the free list.
 * IN:		node	-> Pointer to the node to be added to the free list.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The node is added to the free list.
 * Return:	-
 */
void cFifo::_PushFreeList(Node *node)
{
	node->data = NULL;
	node->next = mFreeList->next;
	(mFreeList->next)->prev = node;
	node->prev = mFreeList;
	mFreeList->next = node;
}

/*
 * cFifo::_PopFreeList()
 *
 * Purpose:	Gets a node of the free list if there is one.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The node is added to the free list.
 * Return:	NULL if the free list is empty, else a free node pointer.
 */
Node* cFifo::_PopFreeList()
{
	Node *node;

	node = mFreeList->next;
	if (node == mFreeList) {
		return NULL; //empty list
	} else {
		(node->prev)->next = node->next;
		(node->next)->prev = node->prev;
		return node;
	}
}

/*
 * cFifo::_ResetNode()
 *
 * Purpose:	Makes the node double point to itself.
 * IN:		node	-> The node to reset.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The node's next and prev point to itself.
 * Return:	-
 */
void cFifo::_ResetNode(Node *node)
{
	node->prev = node;
	node->next = node;
}