/*
 * cEndpointFactory.cpp
 *
 * Creator of endpoints.
 */

#include "cEndpointFactory.h"
#include "Network Layer/cIPEndpoint.h"
#include "SimNet Layer/cSimEndpoint.h"
#include "Util/gError.h"
#include <stdio.h>

unsigned int	cEndpointFactory::mEpAlloc = 0;
unsigned int	cEndpointFactory::mEpFree = 0;

cEndpoint*	cEndpointFactory::AllocIPEndpoint(char* address, u_short port)
{
	cIPEndpoint* newEp;
	newEp = new cIPEndpoint(address, port);
	if(newEp)
	{
		mEpAlloc++;
	}
	return (cEndpoint *)newEp;
}

cEndpoint*	cEndpointFactory::AllocSimEndpoint(unsigned int subnet, unsigned int address)
{
	cSimEndpoint* newEp;
	newEp = new cSimEndpoint(subnet, address);
	if(newEp)
	{
		mEpAlloc++;
	}
	return (cEndpoint *)newEp;
}

bool cEndpointFactory::ReleaseEndpoint(cEndpoint* ep)
{
	if(ep->GetType() == ENDPOINT_TYPE_IP)
	{
		delete (cIPEndpoint *)ep;
	}
	else
	{
		delete ep;
	}
	mEpFree++;
	return true;
}

void cEndpointFactory::ReportStats(ostream &stream)
{
	stream << "$$$$$$$$$$$$$$$$$$$$$$$$" << endl;
	stream << "Endpoint Factory Stats    " << endl;
	stream << "EP Alloced:  " << mEpAlloc << endl;
	stream << "EP Freed:    " << mEpFree  << endl;
	stream << "$$$$$$$$$$$$$$$$$$$$$$$$" << endl;
}

/*
 * cIPEndpoint::AllocEndpoint()
 *
 * Purpose:	Allocates a blank endpoint of the given type.
 * IN:		type		-> The endpoint type to alloc.
 * OUT:		-
 * Cond:	-
 * PostCnd:	An endpoint of a known type is allocated.
 * Return:	The new endpoint or NULL if fail.
 */
cEndpoint* cEndpointFactory::AllocEndpoint(int type)
{
	cEndpoint* retVal = NULL;
	switch(type)
	{
		case ENDPOINT_TYPE_IP:
			retVal = new cIPEndpoint();
			break;

		case ENDPOINT_TYPE_SIM:
			retVal = new cSimEndpoint();
			break;
	}
	if(retVal)
	{
		mEpAlloc++;
	}
	return retVal;
}

/*
 * cIPEndpoint::AllocEndpoint()
 *
 * Purpose:	Allocates an appropriate endpoint and deserializes it from the given buffer.
 * IN:		buffer		-> Address of the buffer in which the serialized endpoint resides.
 *			size		-> Address of an int holding the size of the buffer.
 * OUT:		size		-> The size remaining in the buffer, or the size needed if fail.
 * Cond:	-
 * PostCnd:	An endpoint of a known type is in the buffer.
 * Return:	The address of the location in the buffer after the serialized representation
 *			OR NULL if failure.
 */
cEndpoint*	cEndpointFactory::AllocEndpoint(char** buffer, int *size)
{
	cEndpoint*	retVal = NULL;
	char *inBuf = *buffer;
	unsigned char type;

	if(*size == 0)
	{
		return NULL;
	}

	type = inBuf[0];
	retVal = AllocEndpoint(type);
	if(retVal)
	{
		inBuf = retVal->Deserialize(inBuf, size);
		if(inBuf)
		{
			*buffer = inBuf;	// Update the buffer loc. (size is already updated)
		}
		else
		{
		  retVal->Release();
		  retVal = NULL;
		}
	}
	return retVal;
}