#include "atom.h"
#include "system.h"

Atom::Atom(float x, float y, float mass, float ix, float iy, int level, const bool levels[MAXLEVELS], float mk, int rigidity, float rdist) {
	int i;

	init();

	if(level != 0) {
		for(i = 0; i < MAXLEVELS; i++) {
			this->levels[i] = levels[i];
		}
	}
	else {
		for(i = 0; i < MAXLEVELS; i++) {
			this->levels[i] = false;
		}
	}

	this->level = level;
	this->x = x;
	this->y = y;
	this->mk = mk;
	this->rigidity = rigidity;
	rrigidity = 1.0f/ rigidity;
	this->rdist = rdist;
	this->mass = mass;
	rm = 1.0f / mass;
	if(interpolate) {
		ix *= rrigidity;
		iy *= rrigidity;
	}
	ox = x - ix;
	oy = y - iy;
}

Atom::~Atom() {
	object->removeAtom(this);
}

void Atom::destroy() {
	being_deleted = true;
}

void Atom::init() {
	int i;

	being_deleted = false;
	interpolate = false;
	draw = true;
	x = y = 0;
	ox = oy = 0;
	next = NULL;
	//weight = 0.0f;
	//rw = 1.0f;
	fx = 0.0f;
	fy = 0.0f;
	cx = cy = 0;
	index = -1;
	next = NULL;
	group = 0;
	mass = 0.0f;
	ix = iy = 0.0f;

	numlinks = 0;

	for(i = 1; i < MAXLEVELS; i++) {
		levels[i] = true;
	}
	levels[0] = false;

	level = 0;

	object = NULL;
}

void Atom::repulse(Atom *b) {
	float mult;
	float dx, dy;
	float dist;
	float trdist = rdist + b->rdist;

	dx = x - b->x;
	dy = y - b->y;

	dist = dx * dx + dy * dy;

	if(dist < 0.05f) {
		//dx = (rand() % 5) * 1.0f;
		dx = 0;
		dy = 1.0f;
		dist = dx * dx + dy * dy;
	}

	if(dist > trdist * trdist) return;
	mult = float(trdist / sqrt(dist));
	object->onRepulse(this, b, mult);
	b->object->onRepulse(b, this, mult);

	float comp = 1.0f / (rm + b->rm);
	float forcex = comp * (dx * mult - dx);
	float forcey = comp * (dy * mult - dy);

	fx += forcex;
	fy += forcey;

	b->fx -= forcex;
	b->fy -= forcey;
}

void Atom::updateContainer() {
	cx = int(x * system->rcontainerw) + 1;
	cy = int(y * system->rcontainerh) + 1;
	system->containers[cx][cy].add(this, system->tick);
}

void Atom::repel() {
	Atom *b;
	Container *c;

	int xo, yo;
	for(xo = -1; xo <= 1; xo++) {
		for(yo = -1; yo <= 1; yo++) {
			c = &(system->containers[cx + xo][cy + yo]);
			if(c->stamp == system->tick) {
				b = c->first;

				while(b != NULL) {
					if((system->interacttable.test(index * MAXATOMS + b->index)) && levels[b->level] && object->allowRepulse(this, b) && b->object->allowRepulse(b, this))
						repulse(b);

					b = b->next;
				}
			}
		}
	}

	if(interpolate) {
		x += ix;
		y += iy;

		//fx *= rm * rrigidity;
		//fy *= rm * rrigidity;
	}
}

void Atom::addLink(Link *l) {
	numlinks++;
	draw = false;
}

void Atom::removeLink(Link *l) {
	numlinks--;
	if(numlinks == 0)
		draw = true;
}

bool Atom::in(Object *o) {
	Object *c = object;
	while(c) {
		if(c == o) return true;
		c = c->parent;
	}
	return false;
}

void Atom::update() {
	if(being_deleted) {
		delete this;
		return;
	}

	if(!interpolate) {
		float newpos;
		newpos = 2.0f * x - ox;
		ox = x;
		x = newpos;
		newpos = 2.0f * y - oy + system->gravity;
		oy = y;
		y = newpos;

		if(x < 0) x = 0.0f;
		if(x >= system->width) x = system->width;
		if(y < 0) y = 0.0f;
		if(y >= system->height) y = system->height;
	}
	else {
		iox = ox;
		ioy = oy;
		ix = (x - ox) * rrigidity;
		iy = (y - oy) * rrigidity;
		ox = x;
		oy = y;
	}

	object->atomFunc(this);
	updateContainer();
}

void Atom::snap() {
	fx *= rm;
	fy *= rm;

	x += fx;
	y += fy;
	ox += fx * mk;
	oy += fy * mk;
	fx = fy = 0;

	if(x < 0) x = 0.0f;
	if(x > system->width) x = system->width;
	if(y < 0) y = 0.0f;
	if(y > system->height) y = system->height;
	
	updateContainer();
}

void Atom::save(FILE *output) {
	fwrite(&(x), sizeof(float), 1, output);
	fwrite(&(y), sizeof(float), 1, output);
	fwrite(&(ox), sizeof(float), 1, output);
	fwrite(&(oy), sizeof(float), 1, output);
	fwrite(&mass, sizeof(float), 1, output);
	fwrite(&(rigidity), sizeof(int), 1, output);
	fwrite(&(group), sizeof(int), 1, output);
	fwrite(&(level), sizeof(int), 1, output);
	fwrite(levels, sizeof(bool), 10, output);
}

void Atom::load(FILE *input, float offx, float offy) {
	//init();

	fread(&(x), sizeof(float), 1, input);
	fread(&(y), sizeof(float), 1, input);
	fread(&(ox), sizeof(float), 1, input);
	fread(&(oy), sizeof(float), 1, input);
	fread(&mass, sizeof(float), 1, input);
	fread(&(rigidity), sizeof(int), 1, input);
	fread(&(group), sizeof(int), 1, input);
	fread(&(level), sizeof(int), 1, input);
	fread(levels, sizeof(bool), 10, input);

	rm = 1.0f / mass;
	rrigidity = 1.0f/ rigidity;

	x += offx;
	y += offy;
	ox += offx;
	oy += offy;
}