package LDMS::Status;

# Status.pm
#
# Handles status messages.
#
# $Header: /home/LDMS/cvsroot/LDMS/LDMS/Status.pm,v 1.11 2000/12/01 03:42:08 cs58 Exp $
#
# Author: Charles Shagong (cs58@cornell.edu)
#

use strict;
use warnings;
require 5.005;
use lib '..';  # Make sure all modules are included AFTER this.
use LDMS::State;


BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    # Set our version (for module version checking).
    $VERSION = do {my @r = (q$Revision: 1.11 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};

    @ISA = qw(Exporter);

    # Exported functions.
    @EXPORT = qw(&initStatus &defineIntervalStatus &updateStatus &updateAndPrintStatus &printStatus &printProcessingLevel);
    %EXPORT_TAGS = ( );

    # Exported global variables and optional exported functions.
    @EXPORT_OK = qw();

}
our @EXPORT_OK;

# Non-exported global variables.


# Initialize exported global variables.


# Initialize non-exported global variables.


# Create private global variables.
my $status_currLineNum = 1;  # Current line number during processing.
my $status_statusInterval = 1;  # Status output interval.


# Prototype functions.
sub initStatus($);
sub defineIntervalStatus($);
sub updateStatus();
sub updateAndPrintStatus();
sub printStatus();
sub printProcessingLevel();


# Module destructor.
END { }


# SUBROUTINES

##########
#
# &initStatus($initLineNum);
#
# Purpose: 
# initialize the status indicator with the first line number
#
# Precondition:
# given scalar argument is a positive number
#
# Postcondition:
# current line number indicator is set to the given argument
#
# 
sub initStatus($)
{
   if($_[0] > 0)
   {
      $status_currLineNum = $_[0];
   }
}


##########
#
# &defineIntervalStatus($interval);
#
# Purpose: 
# define the status output interval
# 
# Precondition:
# given scalar argument is a positive integer
#
# Postcondition:
# status interval is set to the given argument
#
#
sub defineIntervalStatus($)
{
   my $interval = $_[0];

   if($interval >= 1)
   {
      $status_statusInterval = $interval;
   }
}


##########
#
# &updateStatus();
#
# Purpose:
# increment the line number of status and prints status if
# defined interval is reached
# 
# Precondition:
# None
#
# Postcondition:
# current line number is incremented by one
# and status is output by current processing line number if
# defined interval is reached
#
#
sub updateStatus()
{
   $status_currLineNum = $status_currLineNum + 1;
   if($status_currLineNum % $status_statusInterval == 0)
   {
      print "Status: at line number $status_currLineNum \n";
   }
}


##########
#
# &updateAndPrintStatus();
#
# Purpose:
# increment the line number of status and print status
# regardless of defined interval
#
# Precondition:
# None
#
# Postcondition:
# current line number is incremented by one
# and status is output regardless of defined status 
# output interval
#
sub updateAndPrintStatus()
{
   $status_currLineNum = $status_currLineNum + 1;
   print "Status: at line number $status_currLineNum \n";
}


##########
#
# &printStatus();
#
# Purpose:
# print the status without updating
#
# Precondition:
# None
#
# Postcondition:
# status is output regardless of defined status 
# output interval
#
sub printStatus()
{
   print "Status: at line number $status_currLineNum \n";
}


#==============================================================================
# PRINT CURRENT LEVEL OF PROCESSING
#
# printProcessingLevel()
#
# Prints the level we're currently processing.
sub printProcessingLevel() {

    print STDERR "Processing " . &getLevelLabel() . " " . &getSequenceLabel() . "\n";

}


1;
__END__;

=pod 
$Log: Status.pm,v $
Revision 1.11  2000/12/01 03:42:08  cs58
Added authorship information

Revision 1.10  2000/11/30 16:19:27  jcl53
Upped the minimum Perl interpreter version from 5.002 to 5.005, just in case someone out there does something like trying to run this script on a non-Unix system...

Revision 1.9  2000/11/29 06:10:23  jcl53
Modified status messages to look better with the new sequence labels.

Revision 1.8  2000/11/29 00:01:02  jcl53
Added new status message style to this module.

Revision 1.7  2000/11/28 16:27:35  jcl53
Added a little comment to the 'use lib' lines.  Modules (especially LDMS ones) should be included after them so that testing compilation with 'perl -c' works.

Revision 1.6  2000/11/21 21:01:15  jcl53
Added the following line to all modules to make including modules less painful:

	use lib '..';

If you're using other modules, include them AFTER this line.  Thanks.

Revision 1.5  2000/11/17 20:04:30  jcl53
Fixed a few prototyping disagreements that were probably introduced during commenting.  :)

Revision 1.4  2000/11/16 22:52:07  cs58
Added source level documentation per design requirements.
Changed variable names to conform to coding standards.

Revision 1.3  2000/11/10 02:55:25  jcl53
Status.pm now includes cleanly.

Revision 1.2  2000/11/10 01:54:08  jcl53
Corrected a few odd things in the logs, such as the status messages left over from the template (oops).

Revision 1.1  2000/11/09 20:55:07  jcl53
Added skeleton Perl module files to the appropriate directory.
