package LDMS::Footnote;

# Footnote.pm
#
# The footnote module for the LDMS project.
#
# $Header: /home/LDMS/cvsroot/LDMS/LDMS/Footnote.pm,v 1.19 2000/12/01 18:20:27 om14 Exp $
#
# author: Omar Mehmood
# email: om14@cornell.edu
#

use strict;
use warnings;
require 5.005;
use lib '..';
use LDMS::Error;
use LDMS::State;
use LDMS::XMLOut;
use LDMS::Xrefs;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    # Set our version (for module version checking).
    $VERSION = do {my @r = (q$Revision: 1.19 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};

    @ISA = qw(Exporter);

    # Exported functions.
    @EXPORT = qw(&tagFootnotes);
    %EXPORT_TAGS = ( );

    # Exported global variables and optional exported functions.
    @EXPORT_OK = qw();

}
our @EXPORT_OK;

# Non-exported global variables.


# Initialize exported global variables.


# Initialize non-exported global variables.


# Create private global variables.
my @text; # ascii text input
my $currentLine; # index for the current line being processed within @text
my $errorMessage; # holds the text of an error message
my $footnoteNumber; # holds the parsed number of a footnote
my $currentWhiteSpace; # holds the number of white space characters for $currentLine
my $previousWhiteSpace; # holds the number of white space characters for the previous line
my $footnoteTextFlag; # indicates whether we are currently processing footnote text (1) or not (0)
my $partialText; # holds the portion of a line that corresponds to footnote text
my @footnoteText; # holds the lines of text of a footnote
my @sequence; # holds the sequence returned by the state machine
my $i; # controls the for loop
my $eid; # place holder for the eid


# Prototype functions.
sub tagFootnotes(@);
sub processFootnote();


# Module destructor.
END { }


# Insert functions, code, &c., here.

# tagFootnotes(@)
#
# Precondition: The input is an array of all text lines within a titledata element.
#
# Postcondition: Footnote numbers in state machine module are updated.
#                Footnote XML tags are output.
#                Xref module is called with all remaining text.

sub tagFootnotes(@)
{
   # initialize private global variables
   @text = @_;
   $currentLine = 0;
   $previousWhiteSpace = "";
   @footnoteText = ();

   # perform an error check for empty input
   if (! defined($text[$currentLine]) )
   {
      $errorMessage = "No input was passed to the tagFootnotes function.";
      &printErrMsg($errorMessage);
      return;
   }

   # check for footnote pattern match
   while ( $currentLine < scalar(@text) )
   {
      $text[$currentLine] =~ /^(\W+)/;
      $currentWhiteSpace = length($1);

      if ( $text[$currentLine] =~ /^(\W+)\(FOOTNOTE\W+(\d+)\)/ )
      {
         $currentWhiteSpace = length($1);

         if ( (defined($previousWhiteSpace)) && ($previousWhiteSpace ne "") && ($previousWhiteSpace ne $currentWhiteSpace) )
         {
            # we need to call Xref module on text before footnotes
            # variable is called @footnoteText but includes basically all the text before footnotes at this point in the code
	    &handleXrefs(@footnoteText);
            @footnoteText = ();

            $footnoteNumber = $2;
            $partialText = $';

            # we need to increment the footnote number in the state machine module
            &incrFootnoteNumber($footnoteNumber);

            # we must output start tag and its attributes before we start processing footnote text
            &BeginTag("FOOTNOTE");
            &AddAttribute("FNUMBER",$footnoteNumber);
            $eid = &getFootnoteNumber($footnoteNumber);
            @sequence = &getSequence();
            for ($i=0; $i < scalar(@sequence); $i++)
            {
	       $eid = $eid . "-" . $sequence[$i];
            }
            &AddAttribute("EID",$eid);

            # now we need to find the end of the footnote and process its text
            &processFootnote();

            # we need to output the end tag after processing the footnote text
            &EndTag("FOOTNOTE");

            # finally, we must flush the @footnoteText buffer
            @footnoteText = ();
         } else {
            # no pattern match exists for current line
            push(@footnoteText,$text[$currentLine]);
            $previousWhiteSpace = $currentWhiteSpace;
            $currentLine = $currentLine + 1;
         }
      } else {
         # no pattern match exists for current line
         push(@footnoteText,$text[$currentLine]);
         $previousWhiteSpace = $currentWhiteSpace;
         $currentLine = $currentLine + 1;
      }
   }
   # call Xref module on the rest of the text that does not contain footnotes
   &handleXrefs(@footnoteText);
}


# processFootnote()
#
# Precondition:
#
# Postcondition:

sub processFootnote()
{
   $footnoteTextFlag = 1;
   while ( $footnoteTextFlag )
   {
      push(@footnoteText,$partialText);
      # the right justified dates yield an overall margin of 73
      # however, the margin for the actual text is 71
      # therefore, 71 - 20 or 51 is first used to determine the end of footnote text
      if ( length($text[$currentLine]) < 51 )
      {
         $previousWhiteSpace = "";
         $currentLine = $currentLine + 1;

         # we must call the Xref module function to process the footnote text for xref and footrefs
         &handleXrefs(@footnoteText);

         # we must flush the @footnoteText buffer
         @footnoteText = ();

         $footnoteTextFlag = 0;
      } else {
         # check if we are currently at the last line of text
         if ( $currentLine == scalar(@text) - 1 )
         {
            # we must call the Xref module function to process footnote text for xref and footrefs
	    &handleXrefs(@footnoteText);

            # we don't have to flush the @footnoteText buffer since this is the last line of text
            @footnoteText = ();

	    $footnoteTextFlag = 0;
            $currentLine = $currentLine + 1;
         } else {
            $previousWhiteSpace = $currentWhiteSpace;
            $currentLine = $currentLine + 1;
            $text[$currentLine] =~ /^(\W+)/;
            $currentWhiteSpace = length($1);

            # check for correct white space indentation and
            #   whether the line is not a new subsection
            if ( ($previousWhiteSpace - $currentWhiteSpace == 3) && ($text[$currentLine] !~ /^(\W+)\(\w\)/) )
            {
               $partialText = $text[$currentLine];
            } else {
               # we must call Xref module function to process footnote text for xrefs and footrefs
	       &handleXrefs(@footnoteText);

               # since we processed an extra line here for footnote text,
               # we must perform a rollback to re-process the line for footnotes
               # therefore, the currentLine remains unchanged
               @footnoteText = ();
               $previousWhiteSpace = "";

	       $footnoteTextFlag = 0;
            }
         }
      }
   }
}



1;
__END__;

=pod 
$Log: Footnote.pm,v $
Revision 1.19  2000/12/01 18:20:27  om14
Changes to FootRef.pm: added use for LDMS::State and uncommented out line 151 which refers to the footnote number
Added authoring for Error.pm and Footnote.pm

VS: ----------------------------------------------------------------------

Revision 1.18  2000/11/30 20:29:23  om14
I fixed the footnote.pm module so that it works with multiple footnote text lines.  The Xref module calls are still commented out.

Revision 1.17  2000/11/30 16:19:27  jcl53
Upped the minimum Perl interpreter version from 5.002 to 5.005, just in case someone out there does something like trying to run this script on a non-Unix system...

Revision 1.16  2000/11/29 13:00:55  jhj4
*** empty log message ***

Revision 1.13  2000/11/29 03:30:14  jhj4
Commented out Xref in Footnote.pm
Attempting bug fix in Table.pm

Revision 1.12  2000/11/29 02:51:31  jhj4
Some changes to Table.pm

Revision 1.11  2000/11/29 01:41:16  om14
I fixed another small bug (actually, it's a big bug with a small fix): I forgot to call Xref on the text before footnotes.  Thanks, Ju! :)

Revision 1.10  2000/11/29 01:35:09  om14
I forgot to call Xref on the text that does not contain footnote text.  I fixed this.

Revision 1.9  2000/11/29 01:19:22  om14
I have included the function call for xref instead of the writeString call by Jason that was in place before hand.  This should fix the problem of only outputting the last line of footnote text.

Revision 1.8  2000/11/28 21:28:00  om14
i fixed the infinite loop bug.

Revision 1.7  2000/11/28 21:14:11  om14
i forgotwhati changed

Revision 1.6  2000/11/28 20:22:28  om14
I fixed the sequence numbering scheme in the Footnote.pm

Revision 1.5  2000/11/28 19:47:28  om14
I fixed the bug.  Basically, I only updated the current white space variable if I pattern matched.  This should take care of the undefined previous white space error and the tagging up of footrefs.

Revision 1.4  2000/11/28 18:35:17  jcl53
Fixed 'undefined variable' errors (on $currentWhiteSpace in LDMS::Footnote).
Added some useful includes (like LDMS::XMLOut, for example).
Tried to put something useful where the XREF module will be--but failed miserably.  I'll see if we can get this working properly...

Revision 1.3  2000/11/27 16:59:54  om14
Fixed some minor compilation errors such as writing $ instead of & and forgetting to type a $ in front a variable.

Revision 1.2  2000/11/27 16:50:29  om14
I fixed a minor bug in the code.  Basically, in the first version of the module, I created the cvs information manually before actually committing the file.  Therefore, there was some redundant information that I had to delete.

Revision 1.1  2000/11/27 16:48:22  om14
I have created the first version of the Footnote.pm module and updated the State.pm module to reflect the footnote "state machine."



