package LDMS::State;

# State.pm
#
# The state machine for the LDMS project.
#
# $Header: /home/LDMS/cvsroot/LDMS/LDMS/State.pm,v 1.9 2000/12/01 19:41:59 jcl53 Exp $
#

use strict;
use warnings;
require 5.005;
use lib '..';  # Make sure all modules are included AFTER this.


BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    # Set our version (for module version checking).
    $VERSION = do {my @r = (q$Revision: 1.9 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};

    @ISA = qw(Exporter);

    # Exported functions.
    @EXPORT = qw(&initState &ascendLevel &descendLevel &getHLevel &getVLevel &getLevelLabel &getSequenceLabel &getSequence &advanceLevel &labelInPath &initFootnoteNumbers &incrFootnoteNumber &getFootnoteNumber &setSequenceLabel);
    %EXPORT_TAGS = ( );

    # Exported global variables and optional exported functions.
    @EXPORT_OK = qw();

}
our @EXPORT_OK;

# Non-exported global variables.


# Initialize exported global variables.


# Initialize non-exported global variables.


# Create private global variables.
my @path = ();
my @sequence = ();
my @sequenceLabels = ();
my @footnoteNumbers = ();


# Prototype functions.
sub initState();
sub ascendLevel();
sub descendLevel($$);
sub getHLevel();
sub getVLevel();
sub getLevelLabel();
sub getSequence();
sub getSequenceLabel();
sub setSequenceLabel($);
sub advanceLevel();
sub labelInPath($);
sub initFootnoteNumbers();
sub incrFootnoteNumber($);
sub getFootnoteNumber($);


# Module destructor.
END { }


# Insert functions, code, &c., here.


#==============================================================================
# INITIALIZE STATE MACHINE
#
# initState()
#
# Preconditions:
#   - None.
#
# Postconditions:
#   - Sets global variables to default values.
#

sub initState() {

    @path = ();
    @sequence = ();

}


#==============================================================================
# DESCEND ONE LEVEL
#
# descendLevel($$)
#
# Preconditions:
#   - Requires two strings to be passed in:  level name and sequence name.
#
# Postconditions:
#   - Adds a new level to the relevant stacks.
#

sub descendLevel($$) {

    my $levelName = shift;
    my $sequenceLabelName = shift;

    unshift(@sequenceLabels, $sequenceLabelName);
    unshift(@path, $levelName);
    unshift(@sequence, 0);

}


#==============================================================================
# ASCEND ONE LEVEL
#
# ascendLevel()
#
# Preconditions:
#   - We should have descended at least one level before trying this.
#
# Postconditions:
#   - One level is removed from each stack.
# 

sub ascendLevel() {

    shift(@path);
    shift(@sequence);
    shift(@sequenceLabels);

}


#==============================================================================
# GET HORIZONTAL LEVEL
#
# getHLevel()
#
# Preconditions:
#   - None, though it's useful to have data in @sequence.
#
# Postconditions:
#   - Returns the sequence number of the current division.
#   - If the sequence stack is empty, returns 0.
#

sub getHLevel() {

    if (scalar(@sequence) > 0) {
	return $sequence[0];
    }
    else {
	return 0;
    }

}


#==============================================================================
# GET VERTICAL LEVEL
#
# getVLevel()
#
# Preconditions:
#   - None, though it helps to have data in @path.
#
# Postconditions:
#   - Returns the current depth of the structural division.
#   - Returns -1 if @path is empty.
#

sub getVLevel() {

    if (scalar(@path) > 0) {
	return (scalar(@path) - 1);
    }
    else {  # There's nothing in the path, doofus!
	return -1;
    }

}


#==============================================================================
# GET SEQUENCE PATH
#
# getSequence()
#
# Preconditions:
#   - It helps to have useful data in @sequence.
#
# Postconditions:
#   - Returns @sequence, initialized or not.
#

sub getSequence() {

    return @sequence;

}


#==============================================================================
# GET SEQUENCE LABEL
#
# getSequenceLabel()
#
# Preconditions:
#   - It helps to have useful data in @sequenceLabels.
#
# Postconditions:
#   - Returns sequence label of current division, if present.
#   - Returns "" if no sequence label present.
#

sub getSequenceLabel() {

    if (scalar(@sequenceLabels) > 0) {
	return $sequenceLabels[0];
    }
    else {
	return "";
    }

}


#==============================================================================
# GET LABEL OF CURRENT LEVEL
#
# getLevelLabel()
#
# Preconditions:
#   - It helps to have useful data in @path.
#
# Postconditions:
#   - Returns hierarchical label of current structural division, if present.
#   - Returns "" if no label defined.
#

sub getLevelLabel() {

    if (scalar(@path) > 0) {
	return $path[0];
    }
    else {
	return "";
    }

}


#==============================================================================
# ADVANCE ONE HORIZONTAL LEVEL
#
# advanceLevel()
#
# Preconditions:
#   - None.
#
# Postconditions:
#   - Current sequence level is incremented.
#

sub advanceLevel() {

    $sequence[0]++;
    return $sequence[0];

}


#==============================================================================
# CHANGE CURRENT LEVEL LABEL
#
# setSequenceLabel($)
#
# Preconditions:
#   - Requires a label to be passed in.
#
# Postconditions:
#   - Sets the current label to whatever was input.
#

sub setSequenceLabel($) {

    $sequenceLabels[0] = shift;

}


#==============================================================================
# CHECK IF PATH CONTAINS LABEL
#
# labelInPath($)
#
# Preconditions:
#   - Requires a label to be passed in.
#   - Helps if @path is not empty.
#
# Postconditions:
#   - Returns 1 if the label is present.
#   - Returns 0 otherwise.
#

sub labelInPath($) {

    my $label = shift;

    my $flag = 0;  # Has the label been found yet?

    for (my $i = 0; $i < scalar(@path); $i++) {
	if ((scalar(@path) > 0) &&
	    ($path[$i] eq $label)) {
	    $flag = 1;
	}
    }

    return $flag;

}


#==============================================================================
# INITIALIZE FOOTNOTE NUMBERS
#
# initFootnoteNumbers()
#
# Preconditions:
#   - None.
#
# Postconditions:
#   - @footnoteNumbers is cleared.
#

sub initFootnoteNumbers() {

    @footnoteNumbers = ();

}


#==============================================================================
# INCREMENT FOOTNOTE NUMBER
#
# incrFootnoteNumber($)
#
# Preconditions:
#   - Requires a new footnote number to be passed in.
#
# Postconditions:
#   - If the given position is undefined, sets to 1.0.
#   - If given position has a value, increments by 0.1.
#

sub incrFootnoteNumber($) {
    my $foo = shift;

    $foo = $foo - 1;

    if ($foo >= 0) {  # Make sure we're not out-of-bounds.
	if (! defined($footnoteNumbers[$foo])) {
	    $footnoteNumbers[$foo] = 1.0;
	} else {
	    $footnoteNumbers[$foo] = $footnoteNumbers[$foo] + .1;
	}
    }

}

#==============================================================================
# GET FOOTNOTE NUMBER
#
# getFootnoteNumber($)
#
# Preconditions:
#   - Requires a position to be passed in.
#
# Postconditions:
#   - Returns footnote number at that position, if present.
#   - Returns 1.0 otherwise.
#

sub getFootnoteNumber($) {

    my $foo = shift;

    $foo = $foo - 1;
    if (! defined($footnoteNumbers[$foo])) {
        return 1.0;
    } else {
        return $footnoteNumbers[$foo];
    }

}



1;
__END__;

=pod 
$Log: State.pm,v $
Revision 1.9  2000/12/01 19:41:59  jcl53
Commented functions more extensively, with preconditions and postconditions.

Revision 1.8  2000/11/30 16:19:27  jcl53
Upped the minimum Perl interpreter version from 5.002 to 5.005, just in case someone out there does something like trying to run this script on a non-Unix system...

Revision 1.7  2000/11/29 06:09:47  jcl53
Added a state stack that keeps track of the sequence label of the current division.  This is basically what the title numbers it as.

Revision 1.6  2000/11/28 18:28:30  jcl53
Added code to use footnote module in LDMS::DataText.
Fixed a few syntax problems introduced into the state machine.

Revision 1.5  2000/11/28 16:27:35  jcl53
Added a little comment to the 'use lib' lines.  Modules (especially LDMS ones) should be included after them so that testing compilation with 'perl -c' works.

Revision 1.4  2000/11/27 19:20:24  om14
I fixed some compilation errors in the State.pm module such as using @ vs. $ for indexing arrays, etc.

Revision 1.3  2000/11/27 16:48:22  om14
I have created the first version of the Footnote.pm module and updated the State.pm module to reflect the footnote "state machine."

Revision 1.2  2000/11/21 21:01:15  jcl53
Added the following line to all modules to make including modules less painful:

	use lib '..';

If you're using other modules, include them AFTER this line.  Thanks.

Revision 1.1  2000/11/17 20:02:44  jcl53
This module implements a state machine for the LDMS project.  Be nice to it, or it might break.

