package LDMS::FootRef;

# FootRef.pm
#
# The footnote reference module for the LDMS project.
#
# $Header: /home/LDMS/cvsroot/LDMS/LDMS/FootRef.pm,v 1.9 2000/12/01 18:20:27 om14 Exp $
#
# Author: Charles Shagong (cs58@cornell.edu)
#


use strict;
use warnings;
require 5.005;
use lib '..';
use LDMS::XMLOut;
use LDMS::State;


BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    # Set our version (for module version checking).
    $VERSION = do {my @r = (q$Revision: 1.9 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};

    @ISA = qw(Exporter);

    # Exported functions.
    @EXPORT = qw(&markupFootRefs);
    %EXPORT_TAGS = ( );

    # Exported global variables and optional exported functions.
    @EXPORT_OK = qw();

}
our @EXPORT_OK;

# Non-exported global variables.


# Initialize exported global variables.


# Initialize non-exported global variables.


# Create private global variables.
my $lines; # holds the number of lines to parse through 
my $currLineNum; # indicates current line number
my $currLine; # holds the current line of text to parse
my $beforeFootRef; # holds the substring preceding the foot reference
my $theFootRef; # holds the substring of the foot reference
my $afterFootRef; # holds the substring following the foot reference
my $tempSubStr; # temporary placeholder for substrings
my $subStrIndex; # holds the position of interest in the string
my $digitIndex; # holds the position of the digit of the footnote ref num
my $currDigit; # holds the current digit of the footnote ref num
my $currPower; # holds the exponent power necessary to parse footnote ref num
my $footRefNum; # holds the footnote reference number
my $footRefID; # holds the footref ID for target attribute 
my @target; # holds the target attribute for footnote reference
my $targetAttribute; # holds the target attribute as a string
my $numOfAttributes; # holds the number of target attribute components
my $attributeIndex; # holds the position of the target attribute
my @input; # holds the input passed to the function


# Prototype functions.
sub markupFootRefs(@);
sub debugStuff();


# Module destructor.
END { }


# Insert functions, code, &c., here.

##########
#
# markupFootRefs(@);
#
# Purpose:
# markup all footnote references
# in a given block of text
#
# Precondition:
# argument is a block of text passed in as an 
# array where each element is a line of text
#
# Postcondition:
# block of text is written out with all footnote
# references marked up with XML tags
#
#

sub markupFootRefs(@)
{
   # assign the input data to a variable
   @input = @_;

   # get number of lines to parse through
   $lines = scalar(@input);

   for($currLineNum = 0; $currLineNum < $lines; $currLineNum++)
   {
      $currLine = $input[$currLineNum];
      if($currLine =~ /\(FOOTNOTE \d+/ )
      {
         # get the index of the first character of the footnote reference
         $subStrIndex = index($currLine, "\(FOOTNOTE");

         # extract the substring preceding the footnote reference
         $beforeFootRef = substr($currLine, 0, $subStrIndex);

         # work with the substring consisting of the footnote 
         # reference and the substring following it
         $tempSubStr = substr($currLine, $subStrIndex);

         # get the index of the last character of the footnote reference
         $subStrIndex = index($tempSubStr, "\)");

         # extract the footnote reference
         $theFootRef = substr($tempSubStr, 0, $subStrIndex+1);

         # extract the substring following the footnote reference
         $afterFootRef = substr($tempSubStr, $subStrIndex+1);

         # extract the footnote reference number
         $footRefNum = 0; # reset variable
         $currPower = 0;  # reset variable
         for($digitIndex = $subStrIndex-1; $digitIndex >= 10; $digitIndex--)
         {
            # look at the current digit
            $currDigit = substr($theFootRef, $digitIndex, 1);

            # add the current digit to the footnote reference number
            # while determining what power of ten it represents
            $footRefNum = $footRefNum + ($currDigit * (10 ** $currPower));

            # increment the exponent power
            $currPower++;
         }

         # determine part of the attribute for footnote reference XML tag
         @target = &getSequence();

         # determine the subID for current footnote reference
         $footRefID = &getFootnoteNumber($footRefNum);

         # add the footnote reference ID to the target attribute
         unshift(@target, $footRefID);

         # write out the full target attribute to a string
         $targetAttribute = "";
         $numOfAttributes = scalar(@target);
         for($attributeIndex = 0; $attributeIndex < $numOfAttributes; $attributeIndex++)
         {
            $targetAttribute = $targetAttribute . $target[$attributeIndex];
            if($attributeIndex != $numOfAttributes-1)
            {
               # append a dash as long as last attribute component
               # was not just appended
               $targetAttribute = $targetAttribute . "-";
            }
         }

         # write out the marked up text
         &WriteString($beforeFootRef);
         &BeginTag("FOOTREF");
         &AddAttribute("TARGET", $targetAttribute);
         &WriteString($theFootRef);
         &EndTag();
         &WriteString($afterFootRef);

         # debug function can be uncommented below
         # &debugStuff();

      }

      else
      {
         # simply write out the line because
         # there is no text to markup
         &WriteString($currLine);
      }
   }


}

sub debugStuff()
{
   print "$currLine\n";
   print "$beforeFootRef\n";
   print "$theFootRef\n";
   print "$afterFootRef\n";
   print "$footRefNum\n";

   print "\n";
}


 
1;
__END__;

=pod 
$Log: FootRef.pm,v $
Revision 1.9  2000/12/01 18:20:27  om14
Changes to FootRef.pm: added use for LDMS::State and uncommented out line 151 which refers to the footnote number
Added authoring for Error.pm and Footnote.pm

VS: ----------------------------------------------------------------------

Revision 1.8  2000/12/01 18:10:57  jcl53
Actually declares that it wants to use the LDMS::State module now.

Revision 1.7  2000/12/01 03:42:08  cs58
Added authorship information

Revision 1.6  2000/11/30 16:19:27  jcl53
Upped the minimum Perl interpreter version from 5.002 to 5.005, just in case someone out there does something like trying to run this script on a non-Unix system...

Revision 1.5  2000/11/28 18:35:17  jcl53
Fixed 'undefined variable' errors (on $currentWhiteSpace in LDMS::Footnote).
Added some useful includes (like LDMS::XMLOut, for example).
Tried to put something useful where the XREF module will be--but failed miserably.  I'll see if we can get this working properly...

Revision 1.4  2000/11/27 20:10:47  om14
I fixed the FootRef.pm module so that it would compile correctly.  I added @input to store the input passed to the function.

Revision 1.3  2000/11/27 19:49:01  om14
I added the perl module stuff to this module.
