package LDMS::XMLOut;

# XMLOut.pm
#
# Collects and outputs the XML-ized data.
#
# $Header: /home/LDMS/cvsroot/LDMS/LDMS/XMLOut.pm,v 1.16 2000/11/28 16:27:35 jcl53 Exp $
#

use strict;
use warnings;
use lib '..';  # Make sure all modules are included AFTER this.
use IO;           # Required to use XML::Writer to write to files.
use XML::Writer;
use LDMS::Error;
require 5.002;


BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    # Set our version (for module version checking).
    $VERSION = do {my @r = (q$Revision: 1.16 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};

    @ISA = qw(Exporter);

    # Exported functions.
    @EXPORT = qw(&Initialize &BeginTag &AddAttribute &EndTag &WriteString &Finalize);
    %EXPORT_TAGS = ( );

    # Exported global variables and optional exported functions.
    @EXPORT_OK = qw();

}
our @EXPORT_OK;

# Non-exported global variables.


# Initialize exported global variables.


# Initialize non-exported global variables.


# Create private global variables.
my @xmlOut_startTagList = (); #must collect all start tags
                              #to append attributes when found
my @xmlOut_openStartTag = (); #stack to remember location of last starttag
my @xmlOut_textList = ();     #Each element represents the text between tags
my @xmlOut_endTagList = ();   #must preserve order of end tags
                              #noting the starttaglist length for
                              #accuracy
my $xmlOut_textBuffer = "";   #null string buffer to collect text between tags
my $xmlOut_outputFile = "";   # Output filename.


# Prototype functions.
sub Initialize($outputFilename);
sub BeginTag($tagName);
sub AddAttribute($attributeName$attributeKey);
sub EndTag();
sub WriteString($stringToWrite);
sub Finalize;
sub DumpStack();

# Module destructor.
END { }


#pre: Program just beginning to run
#post: outputfile opened, if problems encountered, exit
sub Initialize($outputFilename)
{
   my $xmlOut_outputFilename = shift;

   # Detaint by making sure we're just using alphanumerics, hyphens,
   # and slashes.
   if ($xmlOut_outputFilename =~ /^([-\w.\/]+)$/) {
       $xmlOut_outputFilename = $1;
   }
   else {
       warn "Tainted data in $xmlOut_outputFilename";
       warn "Setting filename to ./output.xml";
       $xmlOut_outputFilename = './output.xml';
   }

   my $outputFile = new IO::File(">$xmlOut_outputFilename");
   return $outputFile;

} # Initialize

#pre: starttag encountered
#post: puts a starttag on list, flushes textBuffer to textlist
sub BeginTag($tagName)
{
   my $location = @xmlOut_startTagList;
   push(@xmlOut_openStartTag,$location);
   my $listRef = [$_[0]];
   push(@xmlOut_startTagList,$listRef);
   push(@xmlOut_textList,$xmlOut_textBuffer);
   $xmlOut_textBuffer = "";
} #BeginTag

#pre: attribute determined
#post:  adds an attribute to the most recently unended
# starttag
sub AddAttribute($attributeName $attributeKey)
{
   #first find the location of the last open start tag
   my $height = @xmlOut_openStartTag; 
   my $location = $xmlOut_openStartTag[$height-1];
   my $startTag = $xmlOut_startTagList[$location];
   push(@$startTag, $_[0] => $_[1]);
   $xmlOut_startTagList[$location] = $startTag;
} #AddAttribute

#pre: endtag encountered
#post: pushes an endtag onto the endtag list, noting how tall
# the start tag list was, flushes textBuffer to textList
sub EndTag()
{
   #first find the location of the last open start tag
   my $height = @xmlOut_openStartTag; 
   my $location = $xmlOut_openStartTag[$height-1];
   
   #get the name
   my $startTagName = $xmlOut_startTagList[$location][0];

   #put the height of starttag list so we know when to endtag 
   my $startHeight = @xmlOut_startTagList;
   my $listRef = [$startTagName, $startHeight];
   
   push(@xmlOut_endTagList, $listRef);

   #the tag on top has ended
   pop(@xmlOut_openStartTag);

   push(@xmlOut_textList,$xmlOut_textBuffer);
   $xmlOut_textBuffer = "";
} #EndTag

#pre: nontag text encountered
#post: adds text to TextBuffer
sub WriteString($stringToWrite)
{
   #if didn't pass anything in, don't do anything
   if ($_[0])
   {   
      #concatenate text to TextBuffer
      $xmlOut_textBuffer = $xmlOut_textBuffer.$_[0];
   } 
} #WriteString

#pre: done parsing inputfile, want to write to outputfile
#post:  the function takes all three stacks,
# feeding them through the XML::Writer to the
# outputfile
sub Finalize
{

    # Get our output file.
    my $outputFile = shift;

    #create the XML Writer
    my $writer = new XML::Writer(OUTPUT => $outputFile);
    
    #a little disclaimer
    $writer->comment('This file created by LDMS v1.2');    

    shift(@xmlOut_textList); #there is no text before the first tag,
                             # but an empty string exists from the 
                             #first call of begintag
    
    push(@xmlOut_textList,""); #this is to keep the if from going out of bounds  
    
    my $endTagRef = ["dummy",-1]; #this is to keep the while loop
    push(@xmlOut_endTagList,$endTagRef); #from running out of bounds
    
    #initialize some loop variables
    $endTagRef = shift(@xmlOut_endTagList);
    my $length = @xmlOut_startTagList;
    my $startTagRef = "";
    my @startTagArgs = {};
    
    #pre: empty file, tagging complete
    #post: tags and text properly inserted to OUTPUTFILE 
    for(my $i = 1; $i <= $length; $i++)
    {
	#write next starttag
	$startTagRef = $xmlOut_startTagList[$i-1];
	@startTagArgs = @$startTagRef;
	$writer->startTag(@startTagArgs);
	
	#write all text since that starttag
	my $text = shift(@xmlOut_textList);
	if($text ne "")
	{    
	    $writer->characters($text);
	} #if(text!="")
	
	#write all endtags that occur before the next starttag
	
	while($$endTagRef[1] == $i)
	{
	    $writer->endTag($$endTagRef[0]);
	    $endTagRef = shift(@xmlOut_endTagList);
	    
	    #write all text since that starttag
	    $text = shift(@xmlOut_textList);
	    if($text ne "")
	    {    
		$writer->characters($text);
	    } #if(text!="") 
	} #while(endtagref[1]==i)
    } #for(i=1;i<=length;i++)
    
    $writer->end();
    $outputFile->close();
    
} #Finalize

#pre: somethings wrong, in the middle of tagging, user wants list status
#post:  status outputted without destroying list integrity
sub DumpStack()
{

   #go through startTagList
   &printErrMsg("StartTags with Attributes: \n");
   #pre: StartTags not printed
   #post: StartTags printed  
   &printErrMsg("All StartTags: \n");
   my $i = 0;
   foreach my $x (@xmlOut_startTagList)
   {
      &printErrMsg("$i: ");
      foreach my $y (@$x)
      {
         &printErrMsg($y);
      } # foreach $y (@$x)
      &printErrMsg("\n");
      $i = $i + 1;

   } #foreach $x (@xmlOut_startTagList)
   &printErrMsg("\n");

   #pre: open starttag heights not printed
   #post: open starttag heights printed 
   &printErrMsg("Open starttag heights, most recent last: \n");
   foreach my $x (@xmlOut_openStartTag)
   {
       &printErrMsg($x);
   } # foreach $x (@xmlOut_openStartTag)
   &printErrMsg("\n");

   #pre: endtags not printed
   #post: endtags printed
   &printErrMsg("EndTags: \n");
   foreach my $x (@xmlOut_endTagList)
   {
      foreach my $y (@$x)
      {
	  &printErrMsg($y);
      } # foreach $y (@$x)
     &printErrMsg("\n");
   } # foreach $x (@xmlOut_openStartTag)
   &printErrMsg("\n");

   #pre: textlist not printed
   #post: testlist printed
   &printErrMsg("Text List: \n");
   $i = 0;
   foreach my $x (@xmlOut_textList)
   {
       &printErrMsg("\n $i: \n $x \n");
   } # foreach $x (@xmlOut_textList)

} #DumpStack

1;
__END__;

=pod 
$Log: XMLOut.pm,v $
Revision 1.16  2000/11/28 16:27:35  jcl53
Added a little comment to the 'use lib' lines.  Modules (especially LDMS ones) should be included after them so that testing compilation with 'perl -c' works.

Revision 1.15  2000/11/26 18:59:49  bww3
corrected the bounds checking on WriteString, consulted with Ju
on how to do it properly - bww3

Revision 1.14  2000/11/26 16:55:12  bww3
made writeString check for uninitialized value - bww3

Revision 1.13  2000/11/21 20:54:28  jcl53
Fixed include of LDMS::Error to not trigger warnings when the module is checked through 'perl -c'.  Might need to be fixed in everything else.

Revision 1.12  2000/11/21 20:38:09  bww3
corrected bugs created by DumpStack that caused it not to compile - bww3

Revision 1.11  2000/11/21 20:24:41  bww3
added DumpStack, a function that prints all lists to error module
without changing anything - bww3 2000-11-21 15:30

Revision 1.10  2000/11/15 01:23:40  jcl53
Moved the output filename munging (i.e. tacking '.xml' onto filenames) into the main loop, as that is where filename control should be.  After all, we don't want to attach '.xml' onto a filename the user explicitly provides us...  The customer is always right, after all!

Revision 1.9  2000/11/11 17:33:52  jcl53
Fixed a smallish bug in which 'writer' was being used instead of '$writer'.

Also changed the output back to using objects of type IO::File as its handle.  I really wish XML::Writer could just use plain filehandles, but for now, we'll work around it...

Revision 1.8  2000/11/11 14:47:53  bww3
11-11-2000 removed bug in EndTag(), it didn't get the startHeight bww3

Revision 1.7  2000/11/11 02:14:18  bww3
forgot to add one line - bww3

Revision 1.6  2000/11/11 02:10:17  bww3
added stack to correctly find most recent unended start tag

Revision 1.5  2000/11/10 06:32:20  jcl53
XMLOut has been modified to work with filehandle arguments.  This is cleaner than making a module-global filehandle to hit XML::Writer with.

LDMS::LP=>getCiteTitleNumber has been simplified a bit, for ease of debugging.  The original code is commented out.  This should change once every module's up and running.

Revision 1.4  2000/11/10 02:34:06  jcl53
XMLOut.pm can now be cleanly included.  Now working on the others...

Functionality is not an issue yet.  I need to make sure all of these modules play well with others...

Revision 1.3  2000/11/10 01:54:08  jcl53
Corrected a few odd things in the logs, such as the status messages left over from the template (oops).

Revision 1.2  2000/11/10 01:41:56  jcl53
Renamed InputModule.pm to Input.pm just because I thought it looked neat.

In fact, Input.pm has been confirmed to be loadable as a module.  The rest--I'm working on...

Speaking of the rest, the actual code from the separate script files has been C-x i'ed into the corresponding module file.  (For you non-emacs folks out there, that means they've been inserted.)  These AREN'T confirmed to work as modules.  As I said, I'm working on them...

On a final note, the 'my' keyword is our friend, folks.  Use it like Mountain Dew on an all-nighter.

Revision 1.1  2000/11/09 20:55:07  jcl53
Added skeleton Perl module files to the appropriate directory.
