package LDMS::Footnote;

# Footnote.pm
#
# The footnote module for the LDMS project.
#
# $Header: /home/LDMS/cvsroot/LDMS/LDMS/Footnote.pm,v 1.3 2000/11/27 16:59:54 om14 Exp $
#

use strict;
use warnings;
require 5.002;
use lib '..';


BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    # Set our version (for module version checking).
    $VERSION = do {my @r = (q$Revision: 1.3 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};

    @ISA = qw(Exporter);

    # Exported functions.
    @EXPORT = qw(&tagFootnotes);
    %EXPORT_TAGS = ( );

    # Exported global variables and optional exported functions.
    @EXPORT_OK = qw();

}
our @EXPORT_OK;

# Non-exported global variables.


# Initialize exported global variables.


# Initialize non-exported global variables.


# Create private global variables.
my @text; # ascii text input
my $currentLine; # index for the current line being processed within @text
my $errorMessage; # holds the text of an error message
my $footnoteNumber; # holds the parsed number of a footnote
my $currentWhiteSpace; # holds the number of white space characters for $currentLine
my $previousWhiteSpace; # holds the number of white space characters for the previous line
my $footnoteTextFlag; # indicates whether we are currently processing footnote text (1) or not (0)
my $partialText; # holds the portion of a line that corresponds to footnote text
my @footnoteText; # holds the lines of text of a footnote


# Prototype functions.
sub tagFootnotes(@);
sub processFootnote();


# Module destructor.
END { }


# Insert functions, code, &c., here.

# tagFootnotes(@)
#
# Precondition: The input is an array of all text lines within a titledata element.
#
# Postcondition: Footnote numbers in state machine module are updated.
#                Footnote XML tags are output.
#                Xref module is called with all remaining text.

sub tagFootnotes(@)
{
   # initialize private global variables
   @text = @_;
   $currentLine = 0;
   $previousWhiteSpace = "";

   # perform an error check for empty input
   if (! defined($text[$currentLine]) )
   {
      $errorMessage = "No input was passed to the tagFootnotes function.";
      &printErrMsg($errorMessage);
      return;
   }

   # check for footnote pattern match
   while ( $currentLine < scalar(@text) )
   {
      $_ = $text[$currentLine];
      if ( /^(\W+)\(FOOTNOTE\W+(\d+)\)/ )
      {
         $currentWhiteSpace = length($1);
         if ( $previousWhiteSpace != "" && $previousWhiteSpace != $currentWhiteSpace )
         {
            $footnoteNumber = $2;
            $partialText = $';

            # we need to increment the footnote number in the state machine module
            &incrFootnoteNumber($footnoteNumber);

            # we must output start tag and its attributes before we start processing footnote text
            &BeginTag("FOOTNOTE");
            &AddAttribute("FNUMBER",$footnoteNumber);
            &AddAttribute("EID",&getFootnoteNumber($footnoteNumber));

            # now we need to find the end of the footnote and process its text
            &processFootnote();

            # we need to output the end tag after processing the footnote text
            &EndTag("FOOTNOTE");

            # finally, we must flush the @footnoteText buffer
            @footnoteText = ();
         } else {
            # no pattern match exists for current line
            push(@footnoteText,$text[$currentLine]);
            $previousWhiteSpace = $currentWhiteSpace;
            $currentLine = $currentLine + 1;
         }
      } else {
         # no pattern match exists for current line
         push(@footnoteText,$text[$currentLine]);
         $previousWhiteSpace = $currentWhiteSpace;
         $currentLine = $currentLine + 1;
      }
   }
}


# processFootnote()
#
# Precondition:
#
# Postcondition:

sub processFootnote()
{
   $footnoteTextFlag = 1;
   while ( $footnoteTextFlag )
   {
      push(@footnoteText,$partialText);
      # the right justified dates yield an overall margin of 73
      # however, the margin for the actual text is 71
      # therefore, 71 - 20 or 51 is first used to determine the end of footnote text
      if ( length($text[$currentLine]) < 51 )
      {
         $previousWhiteSpace = "";
         $currentLine = $currentLine + 1;

         # we must call the Xref module function to process the footnote text for xref and footrefs
         # ****ADD XREF MODULE FUNCTION CALL HERE****

         # we must flush the @footnoteText buffer
         @footnoteText = ();

         $footnoteTextFlag = 0;
      } else {
         # check if we are currently at the last line of text
         if ( $currentLine == scalar(@text) - 1 )
         {
            # we must call the Xref module function to process footnote text for xref and footrefs
            # ****ADD XREF MODULE FUNCTION CALL HERE****

            # we don't have to flush the @footnoteText buffer since this is the last line of text

	    $footnoteTextFlag = 0;
         } else {
            $previousWhiteSpace = $currentWhiteSpace;
            $currentLine = $currentLine + 1;
            $_ = $text[$currentLine];
            /^(W+)/;
            $currentWhiteSpace = length($1);

            # check for correct white space indentation and
            #   whether the line is not a new subsection
            if ( ($currentWhiteSpace - $previousWhiteSpace == 3) && (! /^W+\(\w\)/) )
            {
               $partialText = $text[$currentLine];
            } else {
               # we must call Xref module function to process footnote text for xrefs and footrefs
               # ****ADD XREF MODULE FUNCTION CALL HERE****

               # since we processed an extra line here for footnote text,
               # we must perform a rollback to re-process the line for footnotes
               # therefore, the currentLine remains unchanged
               @footnoteText = ();
               $previousWhiteSpace = "";

	       $footnoteTextFlag = 0;
            }
         }
      }
   }
}



1;
__END__;

=pod 
$Log: Footnote.pm,v $
Revision 1.3  2000/11/27 16:59:54  om14
Fixed some minor compilation errors such as writing $ instead of & and forgetting to type a $ in front a variable.

Revision 1.2  2000/11/27 16:50:29  om14
I fixed a minor bug in the code.  Basically, in the first version of the module, I created the cvs information manually before actually committing the file.  Therefore, there was some redundant information that I had to delete.

Revision 1.1  2000/11/27 16:48:22  om14
I have created the first version of the Footnote.pm module and updated the State.pm module to reflect the footnote "state machine."



