functor TextGameFn (P_WHITE: PLAYER)
                   (P_BLACK: PLAYER) :>
                   sig val start : int * Time.time -> unit end = struct

   open Types
   open Game

   val I = Int.toString

   fun display(g:game) =
    let
      val boardSize = getBoardSize(g)
      fun printTop(c:int) =
        if (c = boardSize) then print("\n")
        else (print(Int.toString(c) ^ " ");
              if (c < 10) then print(" ") else (); printTop(c+1))
      fun printRow(x:int,y:int) =
        case (y<boardSize) of
           false => print("\n")
         | true  => ((case getLocation((y,x),g) of
                        NONE => print("-  ")
                      | SOME(BLACK) => print("b  ")
                      | SOME(WHITE) => print("w  ")); printRow(x,y+1))
      fun printRows(x:int) =
        case (x<boardSize) of
           false => ()
         | true  => (print(Int.toString(x) ^ "  ");
                     if (boardSize > 9) andalso
                        (x < 10) then print(" ") else ();
                     printRow(x,0);
                     printRows(x+1))

      fun pieceToString(p: color):string =
        case p of BLACK=> "BLACK" | WHITE => "WHITE"
      val _ = print("   ")
      val _ = if (boardSize > 9) then print(" ") else ()
      val _ = printTop(0)
      val _ = printRows(0)
      val _ = print("next: " ^ pieceToString(getNextPlayer(g)) ^ "\n")
      val _ = case getWinner(g) of
                 NoWinner => print("no winner\n")
               | Win(BLACK) => print("Black won\n")
               | Win(WHITE) => print("White won\n")
               | Draw       => print("Draw\n")
      val _ = print("black has captured: "^Int.toString(blackCaptures(g))
                    ^"\n")
      val _ = print("white has captured: "^Int.toString(whiteCaptures(g))
                    ^"\n")
      val _ = print("white time left: "^Time.toString(timeLeft(g,WHITE))
                    ^"\n")
      val _ = print("black time left: "^Time.toString(timeLeft(g,BLACK))
                    ^"\n")
    in
      ()
    end


   val message = let fun addCR(s) = s^"\n" in  print o addCR end


    fun getHumanMove():coordinate =
      let
        fun processLine ():coordinate =
          case TextIO.inputLine (TextIO.stdIn) of
            "" => raise Fail ""
          | s =>
            let
              (* get entire substring *)
              val s:substring = Substring.extract(s,0,NONE)
              val t = Substring.tokens Char.isSpace s
              val (x,y) = case t of
                  x::xs::nil =>  (Int.fromString(Substring.string x),
                                  Int.fromString(Substring.string xs))
                | _ => raise Fail ""
              val x:int = case x of
                            NONE => raise Fail ""
                          | SOME(x) => x
              val y:int = case y of
                            NONE => raise Fail ""
                          | SOME(y) => y
            in
              (x,y)
            end
      in
        ((*printGame(g);*)
        TextIO.flushOut (TextIO.stdOut);
        print("Enter move: ");
        TextIO.flushOut (TextIO.stdOut);
        processLine ()
          handle e => (message "Unrecognised move. Enter as row space column.";
                       getHumanMove()))
      end


   (* starts a new game with board size as sole argument *)
   (* Started game has white's first move being in the center *)
   (* and it being black's turn *)
   fun start (n:int, timePerPlayer:Time.time):unit = let
       val thisgame = newGame(n,timePerPlayer)
       val _ = display(thisgame)
       val white = P_WHITE.createPlayer WHITE n timePerPlayer
       val black = P_BLACK.createPlayer BLACK n timePerPlayer

       fun loop(g:game):unit = let
         val move = lastMove(g)

         val ai_move = case getNextPlayer(g) of
                         WHITE => white(move,timeLeft(g,WHITE))
                       | BLACK => black(move,timeLeft(g,BLACK))

         val move = case ai_move of
                         Human => Move(getHumanMove())
                       | Move(c) => Move(c)
                       | _ => raise Fail "Not handled move"

         val game' = SOME(step(g,move)) handle
                 InvalidMove(x,y) =>
                    (message ("Invalid move ("^(I x)^","^(I y)^")");
                     loop g; NONE)
       in
          case game' of
               NONE => ()
             | SOME(x) => (display(x);
                           if getWinner(x) = NoWinner then loop(x) else ())

       end
   in
       loop(thisgame)
   end
end

