functor GUIGameFn (P_WHITE: PLAYER) (P_BLACK: PLAYER) :>
sig
 val start : int * Time.time -> unit
end = struct

   fun start (n:int,timePerPlayer:Time.time):unit = let
       open Types
       open Game
       open TkTypes
       open SmlTk
       val I = Int.toString
       val IT = Int32.toString o Time.toSeconds

       val white = P_WHITE.createPlayer WHITE n timePerPlayer
       val black = P_BLACK.createPlayer BLACK n timePerPlayer


       (* Stores the current game while we are waiting for *)
       (* human input via mouse clicks *)
       val thisgame = ref (newGame(n, timePerPlayer))

       (* Stores the last state of the board. *)
       (* Used to find which board locations have changed, and hence *)
       (* which graphics need to be refreshed *)
       val oldBoard = Array.array(n*n,NONE)

       (* ID for the main window. *)
       val mainId = newWinId()

       (* ID for the canvas with the board icons. *)
       val boardId = newWidgetId()

       (* ID's for the labels. *)
       val currentPlId = newWidgetId()
       val lastmoveId = newWidgetId()
       val timeleftId = newWidgetId()
       val captureId = newWidgetId()

       (* Unique image ID's for the board graphics *)
       val imgIDs = List.tabulate(n*n,fn _ => newCItemId())

       (* Prints a message for the user *)
       val message = let fun addCR(s) = s^"\n" in  print o addCR end

       (* List of valid positions (x,y) for the current game. *)
       val positions =
       foldr op@ [] (List.tabulate(n,fn y => List.tabulate(n,fn x=>(x,y))))

       (* Board position to screen pixel location. *)
       fun getDisplayCoord(x:int,y:int):Coord = mkCoord (x*31+15, y*31+15)

       (* String value of coordinates *)
       fun coord2str(x:int,y:int):string = "("^(I x)^","^(I y)^")"

       (* String equavalence of player. *)
       fun player2string(x:color):string =
           if x = WHITE then "White" else "Black"

       (* GIF filename for the given for board position. *)
       fun posFile(x:int, y:int):string =
           "pos" ^
           (case y of 0  => "N" | _ => if y = n - 1 then "S" else "") ^
           (case x of 0  => "W" | _ => if x = n - 1 then "E" else "") ^
           ".gif"

       (* GIF filename for board position taking into consideration *)
       (* player-pieces. *)
       fun pieceFile(g:game, x:int, y:int) =
           (case getLocation((x,y),g) of
                 NONE => ""
               | SOME(BLACK) => "black-"
               | SOME(WHITE) => "white-") ^ (posFile(x,y))

       (* Returns an SmlTk icon with the given id, coordinates and *)
       (* image file. The mouse click event is bound to the click *)
       (* function defined later *)
       fun getIcon(id, pos as (x,y), image) =
           CIcon {citemId = id, coord = getDisplayCoord pos, configs = [],
                  iconkind = FileImage ("gifs/" ^ image,newImageId()),
                  bindings = [BindEv(ButtonPress NONE,mkAction(click pos))]}

       (* Updates the window to match the game state *)
       and display(g: game) = let
           val nm = case getWinner(g) of
                         NoWinner => "Current move: "^
                                     (player2string (getNextPlayer g))
                       | Win(x) => ((player2string x) ^ " won")
                       | Draw => "Game ended in a draw"



           val _ = setConf currentPlId [Text nm]

           val _ = case getWinner(g) of
                        (Draw | Win(_))  => UW.info nm
                      | _ => ()


           val lastm = case lastMove(g) of Move(x) => coord2str x | _ => ""
           val _ = setConf lastmoveId [Text ("Last move: "^lastm)]
           val tl = "Time left: (White "^(IT(timeLeft(g,WHITE)))^
                    ")      ("^
                    "Black "^(IT(timeLeft(g,BLACK)))^")"

           val cp = "Captures by: (White "^(I(whiteCaptures g))^
                    ")      ("^
                    "Black "^(I(blackCaptures g))^")"

           val _ = setConf timeleftId [Text tl]
           val _ = setConf captureId [Text cp]
       in
           List.app
              (fn (x:int, y:int) => let
                  val piece = getLocation ((x,y),g)
                  val pos = y*n + x
                  val id = List.nth(imgIDs,pos)
               in
                  if Array.sub(oldBoard,pos) <> piece then
                    (* This board location has changed. *)
                    (* Update the display to reflect this change. *)
                    (delCItem boardId id;
                     addCItem boardId (getIcon(id,(x,y),pieceFile(g,x,y)));
                     Array.update(oldBoard,pos,piece))
                  else
                    ()
               end) positions
        end

       (* Handles clicks from the user. If the move is valid, it continues *)
       (* the game if it has not been won. If the move was invalid, it *)
       (* waits for another valid input. *)
       and click (x,y) _ = let
           val game' = SOME(step(!thisgame,Move(x,y)))
                       handle InvalidMove(p) =>
                            (UW.error ("Invalid move at "^(coord2str p)); NONE)
                       | GameOver(_) => NONE
        in
           case game' of
                SOME(x) => (display(x);
                            if getWinner(x) = NoWinner then loop(x)
                            else thisgame := x)
              | NONE => ()
        end

       (* Runs the game till a human input is required. *)
       (* In case of AI Vs AI matches, this function returns only when the *)
       (* game is over. In case there is atleast one human player, this *)
       (* function returns when the user input is required. The loop is *)
       (* restarted by the click function when a valid mouse click is made. *)
       and loop(g:game):unit = let
         val move = lastMove(g)
         val ai_move = case getNextPlayer(g) of
                         WHITE => white(move,timeLeft(g,WHITE))
                       | BLACK => black(move,timeLeft(g,BLACK))
       in
          case ai_move of
               Human => (thisgame := g)
             | Move(x) => let
                 val game' = SOME(step(g,Move(x)))
                             handle InvalidMove(pos) => (loop g; NONE)
               in
                  case game' of
                       SOME(x) => (display(x);
                                   if getWinner(x) = NoWinner then loop(x)
                                   else thisgame := x)
                     | NONE => ()
               end
             | _ => raise Fail "Move not handled"
       end

       (* The initial board displayed has no pieces on it. Generate the *)
       (* appropriate list iof images for the empty game board *)
       val icons = ListPair.map (fn (id,pos) => getIcon(id,pos,posFile pos))
                                (imgIDs, positions)

       (* Holder for the picture elements *)
       val gamecanvas = Canvas {widId = boardId, scrolltype = NoneScb,
                                 citems = icons, packings = [],
                                 configs = [Width (n*31), Height (n*31)],
                                 bindings = []}

       (* Whose move *)
       val gamecurrentmove = Label {widId = currentPlId, packings = [],
                              configs = [Text ("Current move: ")],
                              bindings = []}

       (* Last move *)
       val gamelastmove = Label {widId = lastmoveId, packings = [],
                              configs = [Text ("Last move: ")],
                              bindings = []}


       (* Holds gamecurrentmove and gamelastmove *)
       val labelframe1 = Frame {widId = newWidgetId(), bindings=[],
                              widgets = Pack [gamecurrentmove, gamelastmove],
                              configs = [], packings=[Side Left, PadX 30]}

       (* Last move *)
       val gametimeleft = Label {widId = timeleftId, packings = [],
                              configs = [Text ("White:      Black:")],
                              bindings = []}

       (* The bottom label *)
       val gamelabel = Label {widId = captureId, packings = [],
                              configs = [Text ("Do NOT click while " ^
                                               "the AI is thinking.")],
                              bindings = []}


       (* Holds gametimeleft gamelabel *)
       val labelframe2 = Frame {widId = newWidgetId(),
                              widgets = Pack [gametimeleft, gamelabel],
                              configs = [], packings=[Side Right], bindings=[]}

       (* Holds the 2 lable holders *)
       val labelframe = Frame {widId = newWidgetId(),
                              widgets = Pack [labelframe1, labelframe2],
                              configs = [], packings=[], bindings=[]}

       (* Holds the Canvas above the Label *)
       val gameframe = Frame {widId = newWidgetId(),
                              widgets = Pack [gamecanvas, labelframe],
                              configs = [], packings=[], bindings=[]}

       (* The main window. Starts the game loop when initialized. *)
       val gamewin = {winId = mainId,
                       config = [WinTitle ("CS312 PS6: " ^ P_WHITE.getName() ^
                                           " Vs. " ^ P_BLACK.getName()),
                                 WinMinSize (n*31,n*31),
                                 WinMaxSize (n*31,n*31 + 40)],
                       widgets = Pack [gameframe],
                       bindings = [],
                       init = (fn _ => (display(!thisgame); loop(!thisgame)))}
   in
      (SmlTk.init();
       startTcl [mkWindow gamewin])        (* showtime! *)
   end
end
