signature GAME = sig

    type color = Types.color
    type boardLoc = Types.boardLoc
    type coordinate = Types.coordinate

    (* Human datatype to tell the UI that it must get the move from I/O
     * Start datatype is given to a player if it is starting the game *)
    type move = Types.move

    type game

    (* datatype indicating who has won, if its a draw or if it is
     * still in progress *)
    datatype winner = NoWinner | Win of color | Draw

    (* Raised by step when a move is attempted after the game has ended *)
    exception GameOver of winner

    (* Raised by step when an invalid move is attempted *)
    exception InvalidMove of coordinate

    (* Gets value of location (x,y) on the board
     * 1st argument is x, 2nd is y. returns NONE if nothing there.
     * Checks that (x,y) is a a valid location. *)
    val getLocation: coordinate * game -> boardLoc

    (* Checks to see if move is a valid move given the current game state *)
    val validMove: move * game -> bool

    (* newGame(n,t) returns a new game object with a n*n board. The board
     * is initially empty. Each player has up to time t to finish the game 
     * and it is black's move. *)
    val newGame: int * Time.time -> game

    (* Takes in current game state, and the location of the next move
     * and returns the new game state with the current move applied to it.
     * Raises GameOver(p) if the game is already over. p should never be
     * NoWinner.
     * Raises InvalidMove(x,y) if the move is invalid. x,y are the
     * coordinates of the failed move *)
    val step: game * move -> game

    (* returns the last move of the current game
     * returns Start if the game passed in came directly from a call
     * to newGame *)
    val lastMove: game -> move

    (* gets the winner of the game. Returns None if no one has won
     * and it is not a Draw *)
    val getWinner: game -> winner

    (* gets the color of the player who plays next *)
    val getNextPlayer: game -> color

    (* gets the size of the board. Board is always square *)
    val getBoardSize: game -> int

    (* gets how much time left a current player has *)
    val timeLeft: (game * color) -> Time.time

    (* *************************************************** *)
    (* **********    SPECIFICATION CHANGE BELOW ********** *)
    (* *************************************************** *)

    (* Returns the number of captures black has made, same as
       half the number of white stones removed from the board *)
    val blackCaptures: (game) -> int

    (* Returns the number of captures white has made, same as
       half the number of black stones removed from the board *)
    val whiteCaptures: (game) -> int

end

