(***********************************************
 * XML parser
 ***********************************************)

structure XML : XMLS = struct

  open Util

  datatype xml = ELEMENT of string * xml list | STRING of substring | TAG of substring

  val fixle : string -> string = String.map (fn c => if c = #"@" then #"<" else c)

  fun parse (s:string) : xml = let

    fun reduce (s:substring, stack:xml list) : xml list = let
      fun reduce' (stack:xml list, arg:xml list) =
        case stack of
          [] => raise Fail ("unmatched closing XML tag " ^ Substring.string s)
        | TAG x::ss => if Substring.compare(x,s) = EQUAL
            then ELEMENT (fixle (Substring.string x),arg) :: ss
            else raise Fail ("ill-nested XML tags " ^ Substring.string x ^ " and " ^ Substring.string s)
        | e::ss => reduce'(ss,e::arg)
    in
      reduce' (stack,[])
    end

    fun parse' (s:substring,stack:xml list) : xml =
      if Substring.isEmpty s then
        (case stack of
          [top as ELEMENT("kat",_)] => top
        | _ => raise Fail "could not parse XML")
      else let
        val first = Substring.first s
        val (token,rest) = case first of
          SOME #"<" => Substring.splitl (notc #">") (#2(valOf(Substring.getc s)))
        | _ => Substring.splitl (notc #"<") s
       in
         case first of
           SOME #"<" =>
             (case Substring.getc token of
               SOME (#"/",t) => (parse' (#2(valOf(Substring.getc rest)), reduce(t,stack))
                                 handle Option => raise Fail "invalid XML tag")
             | _ => parse' (#2(valOf(Substring.getc rest)), TAG token::stack))
           | _ => parse' (rest, STRING token::stack)
       end
  in
    parse' (Substring.all s,[])
  end

  fun getContent (expected:string, xml:xml) : xml list =
    case xml of
      ELEMENT (tag,content) =>
        if tag = expected then content
        else raise Fail ("unexpected xml tag: wanted " ^ expected ^ ", got " ^ tag)
    | _ => raise Fail ("unexpected xml tag: wanted " ^ expected)

  fun getString (expected:string, xml:xml) : string = let
    val content = case xml of
      ELEMENT (tag,content) =>
        if tag = expected then content
        else raise Fail ("unexpected xml tag: wanted " ^ expected ^ ", got " ^ tag)
      | _ => raise Fail ("unexpected xml element, wanted " ^ expected)
  in
    case content of
      [STRING s] => fixle (Substring.string s)
    | [] => ""
    | _ => raise Fail ("unexpected content of xml element " ^ expected ^", wanted string")
  end

end
