open Ast
open Ast_types
open Eval
open Environment
open Environment_types

type env = Environment.t
type env_types = Environment_types.t
type typedefs = typedef list

exception Parse_error of exn * int * int * string
                              
let parse (s : string) : Ast.toplevel =
  let lexbuf = Lexing.from_string s in
  try Parser.parse Lexer.token lexbuf
  with exn ->
    let curr = lexbuf.Lexing.lex_curr_p in
    let line = curr.Lexing.pos_lnum in
    let char = curr.Lexing.pos_cnum - curr.Lexing.pos_bol in
    let token = Lexing.lexeme lexbuf in
    raise (Parse_error (exn, line, char, token))

(* top-level bindings *)
let top = ref (Util.HashSet.make ())

(* read lines from the console, appending them to s,
 * until the user enters ;; at end of line *)
let read_console () : string =
  let rec read_lines (s : string) : string =
    let input = Util.trim (read_line ()) in
    if Util.ends_with ";;" input
    then s ^ (String.sub input 0 (String.length input - 2))
    else read_lines (s ^ input ^ "\n")
  in read_lines ""
  
let eval_expression (e : expr) ((o,oT,td) : env * env_types * typedefs) 
                    : env * env_types * typedefs =
  let (typeE, p, td2) = 
    if !Inference_types.type_inference 
    then Inference_types.getType e (oT, td)
    else (DummyType, Environment_types.empty, [ ]) in
  let (v, t) = Eval.eval (e, o) in
  if !Inference_types.type_inference
  then Printf.printf "- : %s = %s\n" (Ast_types.to_string typeE) (Printing.to_string (v, t))
  else print_endline (Printing.to_string (v, t));
  (* garbage collect *)
  (Environment.gc !top t, oT, td2)
  
let declare (verbose : bool) ((o,oT,td) : env * env_types * typedefs) (d : declaration) 
            : env * env_types * typedefs =
  (* a special treatment for typedefs *)
  match d with
    | Typedef t -> (o, oT, t :: (List.remove_assoc (fst t) td))
    | _ -> (
  let (f, args, e) = match d with
    | Def (f,a,e) | DefType (f,_,a,_,e) | Defrec (f,a,e) | DefrecType (f,_,a,_,e) -> (f,a,e)
    | Defcorec (s,f,a,e) | DefcorecType (s, f,_,a,_,e) -> (f, [ ], FunCorec(s,f,a,e))
    | Typedef _ -> assert false
  in
  let e = match args with
    | [] -> e
    | _ -> Fun (args, e) in
  let o = match d with
    | (Def _ |  DefType _) -> o
    | (Defrec _ | DefrecType _ | Defcorec _ | DefcorecType _ ) ->
        Environment.bind f Dummy o 
    | Typedef _ -> assert false in
  let (typeE, p, td2) =
    if !Inference_types.type_inference
    then Inference_types.getTypeDecl d (oT, td)
    else (Null, oT, td) in
    let (v, t) = eval (e, o) in
  let Schema (_, typ) as gen = Equations_types.generalize typeE p in
  Util.HashSet.add !top f;
  let env = Environment.bind f v t in
  if verbose then
    if !Inference_types.type_inference
    then Printf.printf "val %s : %s = %s\n" f 
           (Ast_types.to_string typ) (Printing.to_string_bv (v, env) [ f ])
    else Printf.printf "%s = %s\n" f (Printing.to_string (v, env));
  (env, Environment_types.bind f gen p, td2))

let eval_declarations (dl : declaration list) ((o,oT,td) : env * env_types * typedefs) 
    (verbose : bool) : env * env_types * typedefs =
  let (o,oT,td) = List.fold_left (declare verbose) (o,oT,td) dl in
  (* garbage collect *)
  (Environment.gc !top o, oT, td)

let eval_directive (d : directive) ((o,oT,td) : env * env_types * typedefs) 
    : env * env_types * typedefs =
  match d with
    | Quit -> raise Exit
    | Chdir s ->
       if not (Sys.file_exists s) then runtime "directory does not exist" else
       if not (Sys.is_directory s) then runtime "file is not a directory" else
       Sys.chdir s; (o,oT,td)
    | Pwd -> print_endline (Sys.getcwd ()); (o,oT,td)
    | Load s ->
       if not (Sys.file_exists s) then runtime "file does not exist" else
       if Sys.is_directory s then runtime "file is a directory" else
       (match parse (read_file s) with
        | Declarations d -> eval_declarations d (o,oT,td) true
        | Expression e -> eval_expression e (o,oT,td)
        | Directive d -> runtime "directives not allowed in file")
    | Ls ->
       let cwd = Sys.getcwd () in
       let files = Sys.readdir cwd in
       Array.iter print_endline files; (o,oT,td)
    | Env ->
       if !Inference_types.type_inference then
         let f x (Schema (_, typ)) = Printf.printf "%s : %s\n" x (Ast_types.to_string typ) in
         Environment_types.iter f oT
       else ();
       Environment.iter (fun x v -> Printf.printf "%s = %s\n" x 
                        (Printing.to_string (v, Environment.empty))) o; (o,oT,td)
    | Scope ->
       Eval.static_scope := not !static_scope;
       Printf.printf "%s scope\n" (if !static_scope then "static" else "dynamic"); (o,oT,td)
    | Type_inf ->
       Inference_types.type_inference := not !Inference_types.type_inference;
       Printf.printf "type inference %s\n" (if !Inference_types.type_inference then "on" else "off"); (o,oT,td)

(* read eval print loop *)
let rec repl ((o,oT,td) : env * env_types * typedefs) : unit =
  try
    print_string "> ";
    let cmd = parse (read_console()) in
    let ((o,oT,td) : env * env_types * typedefs) =
      match cmd with
      | Directive d -> eval_directive d (o,oT,td)
      | Declarations d -> eval_declarations d (o,oT,td) true
      | Expression e -> eval_expression e (o,oT,td) in
    repl (o,oT,td)
  with
    | Unbound s -> Printf.printf "Unbound variable %s\n" s; repl (o,oT,td)
    | Parse_error (_,l,c,_) -> Printf.printf "Parse error at line %d char %d\n" l c; 
                               repl (o,oT,td)
    | Lexer.LexError s -> Printf.printf "Lex Error: %s\n" s; repl (o,oT,td)
    | Ast.Runtime s -> Printf.printf "Runtime error: %s\n" s; repl (o,oT,td)
    | Equations_types.Type_error s -> Printf.printf "Type error: %s\n" s; repl (o,oT,td)
    | Util.Fatal s -> Printf.printf "Fatal error: %s\nSorry :(\n" s; repl (o,oT,td)
    | Exit -> print_endline "bye"
 
let run () =
  let natives =
    List.fold_left
      (fun env (id, impl, typ) -> Environment.bind id (Native (id, impl, typ)) env)
      Environment.empty Natives.natives in

  let nativesT = 
    List.fold_left
      (fun env (id, impl, typ) -> 
        let (t, _, _) = Inference_types.getType (Native (id, impl, typ)) (env, [ ]) in
        Environment_types.bind id (Equations_types.generalize t env) env)
      Environment_types.empty Natives.natives in
      
  List.iter (fun (x, _, _) -> Util.HashSet.add !top x) Natives.natives;

  (* load pervasives *)
  let pervasives =
    match parse (read_file "etc/pervasives.txt") with
    | Declarations dl -> eval_declarations dl (natives, nativesT,[ ]) false
    | _ -> raise (Util.Fatal "corrupt pervasives.txt file") in

  repl pervasives
  
