open Ast
open Environment

let precedence = function
	| Dummy -> failwith "unexpected dummy"
        | Unknown _ -> 9
	| Native _ -> 9
	| Var _ -> 9
	| Int _ -> 9
	| Float _ -> 9
	| Strg _ -> 9
	| Bool _ -> 9
	| Unit -> 9
	| List _ -> 9
	| ListMake _ -> 9
	| Not _ -> 8
	| App _ -> 7
        | Inj _ -> 7
	| Cons _ -> 6
	| Eq _ -> 5
	| Neq _ -> 5
	| Lt _ -> 5
	| Le _ -> 5
	| Gt _ -> 5
	| Ge _ -> 5
	| Plus _ -> 4
	| Minus _ -> 4
	| PlusF _ -> 4
	| MinusF _ -> 4
	| Concat _ -> 4
	| Or _ -> 4
	| Mul _ -> 3
	| Div _ -> 3
        | Mod _ -> 3
	| MulF _ -> 3
	| DivF _ -> 3
	| And _ -> 3
        | Tuple _ -> 3 (* should be 2.5 *)
	| Assg _ -> 2
	| Seq _ -> 1
        | Match _ -> 0
	| If _ -> 0
	| While _ -> 0
	| For _ -> 0
	| Fun _ -> 0
	| FunType _ -> 0
        | FunCorec _ -> 0
	| Let _ -> 0
	| LetType _ -> 0
	| Letrec _ -> 0
	| LetrecType _ -> 0
        | Letcorec _ | LetcorecType _ -> 0
	

type assoc = LEFT | RIGHT | PREFIX | SUFFIX | ASSOC | NONASSOC

let vars2ids l = 
  List.map (function Var x -> x | _ -> Ast.runtime "arguments have to be variables") l

let assoc = function
  | App _ -> LEFT
  | Assg _ -> RIGHT
  | Cons _ -> RIGHT
  | Seq _ -> ASSOC
  | If _ -> PREFIX
  | Fun _ -> PREFIX
  | Let _ -> PREFIX
  | Not _ -> PREFIX
  | _ -> NONASSOC

let rec pattern_to_string (p : pattern) : (string * id list) = match p with
(* second argument is the list of variables bound *)
  | PVar id -> id, [ id ]
  | PInt i -> string_of_int i, [ ]
  | PFloat f -> string_of_float f, [ ]
  | PStrg s -> s, [ ]
  | PBool b -> string_of_bool b, [ ]
  | PUnit -> "()", [ ]
  | PCons(p1, p2) -> 
      let s1, l1 = pattern_to_string p1 and s2, l2 = pattern_to_string p2 in
      Printf.sprintf "%s :: %s" s1 s2, l1 @ l2
  | PList(l) -> let s, l = List.split (List.map pattern_to_string l) in
                Printf.sprintf "[%s]" (String.concat "; " s),
                List.flatten l
  | PTuple(l) -> let s, l = List.split (List.map pattern_to_string l) in
                 Printf.sprintf "(%s)" (String.concat "," s),
                 List.flatten l
  | PInj(i, None) -> i,  [ ]
  | PInj(i, Some p) -> let s, l = pattern_to_string p in
        Printf.sprintf "%s %s" i s, l

let to_string_aux2 (e, env : state) (d : int option) (bv : id list): string =
  let rec to_string_aux (e : expr) d bv =
(* we only go to depth d to print, to avoid looping forever *)
(* bv represents the variables that are bounded and should not be looked up 
   in the environment *)
  (* parenthesize based on associativity and precedence *)
  let protect_left e e1 d bv: string =
    let (p, p1) = (precedence e, precedence e1) in
    let s = to_string_aux e1 d bv in
    if (p1 < p && assoc e1 <> SUFFIX) || (p = p1 && (assoc e1 = RIGHT || assoc e1 = NONASSOC)) 
    then Printf.sprintf "(%s)" s else s in
  let protect_right e e1 d bv: string =
    let (p, p1) = (precedence e, precedence e1) in
    let s = to_string_aux e1 d bv in
    if (p1 < p && assoc e1 <> PREFIX) || (p = p1 && (assoc e1 = LEFT || assoc e1 = NONASSOC))
    then Printf.sprintf "(%s)" s else s in
	let binop_print op e e1 e2 = Printf.sprintf "%s %s %s"
           (protect_left e e1 d bv) op (protect_right e e2 d bv) in
  let arglist x = String.concat " " (vars2ids x) in
  if d = Some(0) then " ... " else
  match e with
    | Var x -> if List.mem x bv || d = None then x else 
               if Environment.bound x env 
               then (to_string_aux (Environment.lookup x env)
                     (match d with None -> assert false | Some(d0) -> Some(d0-1)) bv)
               else x
    | Unknown s -> "Unk" ^ s
    | Int x -> string_of_int x
    | Float x -> string_of_float x
    | Strg x -> "\"" ^ String.escaped x ^ "\""
    | Bool b -> string_of_bool b
    | Fun (x, e1) ->
        Printf.sprintf "fun %s -> %s" (arglist x) (protect_left e e1 None ((vars2ids x) @ bv))
    | FunType(x, _, e1) -> to_string_aux (Fun(x,e1)) None ((vars2ids x) @ bv)
    | FunCorec (s, f, x, e1) ->
        Printf.sprintf "fun %s corec[%s] %s -> %s" f
          (to_string_aux s d bv) (arglist x) (protect_left e e1 None (f ::((vars2ids x) @ bv)))
    | If (e1, e2, e3) ->
        Printf.sprintf "if %s then %s else %s" (protect_left e e1 d bv)
           (protect_left e e2 d bv) (protect_left e e3 d bv)
    | While (e1, e2) ->
        Printf.sprintf "while %s do %s done" (to_string_aux e1 d bv) (to_string_aux e2 d bv)
    | For (x, e1, e2, e3) ->
        Printf.sprintf "for %s = %s to %s do %s done" x
          (to_string_aux e1 d bv) (to_string_aux e2 d (x :: bv)) (to_string_aux e3 d (x :: bv))
    | Let (x, [], e1, e2) ->
        Printf.sprintf "let %s = %s in %s" x (protect_left e e1 d bv) 
                                             (protect_left e e2 d (x :: bv))
    | Let (x, y, e1, e2) ->
		Printf.sprintf "let %s %s = %s in %s" x (arglist y)
                (protect_left e e1 d bv) (protect_left e e2 d (x :: ((vars2ids y) @ bv)))
    | LetType (x ,_,y,_,e1,e2) -> to_string_aux (Let(x,y,e1,e2)) d bv
    | Letrec (x, [], e1, e2) ->
  	   Printf.sprintf "let rec %s = %s in %s" x (protect_left e e1 d bv) 
                                             (protect_left e e2 d (x :: bv))
    | Letrec (x, y, e1, e2) ->
	   Printf.sprintf "let rec %s %s = %s in %s" x (arglist y) 
                (protect_left e e1 d bv) (protect_left e e2 d (x :: ((vars2ids y) @ bv)))
    | LetrecType (x ,_,y,_,e1,e2) -> to_string_aux (Letrec(x,y,e1,e2)) d bv
    | Letcorec (s, x, [], e1, e2) ->
  			Printf.sprintf "let corec[%s] %s = %s in %s" (to_string_aux s d bv) x
                           (protect_left e e1 d bv) 
                           (protect_left e e2 d (x :: bv))
    | Letcorec (s, x, y, e1, e2) ->
			  Printf.sprintf "let corec[%s] %s %s = %s in %s"
             (to_string_aux s d bv) x (arglist y) 
                (protect_left e e1 d bv) (protect_left e e2 d (x :: ((vars2ids y) @ bv)))
    | LetcorecType (s, x ,_,y,_,e1,e2) -> to_string_aux (Letcorec(s,x,y,e1,e2)) d bv
    | App (e1, e2) -> Printf.sprintf "%s %s" (protect_left e e1 d bv) (protect_right e e2 d bv)
    | Assg (x, e1) -> Printf.sprintf "%s := %s" x (protect_right e e1 d bv)
    | Seq (e1, e2) -> Printf.sprintf "%s; %s" (protect_left e e1 d bv) (protect_right e e2 d bv)
    | Match (e1, l) -> Printf.sprintf "match %s with %s" (protect_left e e1 d bv)
        (String.concat " | " 
          (List.map (fun (p, e2) -> let s, l = pattern_to_string p in
          s ^ " -> " ^ (protect_right e e2 d (l @ bv))) l))

    (* Lists *)
    | ListMake e1 -> Printf.sprintf "[%s]" (to_string_aux e1 d bv)
    | Cons (e1, e2) -> Printf.sprintf "%s :: %s" (protect_left e e1 d bv) (protect_right e e2 d bv)
    | List l -> Printf.sprintf "[%s]" (String.concat "; " 
        (List.map (fun e1 -> to_string_aux e1 d bv) l))

    (* Comparison operators *)
    | Eq (e1, e2) -> binop_print "=" e e1 e2
    | Neq (e1, e2) -> binop_print "!=" e e1 e2
    | Le (e1, e2) -> binop_print "<=" e e1 e2
    | Lt (e1, e2) -> binop_print "<" e e1 e2
    | Gt (e1, e2) -> binop_print ">" e e1 e2
    | Ge (e1, e2) -> binop_print ">=" e e1 e2

    (* Arithmetic operators on integers*)
    | Plus (e1, e2) -> binop_print "+" e e1 e2
    | Minus (e1, e2) -> binop_print "-" e e1 e2
    | Mul (e1, e2) -> binop_print "*" e e1 e2
    | Div (e1, e2) -> binop_print "/" e e1 e2
    | Mod (e1, e2) -> binop_print "mod" e e1 e2

    (* Arithmetic operators on floats *)
    | PlusF (e1, e2) -> binop_print "+." e e1 e2
    | MinusF (e1, e2) -> binop_print "-." e e1 e2
    | MulF (e1, e2) -> binop_print "*." e e1 e2
    | DivF (e1, e2) -> binop_print "/." e e1 e2

    (* String operators *)
    | Concat (e1, e2) -> binop_print "^" e e1 e2

    (* Boolean operators *)
    | Not e1 -> Printf.sprintf "not %s" (protect_left e e1 d bv)
    | And (e1,e2) -> binop_print "&&" e e1 e2
    | Or (e1,e2) ->  binop_print "||" e e1 e2
    | Tuple l -> String.concat ", " (List.map (fun e1 -> protect_right e e1 d bv) l)
    | Inj(s, None) -> s
    | Inj(s, Some(e2)) -> Printf.sprintf "%s %s" s (protect_right e e2 d bv)
    | Unit -> "()"
    | Dummy -> "<dummy>"
    | Native _ -> "<native>"
in to_string_aux e d bv

let to_string_bv s (bv : string list) = match (fst s) with
 | Fun _ | FunType _ | FunCorec _ -> to_string_aux2 s None bv
  (* to avoid replacing themselves in recursive functions *)
 | _ -> to_string_aux2 s (Some 20) [ ]
(* None means do not replace anything *)
let to_string s = to_string_bv s [ ]