%{
open Ast
%}

%token FUN LET REC IN LOAD
%token EQ NEQ LT LE GT GE
%token PLUS MINUS MUL DIV CONCAT PLUSF MINUSF MULF DIVF
%token APP
%token <int> INT
%token <float> FLOAT
%token <string> VAR
%token <string> STRING
%token IF THEN ELSE WHILE DO FOR TO DONE
%token TRUE FALSE
%token SEQ IMP ASSG
%token CONS
%token LPAREN RPAREN LBRACKET RBRACKET
%token NOT AND OR
%token EOF
%token LOAD CHDIR PWD LS ENV SCOPE TYPE_INF QUIT
%token <string> TYPE
%token TYPEDECL
%token <string> TYPEVAR

%nonassoc RPAREN RBRACKET /* lowest precedence - always reduce */
%left SEQ
%right ASSG
%right CONS
%left EQ NEQ LT LE GT GE
%left PLUS MINUS PLUSF MINUSF OR
%left MUL DIV MULF DIVF CONCAT AND
%left APP
%right IMP
%nonassoc NOT
%nonassoc INT FLOAT VAR TRUE FALSE STRING TYPE
%nonassoc LPAREN LBRACKET FUN LET IN ELSE IF WHILE FOR TYPEDECL TYPEVAR
/* highest precedence - always shift */

/* entry points */
%start parse
%type <Ast.toplevel> parse

%%

parse:
  | directive EOF { Directive $1 }
  | declaration_list EOF { Declarations $1 }
  | expr EOF { Expression $1 }
;

directive:
  | LOAD STRING  { Load $2 }
  | CHDIR STRING { Chdir $2 }
  | PWD { Pwd }
  | LS  { Ls }
  | ENV { Env }
  | SCOPE { Scope }
  | TYPE_INF { Type_inf }
  | QUIT  { Quit }
;  

declaration_list:
  | declaration declaration_list { $1 :: $2 }
  | declaration { [$1] }
;

declaration:
	| LET parse_var EQ expr %prec RPAREN
      { DefType ($2, Null, [], [], $4) }
	| LET parse_var TYPEDECL parse_type EQ expr %prec RPAREN
      { DefType ($2, $4, [], [], $6) }
	| LET LPAREN parse_var TYPEDECL parse_type RPAREN EQ expr %prec RPAREN
      { DefType ($3, $5, [], [], $8) }
	| LET parse_var typelist EQ expr %prec RPAREN
      { DefType($2, Null, fst (List.split $3), snd (List.split $3), $5) }
	| LET parse_var typelist TYPEDECL parse_type EQ expr %prec RPAREN
      { DefType($2, $5, fst (List.split $3), snd (List.split $3), $7) }
	| LET REC parse_var EQ expr %prec RPAREN
      { DefrecType ($3, Null, [], [], $5) }
	| LET REC parse_var TYPEDECL parse_type EQ expr %prec RPAREN
      { DefrecType ($3, $5, [], [], $7) }
	| LET REC LPAREN parse_var TYPEDECL parse_type RPAREN EQ expr %prec RPAREN
      { DefrecType ($4, $6, [], [], $9) }
	| LET REC parse_var typelist EQ expr %prec RPAREN
      { DefrecType($3, Null, fst (List.split $4), snd (List.split $4), $6) }
	| LET REC parse_var typelist TYPEDECL parse_type EQ expr %prec RPAREN
      { DefrecType($3, $6, fst (List.split $4), snd (List.split $4), $8) }
;

parse_var:
   | VAR { $1 }
   | LPAREN parse_var RPAREN { $2 }
;

param:
   | LPAREN VAR TYPEDECL parse_type RPAREN { (Var $2, $4) }
   | VAR { (Var $1, Null) }
   | LPAREN LPAREN RPAREN TYPEDECL parse_type RPAREN { (Unit, $5) }
   | LPAREN RPAREN { (Unit, Null) }
   | LPAREN param RPAREN { $2 }
;

paramlist:
   | param paramlist { $1 :: $2 }
   | param { [$1] }
;

func_and_params:
   | VAR TYPEDECL parse_type { [(Var $1, $3)] }
   | VAR { [(Var $1, Null)] }
   | VAR paramlist TYPEDECL parse_type { (Var $1, $4) :: $2 }
   | VAR paramlist { (Var $1, Null) :: $2 }
;

typelist:   
	| LPAREN parse_var TYPEDECL parse_type RPAREN { [(Var $2, $4)] }
	| LPAREN parse_var TYPEDECL parse_type RPAREN typelist { (Var $2, $4) :: $6}
	| parse_var { [(Var $1, Null)] }
	| parse_var typelist { (Var $1, Null) :: $2 }
	| LPAREN LPAREN RPAREN TYPEDECL parse_type RPAREN { [(Unit, $5)] }
	| LPAREN LPAREN RPAREN TYPEDECL parse_type RPAREN typelist { (Unit, $5) :: $7 }
	| LPAREN RPAREN { [(Unit, Null)] }
	| LPAREN RPAREN typelist { (Unit, Null) :: $3 }
;

parse_type : 
	| TYPE { match $1 with
	    | "int" -> Integer
	    | "string" -> String
	    | "bool" -> Boolean
	    | "unit" -> UnitType
	    | _ -> raise Parse_error }
	| TYPEVAR { VarType $1 }
	| parse_type TYPE { match $2 with
	    | "list" -> ListType $1
	    | _ -> raise Parse_error }
	| parse_type IMP parse_type { Arrow ($1, $3) }
	| LPAREN parse_type RPAREN { $2 }
;

arg:
  | VAR { Var $1 }
  | INT { Int $1 }
  | FLOAT { Float $1 }
  | STRING { Strg $1 }
  | TRUE { Bool true }
  | FALSE { Bool false }
  | LPAREN RPAREN { Unit }
  | LPAREN expr RPAREN { $2 }
  | LBRACKET RBRACKET { List [] }
  | LBRACKET expr RBRACKET { ListMake $2 }

expr:
  | FUN typelist IMP expr %prec RPAREN { FunType ( fst ( List.split $2), snd (List.split $2), $4) }
  | declaration IN expr %prec RPAREN {
    match $1 with
      | DefType (x, typeX, a, typeA, e) -> LetType (x, typeX, a, typeA, e, $3)
      | DefrecType (x, typeX, a, typeA, e) -> LetrecType (x, typeX, a, typeA, e, $3)
      | Def (x,a,e) -> raise Parse_error 
      | Defrec (x,a,e) -> raise Parse_error
    }
  | VAR ASSG expr { Assg ($1, $3) }
  | expr SEQ expr { Seq ($1, $3) }
  | IF expr THEN expr ELSE expr %prec RPAREN { If ($2, $4, $6) }
  | WHILE expr DO expr DONE { While ($2, $4) }
  | FOR VAR EQ expr TO expr DO expr DONE { For ($2, $4, $6, $8) }
  | expr EQ expr { Eq ($1, $3) }
  | expr NEQ expr { Neq ($1, $3) }
  | expr LT expr { Lt ($1, $3) }
  | expr LE expr { Le ($1, $3) }
  | expr GT expr { Gt ($1, $3) }
  | expr GE expr { Ge ($1, $3) }
  | expr PLUS expr { Plus ($1, $3) }
  | expr MINUS expr { Minus ($1, $3) }
  | expr MUL expr { Mul ($1, $3) }
  | expr DIV expr { Div ($1, $3) }
  | expr PLUSF expr { PlusF ($1, $3) }
  | expr MINUSF expr { MinusF ($1, $3) }
  | expr MULF expr { MulF ($1, $3) }
  | expr DIVF expr { DivF ($1, $3) }
  | expr CONCAT expr { Concat ($1, $3) }
  | MINUS expr { Minus (Int 0, $2) }
  | MINUSF expr { MinusF (Float 0., $2) }
  | NOT expr { Not ($2) }
  | expr AND expr { And ($1, $3) }
  | expr OR expr { Or ($1, $3) }
  | expr CONS expr { Cons ($1, $3) }
  | expr arg { App ($1, $2) }
  | arg { $1 }
;

%%