(* coalgebra structure for probabilities *)
open Coalg

type 'x term = H | T | Flip of float * 'x term * 'x term | Ind of 'x
type empty = unit
type coalg = empty term

let equal = (==)
    
(* collect all elements of the given finite coalgebra *)
let collect (c : coalg) : coalg list =
  let rec collect (c : coalg) (l : coalg list) : coalg list =
    if List.memq c l then l
    else let l = c :: l in
    match c with
      | H -> l
      | T -> l
      | Flip (p, c1, c2) -> collect c2 (collect c1 l)
      | Ind _ -> failwith "Cannot collect indeterminates"
  in collect c []

(* utilities *)
let rec string_of_term (f : 'x -> string) (t : 'x term) : string =
  match t with
  | H -> "H"
  | T -> "T"
  | Flip (p, t1, t2) -> Printf.sprintf "%.3f:(%s,%s)" p (string_of_term f t1) (string_of_term f t2)
  | Ind x -> f x

(* map - apply only to well-founded terms *)  
let rec map (f : 'x -> 'y) (t : 'x term) : 'y term =
  match t with
    | H -> H
    | T -> T
    | Flip (p, t1, t2) -> Flip (p, map f t1, map f t2)
    | Ind x -> Ind (f x)