(* coalgebra structure for free variables *)
open Coalg

type 'x term = Var of string | App of 'x term * 'x term | Lam of string * 'x term | Ind of 'x
type empty = unit
type coalg = empty term

let equal = (==)
    
(* collect all elements of the given finite coalgebra *)
let collect (c : coalg) : coalg list =
  let rec collect (c : coalg) (l : coalg list) : coalg list =
    if List.memq c l then l
    else let l = c :: l in
    match c with
      | Var x -> l
      | App (c1, c2) -> collect c2 (collect c1 l)
      | Lam (x, c1) -> collect c1 l
      | Ind _ -> failwith "Cannot collect indeterminates"
  in collect c []

(* utilities *)
let rec string_of_term (f : 'x -> string) (t : 'x term) : string =
  match t with
  | Var x -> x
  | App (t1, t2) -> Printf.sprintf "(%s %s)" (string_of_term f t1) (string_of_term f t2)
  | Lam (x, t1) -> Printf.sprintf "/%s.%s" x (string_of_term f t1)
  | Ind x -> f x

(* map - apply only to well-founded terms *)  
let rec map (f : 'x -> 'y) (t : 'x term) : 'y term =
  match t with
    | Var x -> Var x
    | App (t1, t2) -> App (map f t1, map f t2)
    | Lam (x, t1) -> Lam (x, map f t1)
    | Ind x -> Ind (f x)