(* coalgebra structure for automata *)
open Coalg

(* automaton over a 2-letter alphabet *)
type 'x term = State of bool * 'x term * 'x term | Ind of 'x
type empty = unit
type state = empty term
type coalg = Pair of state * state

(* We would like to convert a pair of automata to a *)
(* product automaton.  Thus we would like to write  *)
(* let rec convert (s : state) (t : state) : state = *)
(*   match s, t with                                             *)
(*     | State (b1,s1,t1), State (b2,s2,t2) ->                   *)
(*         State (b1 = b2, convert s1 t1, convert s2 t2)         *)

let equal (Pair (s,t) : coalg) (Pair (u,v) : coalg) = (s == u) && (t == v)

(* collect all states of an automaton *)
let collect_states (c : state) : state list =
  let rec collect (c : state) (l : state list) : state list =
    if List.memq c l then l
    else let l = c :: l in
    match c with
      | State (_,s,t) -> collect s (collect t l)
      | _ -> failwith "Indeterminates not allowed"
  in collect c []

(* collect all elements of the product automaton *)
(* let collect (c : coalg) : coalg list =                        *)
(*   let rec collect (c : coalg) (l : coalg list) : coalg list = *)
(*     if Util.memq equal c l then l                             *)
(*     else let l = c :: l in                                    *)
(*     match c with                                              *)
(*       | Pair (State (_,s1,t1), State (_,s2,t2)) ->            *)
(*           collect (Pair (s1,s2)) (collect (Pair (t1,t2)) l)   *)
(*       | _ -> failwith "Indeterminates not allowed"            *)
(*   in collect c []                                             *)

let collect (Pair (s,t) : coalg) : coalg list =
  let sl = collect_states s in
  let tl = collect_states t in
  let f x y = Pair (x,y) in
  let g x = List.map (f x) tl in
  List.concat (List.map g sl)

(* utilities *)
let rec string_of_term (f : 'x -> string) (t : 'x term) : string =
  match t with
  | State (b,u,v) ->
      Printf.sprintf "%b:[%s,%s]" b (string_of_term f u) (string_of_term f v)
  | Ind x -> f x
      
  (* match t with                                                                *)
  (* | State (b,u,v) ->                                                          *)
  (*     Printf.sprintf "%b:[%s,%s]" b (string_of_term f u) (string_of_term f v) *)
  (* | Ind x -> f x                                                              *)

(* map - apply only to well-founded terms *)
let rec map (f : 'x -> 'y) (t : 'x term) : 'y term =
  match t with
    | State (b, u, v) -> State (b, map f u, map f v)
    | Ind x -> Ind (f x)