package provide Wise 1.0

#
# WISE user interface support
#
# The following procedures support the WISE style user interface.
# There are 4 exported functions:
#
#  Wise_Checklist and Wise_RadioList display a list of check buttons
#      (or radio buttons) for the user to select.  Both these routines
#      display next/back/finish/cancel buttons.
#  Wise_Message displays a long message, such as a copyright notice,
#      along with a set of buttons
#  Wise_GetDirName gets a directory name from the user



# Center the window on the screen.

proc Wise_CenterWindow {w} {
    update idletasks
    set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
            - [winfo vrootx [winfo parent $w]]]
    set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
            - [winfo vrooty [winfo parent $w]]]
    wm geom $w +$x+$y
    wm deiconify $w
    wm transient $w .
    raise $w
}

# Create the logo image (once)

proc WiseCreateLogo {} {
    global TekiInfo WiseInfo
    if {[info commands logoImage] == ""} {
        set WiseInfo(photo) [image create photo logoImage]
        logoImage read [file join $TekiInfo(library) logo.gif]
    }
}


# Create and display the WISE wizard window, centered on the screen.
# Cache the window for better performance

proc WiseMakeWizard {{ted 0}} {
    global WiseInfo TekiInfo

    if {[info commands .wise] == ""} {
        toplevel .wise
        WiseCreateLogo
        label .wise.logo -image logoImage
        grid .wise.logo -row 0 -column 0 -rowspan 2 -sticky n
		wm resizable .wise 0 0

        label .wise.title -font $TekiInfo(varfont)
        grid .wise.title -column 1 -row 0 -columnspan 5 -sticky ew
        grid rowconfigure .wise 1 -weight 1

		if {!$ted} {
	        listbox .wise.list
	        grid .wise.list -row 1 -column 1 -columnspan 5 -sticky nesw
		}

        button .wise.back -text {< Back} -command "set WiseInfo(done) back"
        button .wise.next -text {Next >} -command "set WiseInfo(done) next"

		# For Ted display OK button instead of finish but return finish
		if {$ted} {
			button .wise.finish -text {OK} -command "set WiseInfo(done) finish"
		} else {
			button .wise.finish -text {Finish} -command "set WiseInfo(done) finish"
		}

        button .wise.cancel -text {Cancel} -command "set WiseInfo(done) cancel"
        button .wise.help -text {Help} -command "set WiseInfo(done) help"
        grid .wise.back -row 2 -column 1 -sticky nsew
        grid .wise.next -row 2 -column 2 -sticky nsew
        grid .wise.finish -row 2 -column 3 -sticky nsew
        grid .wise.cancel -row 2 -column 4 -padx 1
		grid .wise.help -row 2 -column 5 -sticky nsew
		Wise_CenterWindow .wise
    } else {
        wm deiconify .wise
    }
}

# Add the listbox for checkboxes and radio buttons

proc Wise_CreateWizList {} {
	listbox .wise.list
	grid .wise.list -row 1 -column 1 -columnspan 5 -sticky nesw
}

# Remove the listbox for checkboxes and radio buttons

proc Wise_DestroyWizList {} {
	destroy .wise.list
}

# Display a checklist for the user in the WISE window.  
#
#    title is the window title
#    items is list {label value on?}
# One checkbox is created for each item, with the label as specified
#
# The return value depends on which button (next, back, finish, or cancel)
# the user presses.  It is one of the following:
#    back
#    cancel
#    {next list-of-selected-items}
#    {finish list-of-selected-items}
# where list-of-selected-items is the value part from the "items" list

proc Wise_Checklist {title items} {
    global WiseInfo

    WiseMakeWizard
    wm title .wise $title
    .wise.title configure -text $title

    set count 0
    foreach i $items {
        set name [lindex $i 0]
        set on [lindex $i 2]
        set w .wise.list.b$count
        set WiseInfo($count) $on
        checkbutton $w -variable WiseInfo($count) -text $name -anchor w
        pack $w -side top -fill x
        # grid $w -row $count -sticky nsw
        incr count
    }

    set WiseInfo(done) 0
    vwait WiseInfo(done)
    set rv $WiseInfo(done)

    if {[lsearch "next finish" $rv] != -1} {
        set count 0
        foreach i $items {
            set value [lindex $i 1]
            if $WiseInfo($count) {
                lappend rv $value
            }
            incr count
        }
    }

    wm withdraw .wise
    set count 0
    foreach i $items {
        destroy .wise.list.b$count
        incr count
    }
    unset WiseInfo
    return $rv
}

# Display a radio-button list for the user in the WISE window.  
#
#    title is the window title
#    items is list {label value on?}
# One checkbox is created for each item, with the label as specified
#
# The return value depends on which button (next, back, finish, or cancel)
# the user presses.  It is one of the following:
#    back
#    cancel
#    {next selected-item}
#    {finish selected-item}
# where selected-item is the value part (from the "items" list) of the
# radio-button the user selected

proc Wise_Radiolist {title items {backDisabled 0}} {
    global WiseInfo

    WiseMakeWizard
    wm title .wise $title
    .wise.title configure -text $title

    set count 0
    set WiseInfo(radio) {}
    foreach i $items {
        set name [lindex $i 0]
        set value [lindex $i 1]
        set on [lindex $i 2]
        set w .wise.list.b$count
        if {$on} {
            set WiseInfo(radio) $value
        }
        radiobutton $w -variable WiseInfo(radio) -text $name -anchor w -value $value
        pack $w -side top -fill x
        incr count
    }

    set WiseInfo(done) 0
    if {$backDisabled} {
        .wise.back configure -state disabled
    }
    vwait WiseInfo(done)
    set rv $WiseInfo(done)

    wm withdraw .wise
    .wise.back configure -state normal

    set count 0
    foreach i $items {
        destroy .wise.list.b$count
        incr count
    }

    if {[lsearch "next finish" $rv] != -1} {
        set rv [concat $rv $WiseInfo(radio)]
    }
    unset WiseInfo
    return $rv
}

#
# Display a long message, such as a copyright notice, to the user
# in a textbox.
#
#    title is the window title
#    msg is the text of the message
#    items is a list of labels.  One button is created for each label.
#
# The return value is the index of the button pressed


#
# WiseGetDrives
# If the platform is windows,the function sets set the
# global variable Drives with the available drives
#

proc WiseGetDrives {} {
	global tcl_platform Drives

	if {[info exists Drives]} {
		return
	}

	set Drives ""
	set pdir [pwd]

	switch $tcl_platform(platform) {
		windows {
			foreach c {c d e f g h i j k l m n o p q r s t u v w x y z} {
				set drive "$c:/"

				if {[catch {cd $drive}] == 0} {
					set Drives [concat $Drives "\[-$c-\]"]
				}
			}
		}
	}

	cd $pdir
}

proc Wise_Message {title msg items {dispLogo 1}} {
    global WiseInfo

    if {[info commands .wisemsg] == ""} {
        toplevel .wisemsg
        WiseCreateLogo
        label .wisemsg.logo -image logoImage
        scrollbar .wisemsg.scroll -command ".wisemsg.text yview"
        text .wisemsg.text -relief sunken -bd 2 -yscrollcommand ".wisemsg.scroll set" -setgrid 1 -height 10 -width 70
        Wise_CenterWindow .wisemsg
		wm resizable .wisemsg 0 0
    }

	focus .wisemsg
    wm title .wisemsg $title
    set WiseInfo(doneMsg) -1
    set count 1
    foreach i $items {
        button .wisemsg.b$count -text $i -command "set WiseInfo(doneMsg) $count"
        grid .wisemsg.b$count -row 2 -column $count
        incr count
    }

    .wisemsg.text delete 1.0 end
    .wisemsg.text insert end $msg
    grid rowconfigure .wisemsg 0 -weight 1
    grid .wisemsg.text -column 1 -row 0 -columnspan [expr $count-1] -sticky nesw
    grid .wisemsg.scroll -column $count -row 0 -sticky ns

    if {$dispLogo} {
        grid .wisemsg.logo -row 0 -column 0 -sticky n
    } else {
        grid forget .wisemsg.logo
    }

    Wise_CenterWindow .wisemsg
    vwait WiseInfo(doneMsg)
    set rv [expr $WiseInfo(doneMsg)-1]
    wm withdraw .wisemsg

    set count 1
    foreach i $items {
        destroy .wisemsg.b$count
        incr count
    }
    unset WiseInfo(doneMsg)
    return $rv
}

proc WiseCreateDirDlg {{title "Select Directory"} {helpcmd ""}} {
	global FileDlgWin newdir

	toplevel .dlgWin
	set FileDlgWin .dlgWin
	wm title $FileDlgWin $title
	Wise_CenterWindow $FileDlgWin
 
    set w ".dlgWin.dirdlg"
    set bw ".dlgWin.dirdlg.btns"

	frame $w -width 100m -height 200m -bd 1
	frame $bw -width 100m -bd 1
	pack $w -side top -fill both -padx 2m -pady 2m
	pack $bw -side bottom -fill both -padx 2m -pady 2m
    scrollbar $w.scroll -command "$w.dirlist yview"
	listbox $w.dirlist  -relief sunken -height 15 -width 50 -selectmode single \
						 -yscrollcommand "$w.scroll set"
	button $bw.ok -command "DirDlgEventHandler ok" -text "  OK  "
	button $bw.cancel -command "DirDlgEventHandler cancel" -text Cancel
	entry $w.editdir -width 50 -relief sunken -textvariable newdir
	pack $w.editdir -side top -fill both
    pack $w.scroll -side right -fill y
	pack $w.dirlist -side left -fill both
	
	if {$helpcmd != ""} {
		button $bw.help -command "grab release $FileDlgWin;$helpcmd;grab set $FileDlgWin" -text Help
		pack $bw.ok $bw.cancel $bw.help -side left -padx 7m -pady 2m
	} else {
		pack $bw.cancel -side right -padx 7m -pady 2m
		pack $bw.ok -side left -padx 7m -pady 2m
	}

    bind $w.editdir <Return> {DirDlgEventHandler return}
    bind $w.dirlist <Double-1> {DirDlgEventHandler doubleclk}
	DirDlgEventHandler update
}

# Directory dialog box event handler

proc DirDlgEventHandler {option} {
	global FileDlgWin SDir Drives newdir tcl_platform

	set w $FileDlgWin
	set pDir [pwd]

	switch $option {
		return -
		update -
		doubleclk {
			set selid [$w.dirdlg.dirlist curselection]

			if {$option == "return"} {
				catch {cd $newdir}
			} elseif {$option == "doubleclk"} {
				if {$selid != ""} {
					set c [$w.dirdlg.dirlist get $selid]

					if [regexp {^\[-} $c] {
						set newdir "[string index $c 2]:/"
					} else {
						set newdir $c
					}

					if {$tcl_platform(platform) == "windows"} {
						set newdir [string tolower $newdir]
					}

					catch {cd $newdir}
					set newdir [pwd]
				}
			}

			$w.dirdlg.dirlist delete 0 end
			$w.dirdlg.dirlist insert end ".."

			set directories [glob -nocomplain */]

			if {[string compare $directories ""] != 0} {
				set directories [lsort -dictionary $directories]
			}

			if {[string compare $Drives ""] != 0} {
				set directories [concat $directories $Drives]
			}
			
			if {[string compare $directories ""] != 0} {
				foreach file $directories {
					$w.dirdlg.dirlist insert end $file
				}
			}
		}

		ok {
			if {$tcl_platform(platform) == "windows"} {
				set newdir [string tolower $newdir]
			}

			set SDir $newdir
			destroy $w
		}
		
		cancel {
			set SDir ""
			destroy $w
		}
	}
}

# Ask the user to enter a directory name.  Return either the
# directory name, or the empty string if they hit "cancel".

proc Wise_GetDirName {{defaultdir ""} {WinName "Select Directory"} {helpcmd ""}} {
	global tcl_platform FileDlgWin SDir newdir

	set OldDir [pwd]

	WiseGetDrives

	if {![catch {cd $defaultdir}]} {
		set SDir $defaultdir
		set newdir $defaultdir
	} else {
		set SDir $OldDir
		set newdir $SDir
	}

	WiseCreateDirDlg $WinName $helpcmd

	grab set $FileDlgWin
	tkwait window $FileDlgWin

	cd $OldDir
	return $SDir
}

# XXX: need to fix this...

proc Wise_OldGetDirName {default} {
    global WiseInfo
    catch {destroy .d}
    toplevel .d
    wm title .d "Select directory"

    set WiseInfo(done) 0
    set WiseInfo(value) $default
    label .d.label -text "Enter directory name"
    entry .d.entry -textvariable WiseInfo(value)
    button .d.ok -text OK -command "set WiseInfo(done) ok"
    button .d.browse -text Browse... -command {
        set WiseInfo(value) [file dirname [tk_getOpenFile -title "Select file in directory"]]
    }
    button .d.cancel -text Cancel -command "set WiseInfo(done) cancel"
    grid .d.label -row 0 -column 0 -sticky e
    grid .d.entry -row 0 -column 1 -columnspan 2 -sticky ew
    grid .d.ok -row 1 -column 0 
    grid .d.browse -row 1 -column 1 
    grid .d.cancel -row 1 -column 2
    grid columnconfigure .d 1 -weight 1
    Wise_CenterWindow .d
    vwait WiseInfo(done)
    if {$WiseInfo(done) == "ok"} {
        set rv $WiseInfo(value)
    } else {
        set rv {}
    }
    destroy .d
    unset WiseInfo
    return $rv
}
