#include "view.h"
#include "appl_intf.h"
#include "timer.h"

static UL MAX_LOCAL_IDLE = 10 ;
static UL MAX_REMOTE_IDLE = 30 ;
static double  DEFAULT_RTO = 0.100 ;

struct node {
  char *hostname ;
  SAin addr ;
  UL cnt ;
  double srtt, rttvar, rto ;
  bool first_sample ;
  struct node *next ;
} ;

typedef struct node Node ;

static Node *local, *remote ;

static struct state {
  double srtt, rttvar, rto ;
  bool first_sample ;
} s ;


static Node *create(char *hostname, SAin addr) {
  Node *p ;

  p = Malloc(sizeof(Node)) ;
  p->hostname = hostname ;
  p->addr = addr ;
  p->cnt = 0 ;
  p->rto = DEFAULT_RTO ;
  p->first_sample = true ;
  p->next = NULL ;
  return(p) ;
}


static void print(Node *p) {
  if (p == NULL) {
    printf("None") ;
    return ;
  }

  while (p->next != NULL) {
    printf("%s ", p->hostname) ;
    p = p->next ;
  }
  printf("%s", p->hostname) ;
  return ;
}


void view_print(void) {
  printf("(View[%lu, %lu]: ", nlocal, nremote) ;
  print(local) ;
  printf("; ") ;
  print(remote) ;
  printf(")\n") ;
  if (appl_recv_view != NULL) {
    /* wait for RTT measurement */
    timer_req("view", 5.0, (req_func)appl_recv_view, (Seqno)(nlocal + nremote)) ;
  }
  return ;
}


static void print_full(Node *head) {
  Node *p ;

  p = head ;
  while (p != NULL) {
    addr_print(p->hostname, p->addr) ;
    printf("srtt=%.3fms rttvar=%.3fms rto=%.3fms\n", p->srtt * 1000, p->rttvar * 1000, p->rto * 1000) ;
    p = p->next ;
  }
  return ;
}


void view_print_full(void) {
  printf("Local members: %lu\n", nlocal) ;
  print_full(local) ;
  printf("Remote members: %lu\n", nremote) ;
  print_full(remote) ;
  printf("End\n") ;
  return ;
}


void view_init(void) {
  if (param_exist("sender")) {
    am_sender = param_bool("sender") ;
  } else {
    am_sender = false ;
  }

  if (am_sender) {
    sender_region = found_sender = true ;
    sender = myaddr ;
    strcpy(sender_name, myname) ;
    s.first_sample = false ;
    s.srtt = s.rttvar = s.rto = 0 ;
    printf("I'm the sender\n") ;
  } else {
    s.first_sample = true ;
    sender_region = found_sender = false ;
    sender_name[0] = '\0' ;
  }

  MAX_LOCAL_IDLE = param_int("max_local_idle") ;
  MAX_REMOTE_IDLE = param_int("max_remote_idle") ;
  DEFAULT_RTO = param_double("default_rto") ;
  remote = NULL ;
  nremote = 0 ;
  local = create(myname, myaddr) ;
  /* "0" delay to myself */
  local->srtt = local->rttvar = local->rto = 0 ;
  nlocal = 1 ;
  max_local_try = 0 ;
  view_print() ;
  return ;
}


void view_incr_local(void) {
  Node *p ;

  p = local ;
  while (p != NULL) {
    /* do not increase my entry */
    if (!streq(p->hostname, myname)) p->cnt++ ;
    p = p->next ;
  }
  return ;
}


void view_incr_remote(void) {
  Node *p ;

  p = remote ;
  while (p != NULL) {
    p->cnt++ ;
    p = p->next ;
  }
  return ;
}


/* how many local requests should I send before concluding that the msg loss is regional? */
static double err_thresh = 0.001 ;

static void cal_max_local_try(void) {
  size_t ntries ;
  double got, wait, k ;
  double p, v ;

  if (nlocal <= 8) {
    max_local_try = nlocal ;
    return ;
  }

  /* assume one member has a copy of the message initially */
  got = 1.0 ;
  wait = nlocal - 1.0 ;
  ntries = 0 ;
  do {
    /* probability a member missing a msg will get a repair */
    p = got / nlocal ;
    k = p * wait ;
    got += k ;
    wait -= k ;
    v = (nlocal - 1 - wait) / (nlocal - 1) ;
    ntries++ ;
  } while (1.0 - v > err_thresh) ;
  max_local_try = ntries ;
  return ;
}


void view_check_local(void) {
  Node *p, *pre ;

  p = local ;
  pre = NULL ;
  while (p != NULL) {
    if (p->cnt > MAX_LOCAL_IDLE) {
      if (pre != NULL) {
	pre->next = p->next ;
      } else {
	local = p->next ;
      }
      nlocal-- ;
      free(p->hostname) ;
      free(p) ;
      cal_max_local_try() ;
      view_print() ;
      break ;
    }
    pre = p ;
    p = p->next ;
  }
  return ;
}


void view_check_remote(void) {
  Node *p, *pre ;

  p = remote ;
  pre = NULL ;
  while (p != NULL) {
    if (p->cnt > MAX_REMOTE_IDLE) {
      if (pre != NULL) {
	pre->next = p->next ;
      } else {
	remote = p->next ;
      }
      nremote-- ;
      free(p->hostname) ;
      free(p) ;
      view_print() ;
      break ;
    }
    pre = p ;
    p = p->next ;
  }
  return ;
}


bool local_member(SAin addr) {
  Node *p ;

  p = local ;
  while (p != NULL) {
    if (addr_ip_eq(p->addr, addr)) {
      return(true) ;
    }
    p = p->next ;
  }
  return(false) ;
}


static bool clear(Node *head, char *hostname) {
  Node *p ;

  p = head ;
  while (p != NULL) {
    if (!strcmp(hostname, p->hostname)) {
      p->cnt = 0 ;
      return(true) ;
    }
    p = p->next ;
  }
  return(false) ;
}


void update_local_view(char *hostname, SAin addr) {
  Node *p ;

  if (!clear(local, hostname)) {
    p = create(hostname, addr) ;
    p->next = local ;
    local = p ;
    nlocal++ ;
    view_print() ;
    cal_max_local_try() ;
  }
  return ;
}


void add_sender(char *hostname, SAin addr) {
  assert (remote == NULL) ;
  remote = create(hostname, addr) ;
  remote->first_sample = s.first_sample ;
  remote->srtt = s.srtt ;
  remote->rttvar = s.rttvar ;
  remote->rto = s.rto ;
  nremote = 1 ;
  view_print() ;
  return ;
}


/* remove members which are too far away */
static bool remove_it(double thresh) {
  Node *p, *pre ;

  p = remote ;
  pre = NULL ;
  while (p != NULL) {
    if (!p->first_sample) {
      if (p->srtt > thresh) {
	if (pre == NULL) {
	  remote = p->next ;
	} else {
	  pre->next = p->next ;
	}
	nremote-- ;
	free(p->hostname) ;
	free(p) ;
	view_print() ;
	return(true) ;
      }
    }
    pre = p ;
    p = p->next ;
  }
  return(false) ;
}


static double T = 0.01 ;

static void check_remote(void) {
  Node *p ;
  double min_rtt, thresh ;

  min_rtt = 1.0 ;
  p = remote ;
  while (p != NULL) {
    if (!p->first_sample && p->srtt < min_rtt) {
      min_rtt = p->srtt ;
    }
    p = p->next ;
  }
  
  thresh = min_rtt + T ;
  while (remove_it(thresh)) ;
  return ;
}


void update_remote_view(char *hostname, SAin addr) {
  Node *p ;

  if (!clear(remote, hostname)) {
    p = create(hostname, addr) ;
    p->next = remote ;
    remote = p ;
    nremote++ ;
    check_remote() ;
    view_print() ;
  }
  return ;
}


char *choose_local(SAin *addr, double *rto) {
  size_t idx ;
  Node *p ;

  assert (nlocal > 1) ;
  idx = rand_int(nlocal-1) ;
  p = local ;
  while (idx > 0) {
    if (!addr_eq(p->addr, myaddr)) idx-- ;
    p = p->next ;
  }
  if (addr != NULL) (*addr) = p->addr ;
  if (rto != NULL)  (*rto) = p->rto ;
  return(p->hostname) ;
}


char *choose_remote(SAin *addr, double *rto) {
  size_t idx ;
  Node *p ;

  assert (nremote > 0) ;
  idx = rand_int(nremote) ;
  p = remote ;
  while (idx > 0) {
    idx-- ;
    p = p->next ;
  }
  if (addr != NULL) (*addr) = p->addr ;
  if (rto != NULL)  (*rto) = p->rto ;
  return(p->hostname) ;
}


char *choose_any(SAin *addr, double *rto) {
  size_t nmembers, idx ;
  
  nmembers = nlocal + nremote ;
  assert (nmembers > 1) ;
  idx = rand_int(nmembers-1) ;
  if (idx < nlocal-1) {
    return(choose_local(addr, rto)) ;
  }
  return(choose_remote(addr, rto)) ;
}


/* 
 * TCP-like RTO estimation
 * SRTT: smoothed estimate of RTT
 * RTTVAR: smoothed estimate of the variation of RTT
 * RTO: retransmission timer
 * alpha: weighting for SRTT
 * beta: weighting for RTTVAR
 * k: weighting for RTO
 */
static double alpha = 0.125 ;
static double beta = 0.25 ;
static double k = 2.0 ;
static double lower_bnd = 0.001 ;
static double upper_bnd = 1.0 ;

static void set_rtt(Node *head, char *hostname, double rtt) {
  Node *p ;
  double var ;

  /* ignore extraordinary values */
  if (rtt < lower_bnd || rtt > upper_bnd) return ;

  p = head ;
  while (p != NULL) {
    if (!strcmp(hostname, p->hostname)) {
      if (p->first_sample) {
	p->first_sample = false ;
	p->srtt = rtt ;
	p->rttvar = 0.5 * p->srtt ;
	p->rto = p->srtt + k * p->rttvar ;
      } else {
	var = fabs(rtt - p->srtt) ;
	p->srtt = (1 - alpha) * p->srtt + alpha * rtt ;
	p->rttvar = (1 - beta) * p->rttvar + beta * var ;
	p->rto = p->srtt + k * p->rttvar ;
      }
      return ;
    }
    p = p->next ;
  }
  return ;
}


bool get_remote_rtt(char *hostname, double *rtt) {
  Node *p ;

  p = remote ;
  while (p != NULL) {
    if (!strcmp(hostname, p->hostname)) {
      if (p->first_sample) {
	return(false) ;
      }
      (*rtt) = p->rto ;
      return(true) ;
    }
    p = p->next ;
  }
  return(false) ;
}


static void update_sender_rtt(double rtt) {
  double var ;

  /* ignore extraordinary values */
  if (rtt < lower_bnd || rtt > upper_bnd) return ;

  if (s.first_sample) {
    s.first_sample = false ;
    s.srtt = rtt ;
    s.rttvar = 0.5 * s.srtt ;
    s.rto = s.srtt + k * s.rttvar ;
  } else {
    var = fabs(rtt - s.srtt) ;
    s.srtt = (1 - alpha) * s.srtt + alpha * rtt ;
    s.rttvar = (1 - beta) * s.rttvar + beta * var ;
    s.rto = s.srtt + k * s.rttvar ;
  }
  return ;
}


bool get_sender_rtt(double *rtt) {
  if (s.first_sample) {
    /* no measurement is available */
    return(false) ;
  }
  (*rtt) = s.rto ;
  return(true) ;
}


void set_local_rtt(char *hostname, double rtt) {
  if (streq(hostname, sender_name)) {
    update_sender_rtt(rtt) ;
  }
  set_rtt(local, hostname, rtt) ;
  return ;
}


double get_max_rtt(void) {
  double max_rto ;
  Node *p ;

  max_rto = 0.0 ;
  p = local ;
  while (p != NULL) {
    if (p->rto > max_rto) max_rto = p->rto ;
    p = p->next ;
  }
  return(max_rto) ;
}


void set_remote_rtt(char *hostname, double rtt) {
  if (streq(hostname, sender_name)) {
    update_sender_rtt(rtt) ;
  }
  set_rtt(remote, hostname, rtt) ;
  return ;
}


SAin get_remote_addr(char *name) {
  Node *p ;

  p = remote ;
  while (p != NULL) {
    if (!strcmp(name, p->hostname)) {
      return(p->addr) ;
    }
    p = p->next ;
  }
  printf("Error[view]: entry for %s not found.\n", name) ;
  exit(1) ;
}


char *get_remote_name(SAin addr) {
  Node *p ;

  p = remote ;
  while (p != NULL) {
    if (addr_ip_eq(addr, p->addr)) {
      return(p->hostname) ;
    }
    p = p->next ;
  }
  return(NULL) ;
}

