#include "timer.h"

#define TIMER_SWEEP     0
#define TIMER_REQ       1
#define TIMER_MSG       2

/* overflow of "id" field is OK as long as each "id" is unique */
struct node {
  Type type ;
  char *dbg ;
  TV t ;
  sweep_func sf ;
  req_func rf ;
  msg_func mf ;
  Seqno seqno ;
  char *msg ;
  size_t size ;
  timer_id id ;
  struct node *next ;
} ;

typedef struct node Node ;

static Node *head = NULL ;

/* "0" is invalid timer_id */
static timer_id next_id = 1 ;


void timer_print(void) {
  TV now ;
  Node *p ;

  now = Gettimeofday() ;
  printf("Timer list: \n") ;
  p = head ;
  while (p != NULL) {
    printf("  id: %lu ", p->id) ;
    time_print(p->dbg, time_sub(p->t, now)) ;
    now = p->t ;
    p = p->next ;
  }
  printf("END\n") ;
  return ;
}


static void insert(Node *np) {
  Node *p, *pre ;

  if (head == NULL) {
    head = np ;
    return ;
  }

  p = head ;
  pre = NULL ;
  while (p != NULL) {
    if (time_gt(p->t, np->t)) break ;
    pre = p ;
    p = p->next ;
  }

  np->next = p ;
  if (pre == NULL) head = np ; 
  else pre->next = np ;

  return ;
}


void timer_sweep(char *dbg, double t, sweep_func sf) {
  TV now ;
  Node *p ;

  now = Gettimeofday() ;
  p = Malloc(sizeof(Node)) ;
  p->type = TIMER_SWEEP ;
  p->dbg = dbg ;
  p->t = time_add(now, time_of_double(t)) ;
  p->sf = sf ;
  p->id = next_id++ ;
  p->next = NULL ;
  insert(p) ;
  return ;
}


timer_id timer_req(char *dbg, double t, req_func rf, Seqno seqno) {
  TV now ;
  Node *p ;

  now = Gettimeofday() ;
  p = Malloc(sizeof(Node)) ;
  p->type = TIMER_REQ ;
  p->dbg = dbg ;
  p->t = time_add(now, time_of_double(t)) ;
  p->rf = rf ;
  p->seqno = seqno ;
  p->id = next_id++ ;
  p->next = NULL ;
  insert(p) ;
  return(p->id) ;
}


timer_id timer_msg(char *dbg, double t, msg_func mf, char *msg, size_t size) {
  TV now ;
  Node *p ;

  now = Gettimeofday() ;
  p = Malloc(sizeof(Node)) ;
  p->type = TIMER_MSG ;
  p->dbg = dbg ;
  p->t = time_add(now, time_of_double(t)) ;
  p->mf = mf ;
  p->msg = msg ;
  p->size = size ;
  p->id = next_id++ ;
  p->next = NULL ;
  insert(p) ;
  return(p->id) ;
}


void timer_cancel(timer_id id) {
  Node *p, *pre ;

  p = head ;
  pre = NULL ;
  while (p != NULL) {
    if (p->id == id) {
      if (pre != NULL) {
	pre->next = p->next ;
      } else {
	head = p->next ;
      }
      free(p) ;
      return ;
    }
    pre = p ;
    p = p->next ;
  }
  return ;
}


TV timer_run(void) {
  TV now ;
  Node *p ;

  now = Gettimeofday() ;
  while (head != NULL && time_ge(now, head->t)) {
    p = head ;
    head = head->next ;
    switch (p->type) {
      case TIMER_SWEEP:
	p->sf() ;
	break ;

      case TIMER_REQ:
	p->rf(p->seqno) ;
	break ;

      case TIMER_MSG:
	p->mf(p->msg, p->size) ;
	break ;

      default:
	printf("Error[timer]: unknown func type\n") ;
	exit(1) ;
    }

    free(p) ;
  }

  if (head != NULL) {
    return(time_sub(head->t, now)) ;
  }

  return(time_invalid()) ;
}
