#include "sess.h"

#define LOCAL_SESS       21
#define GLOBAL_SESS      22
#define LOCAL_QUERY      23
#define LOCAL_RESPONSE   24
#define REMOTE_QUERY     25
#define REMOTE_RESPONSE  26
#define LOCAL_CAST       16

#define lambda   4

/* 
 * MAX_LOCAL: max interval between any two local requests
 * MAX_REMOTE: max interval between any two remote requests
 */
static double MAX_LOCAL = 1.0 ;
static double MAX_REMOTE = 1.0 ;
static double LOCAL_SESS_INTERVAL = 1.0 ;
static double GLOBAL_SESS_INTERVAL = 1.0 ;

static bool DEBUG = 0 ;


struct hdr {
  Type type ;
  char *hostname ;
  SAin addr, target ;
  TV ts ;
  Seqno hi ;
  bool am_sender, valid ;
  double rtt ;
} ;

typedef struct hdr Hdr ;


static void hdr_ms(char *buf, size_t *offset, Hdr hdr) {
  type_ms(buf, offset, SESS) ;
  type_ms(buf, offset, hdr.type) ;
  str_ms(buf, offset, hdr.hostname) ;
  addr_ms(buf, offset, hdr.addr) ;
  tv_ms(buf, offset, hdr.ts) ;
  seqno_ms(buf, offset, hdr.hi) ;
  bool_ms(buf, offset, hdr.am_sender) ;
  bool_ms(buf, offset, hdr.valid) ;
  tv_ms(buf, offset, time_of_double(hdr.rtt)) ;
  return ;
}


static Hdr hdr_um(char *buf, size_t *offset) {
  Hdr hdr ;

  hdr.type = type_um(buf, offset) ;
  hdr.hostname = str_um(buf, offset) ;
  hdr.addr = addr_um(buf, offset) ;
  hdr.ts = tv_um(buf, offset) ;
  hdr.hi = seqno_um(buf, offset) ;
  hdr.am_sender = bool_um(buf, offset) ;
  hdr.valid = bool_um(buf, offset) ;
  hdr.rtt = time_to_double(tv_um(buf, offset)) ;
  return(hdr) ;
}


static struct state {
  /* accounting */
  UL local_sess, global_sess ;
  UL local_query, remote_query ;
} s ;


static void print_account(void) {
  printf("SESS: local_sess=%lu global_sess=%lu\n", s.local_sess, s.global_sess) ;
  printf("SESS: local_query=%lu remote_query=%lu\n", s.local_query, s.remote_query) ;
  return ;
}


void sess_final(void) {
  print_account() ;
  return ;
}


static void send_sess(Type type) {
  Hdr hdr ;
  char buf[1024] ;
  size_t offset ;

  hdr.type = type ;
  hdr.hostname = myname ;
  hdr.addr = myaddr ;
  hdr.ts = Gettimeofday() ;
  hdr.hi = iq->hi ;
  hdr.am_sender = am_sender ;
  hdr.valid = get_sender_rtt(&hdr.rtt) ;

  offset = 0 ;
  hdr_ms(buf, &offset, hdr) ;
  Sendto(cast_xmit_sock, buf, offset, &grpaddr) ;
  return ;
}


static void send_msg(Type type, TV ts, SAin dst) {
  Hdr hdr ;
  char buf[1024] ;
  size_t offset ;

  hdr.type = type ;
  hdr.hostname = myname ;
  hdr.addr = myaddr ;
  hdr.ts = ts ;

  offset = 0 ;
  hdr_ms(buf, &offset, hdr) ;
  dst.sin_port = pt2pt_port ;
  Sendto(pt2pt_xmit_sock, buf, offset, &dst) ;
  return ;
}


static void send_local_cast(char *hostname, double rtt) {
  Hdr hdr ;
  char buf[1024] ;
  size_t offset ;

  hdr.type = LOCAL_CAST ;
  hdr.hostname = myname ;
  hdr.addr = myaddr ;
  hdr.ts = Gettimeofday() ;
  hdr.hi = iq->hi ;
  hdr.am_sender = am_sender ;
  hdr.valid = true ;
  hdr.rtt = rtt ;
  hdr.target = get_remote_addr(hostname) ;

  offset = 0 ;
  hdr_ms(buf, &offset, hdr) ;
  Sendto(cast_xmit_sock, buf, offset, &grpaddr) ;
  return ;
}


static void send_local_sess(void) {
  timer_sweep("SESS", LOCAL_SESS_INTERVAL, send_local_sess) ;
  view_incr_local() ;
  view_check_local() ;
  send_sess(LOCAL_SESS) ;
  s.local_sess++ ;
  return ;
}


static void send_global_sess(void) {
  timer_sweep("SESS", GLOBAL_SESS_INTERVAL, send_global_sess) ;
  view_incr_remote() ;
  view_check_remote() ;
  if (rand_int(nlocal) < lambda) {
    send_sess(GLOBAL_SESS) ;
    s.global_sess++ ;
  }
  return ;
}


static void send_local_query(void) {
  SAin dst ;

  timer_sweep("SESS", MAX_LOCAL, send_local_query) ;
  if (nlocal > 1) {
    choose_local(&dst, NULL) ;
    send_msg(LOCAL_QUERY, Gettimeofday(), dst) ;
    s.local_query++ ;
  }
  return ;
}


static void send_remote_query(void) {
  SAin dst ;

  timer_sweep("SESS", MAX_REMOTE, send_remote_query) ;
  if (nremote > 0) {
    if (rand_int(nlocal) < lambda) {
      choose_remote(&dst, NULL) ;
      send_msg(REMOTE_QUERY, Gettimeofday(), dst) ;
      s.remote_query++ ;
    }
  }
  return ;
}


void sess_init(void) {
  DEBUG = param_bool("sess_dbg") ;
  timer_sweep("SESS", LOCAL_SESS_INTERVAL, send_local_sess) ;
  timer_sweep("SESS", GLOBAL_SESS_INTERVAL, send_global_sess) ;
  timer_sweep("SESS", MAX_LOCAL, send_local_query) ;
  timer_sweep("SESS", MAX_REMOTE, send_remote_query) ;
  s.local_sess = s.global_sess = 0 ;
  s.local_query = s.remote_query = 0 ;
  return ;
}


static void recv_local_sess(char *hostname, SAin addr, Seqno hi, bool is_sender) {
  if (DEBUG) {
    printf("recv local sess from %s.\n", hostname) ;
  }


  if (is_sender) {
    if (!found_sender) {
      sender_region = found_sender = true ;
      sender = addr ;
      strcpy(sender_name, hostname) ;
    } else if (!addr_eq(addr, sender)) {
      fprintf(stderr, "Error: mult-sender not supported\n") ;
      exit(1) ;
    }
  }

  if (hi > iq->hi) {
    nack(hi) ;
    iq->hi = hi ;
  }
  update_local_view(hostname, addr) ;
  return ;
}


static void recv_global_sess(char *hostname, SAin addr, Seqno hi, bool is_sender, bool valid, double hrtt) {
  double rtt ;

  if (DEBUG) {
    printf("recv global sess from %s.\n", hostname) ;
  }

  if (local_member(addr)) {
    /* ignore global session msgs from local members */
    return ;
  }

  if (is_sender) {
    if (!found_sender) {
      found_sender = true ;
      sender = addr ;
      strcpy(sender_name, hostname) ;
      add_sender(hostname, addr) ;
    } else if (!addr_eq(addr, sender)) {
      fprintf(stderr, "Error: mult-sender not supported\n") ;
      exit(1) ;
    }
  }

  /* receivers in the sender's region have no parent region */
  if (!found_sender || sender_region) return ;

  if (hi > iq->hi) {
    nack(hi) ;
    iq->hi = hi ;
  }

  if (!valid) {
    return ;
  }

  if (!get_sender_rtt(&rtt)) {
    return ;
  }

  if (hrtt < rtt) {
    update_remote_view(hostname, addr) ;
  }

  return ;
}


void sess_recv(char *buf, size_t offset) {
  Hdr hdr ;
  double rtt ;
  char *name ;

  hdr = hdr_um(buf, &offset) ;
  switch (hdr.type) {
    case LOCAL_SESS:
      recv_local_sess(hdr.hostname, hdr.addr, hdr.hi, hdr.am_sender) ;
      break ;

    case GLOBAL_SESS:
      recv_global_sess(hdr.hostname, hdr.addr, hdr.hi, hdr.am_sender, hdr.valid, hdr.rtt) ;
      break ;

    case LOCAL_QUERY:
      if (DEBUG) {
	printf("SESS: recv local query from %s.\n", hdr.hostname) ;
      }
      send_msg(LOCAL_RESPONSE, hdr.ts, hdr.addr) ;
      break ;

    case REMOTE_QUERY:
      if (DEBUG) {
	printf("SESS: recv global query from %s.\n", hdr.hostname) ;
      }
      send_msg(REMOTE_RESPONSE, hdr.ts, hdr.addr) ;
      break ;

    case LOCAL_RESPONSE:
      rtt = time_diff(Gettimeofday(), hdr.ts) ;
      if (DEBUG) printf("Local RTT to %s: %f\n", hdr.hostname, 1000 * rtt) ;
      set_local_rtt(hdr.hostname, rtt) ;
      break ;

    case REMOTE_RESPONSE:
      rtt = time_diff(Gettimeofday(), hdr.ts) ;
      if (DEBUG) printf("Remote RTT to %s: %f\n", hdr.hostname, 1000 * rtt) ;
      set_remote_rtt(hdr.hostname, rtt) ;
      send_local_cast(hdr.hostname, rtt) ;
      break ;

    case LOCAL_CAST:
      if (DEBUG) printf("recv local cast from %s\n", hdr.hostname) ;
      name = get_remote_name(hdr.target) ;
      if (name != NULL) {
	set_remote_rtt(name, hdr.rtt) ;
      }
      break ;

    default:
      printf("Error[SESS]: Unknown msg type\n") ;
      exit(1) ;
  }
  return ;
}
