/* Author: Zhen Xiao */

#include "proto.h"

#define DATA            11
#define LOCAL_REQUEST   12
#define LOCAL_REPAIR    13
#define REMOTE_REQUEST  14
#define REMOTE_REPAIR   15
#define LOCAL_CAST      16
#define SEARCH_REQUEST  17
#define SEARCH_DONE     18

/*
 * lambda: expected number of remote requests.
 * C: expected number of long-term bufferers.
 */
#define lambda         4
#define C             10

#define INIT_IQ_SIZE  1024

static int DEBUG = 0 ;


struct hdr {
  Type type ;
  char *name ;
  SAin addr, target ;
  Seqno seqno ;
  TV ts ;
  double delta, rtt ;
} ;

typedef struct hdr Hdr ;


static void hdr_ms(char *buf, size_t *offset, Hdr hdr) {
  type_ms(buf, offset, RRMP) ;
  type_ms(buf, offset, hdr.type) ;
  str_ms(buf, offset, hdr.name) ;
  addr_ms(buf, offset, hdr.addr) ;
  addr_ms(buf, offset, hdr.target) ;
  seqno_ms(buf, offset, hdr.seqno) ;
  tv_ms(buf, offset, hdr.ts) ;
  tv_ms(buf, offset, time_of_double(hdr.delta)) ;
  tv_ms(buf, offset, time_of_double(hdr.rtt)) ;
  return ;
}


static Hdr hdr_um(char *buf, size_t *offset) {
  Hdr hdr ;

  hdr.type = type_um(buf, offset) ;
  hdr.name = str_um(buf, offset) ;
  hdr.addr = addr_um(buf, offset) ;
  hdr.target = addr_um(buf, offset) ;
  hdr.seqno = seqno_um(buf, offset) ;
  hdr.ts = tv_um(buf, offset) ;
  hdr.delta = time_to_double(tv_um(buf, offset)) ;
  hdr.rtt = time_to_double(tv_um(buf, offset)) ;
  return(hdr) ;
}


static struct state {
  double idle_thresh ;
  double hbt_rate ;

  bool test_hier ;
  IntLst requests ;
  IntLst repairs ;
  size_t npoints ;

  IntLst delay ;
  bool test_incr ;
  bool test_delay ;

  /* accounting */
  UL bad_ret, recv_ret ;
  UL local_request, remote_request, search_request ;
  UL local_repair, remote_repair ;
  UL fast_read, slow_read ;
  UL drop_msg ;
} s ;


static void print_account(void) {
  printf("RRMP: recv_ret=%lu  bad_ret=%lu %.1f%%\n", s.recv_ret, s.bad_ret, ratio(s.bad_ret, s.recv_ret)) ;
  printf("RRMP: local_request=%lu remote_request=%lu\n", s.local_request, s.remote_request) ;
  printf("RRMP: local_repair=%lu remote_repair=%lu\n", s.local_repair, s.remote_repair) ;
  printf("RRMP: fast_read=%lu slow_read=%lu\n", s.fast_read, s.slow_read) ;
  printf("RRMP: drop_msg=%lu\n", s.drop_msg) ;
  return ;
}


static void check_short_buf(void) {
  TV now, thresh ;
  Seqno seqno ;
  Item p ;

  now = Gettimeofday() ;
  thresh = time_sub(now, time_of_double(s.idle_thresh)) ;
  for (seqno = iq->low; seqno < iq->read; seqno++) {
    p = iq_get_item(iq, seqno) ;
    assert (p != NULL) ;
    switch (p->type) {
      case IQ_DATA:
	/* Garbage collect only continously */
	if (time_gt(p->last, thresh)) {
	  iq_set_low(iq, seqno) ;
	  return ;
	}
	if (rand_int(nlocal) < C) {
	  longbuf_add(p->msg, p->size, seqno, p->last) ;
	}
	break ;

      case IQ_LOST:
	/* skip the entry */
	break ;

      case IQ_PENDING:
	fprintf(stderr, "Error(check_short_buf): unexpected iq type\n") ;
	exit(1) ;

      default:
	fprintf(stderr, "report_lost: unknown item type\n") ;
	exit(1) ;
    }
  }
}


static void recv_hbt(void) {
  timer_sweep("RRMP", s.hbt_rate, recv_hbt) ;
  check_short_buf() ;
  longbuf_check() ;
  /* printf("RRMP: %lu msgs in short_buf %lu msgs in long_buf\n", iq->nmsgs, longbuf_nmsgs()) ; */
  return ;
}


void rrmp_init(void) {
  iq = iq_create(INIT_IQ_SIZE) ;
  DEBUG = param_bool("rrmp_dbg") ;
  s.idle_thresh = param_double("idle_thresh") ;
  s.hbt_rate = 1.0 ;
  timer_sweep("RRMP", s.hbt_rate, recv_hbt) ;
  s.bad_ret = s.recv_ret = 0 ;
  s.fast_read = s.slow_read = 0 ;
  s.local_request = s.remote_request = s.search_request = 0 ;
  s.local_repair = s.remote_repair = 0 ;
  s.drop_msg = 0 ;
  s.requests = intlst_init() ;
  s.repairs = intlst_init() ;
  s.delay = intlst_init() ;
  s.npoints = param_int("npoints") ;
  s.test_hier = param_bool("test_hier") ;
  s.test_incr = param_bool("test_incr") ;
  s.test_delay = param_bool("test_delay") ;
  return ;
}


void rrmp_final(void) {
  char fname[1024] ;

  print_account() ;
  if (s.test_hier) {
    intlst_print("requests", s.requests) ;
    intlst_print("repairs", s.repairs) ;
  }
  if (s.test_incr) {
    printf("requests: %lu\n", s.remote_request) ;
    printf("repairs: %lu\n", s.remote_repair) ;
  }
  if (s.test_delay) {
    sprintf(fname, "data/%s", myname) ;
    intlst_print_file(s.delay, fname) ;
    /* intlst_print("latency", s.delay) ; */
  }
  return ;
}


static void send_request(Type type, SAin dst, Seqno seqno) {
  char buf[1024] ;
  size_t offset ;
  Hdr hdr ;
  
  hdr.type = type ;
  hdr.name = myname ;
  hdr.addr = myaddr ;
  hdr.seqno = seqno ;
  hdr.ts = Gettimeofday() ;
  offset = 0 ;
  hdr_ms(buf, &offset, hdr) ;
  dst.sin_port = pt2pt_port ;
  Sendto(pt2pt_xmit_sock, buf, offset, &dst) ;
  return ;
}


static void send_search_done(Seqno seqno) {
  char buf[1024] ;
  size_t offset ;
  Hdr hdr ;

  hdr.type = SEARCH_DONE ;
  hdr.name = myname ;
  hdr.addr = myaddr ;
  hdr.seqno = seqno ;
  offset = 0 ;
  hdr_ms(buf, &offset, hdr) ;
  Sendto(cast_xmit_sock, buf, offset, &grpaddr) ;
  return ;
}


static void report_lost(Seqno lost) {
  Seqno seqno ;
  Item p ;

  for (seqno = iq->read; seqno < iq->hi; seqno++) {
    p = iq_get_item(iq, seqno) ;
    switch (p->type) {
      case IQ_DATA:
	appl_deliver(p->msg, p->size) ;
	s.slow_read++ ;
	break ;

      case IQ_PENDING:
	if (p->local_try > 0) {
	  timer_cancel(p->local_timer) ;
	}
	if (p->remote_try > 0) {
	  timer_cancel(p->remote_timer) ;
	}
	p->type = IQ_LOST ;
	printf("RRMP: msg %lu is lost\n", seqno) ;
	appl_report_lost() ;
	break ;

      default:
	fprintf(stderr, "report_lost: unknown item type\n") ;
	exit(1) ;
    }
  }
  iq->read = iq->hi ;
  return ;
}


static void send_local_request(Seqno seqno) {
  SAin dst ;
  double delay ;
  char *name ;
  Item p ;

  if (nlocal == 1) return ;
  p = iq_get_item(iq, seqno) ;
  assert (p != NULL) ;
  assert (p->type == IQ_PENDING) ;
  p->local_try++ ;
  /* If a region has no parent region, keep sending local requests until a loss is recovered.
   * However, if the process takes unreasonably long, give up on the msg and report lost.
   */
  if (p->local_try <= max_local_try || (nremote == 0 && p->local_try <= nlocal * 2)) {
    name = choose_local(&dst, &delay) ;
    p->local_timer = timer_req("Local requst", delay, send_local_request, seqno) ;
    if (DEBUG) {
      printf("Local request: seqno=%lu dst=%s ntries: (%lu, %lu) wait: %.2fms Time: %.3f\n", 
	     seqno, name, p->remote_try, p->local_try, delay*1000, get_time()) ;
    }
    send_request(LOCAL_REQUEST, dst, seqno) ;
    s.local_request++ ;
  } else {
    p->local_try = 0 ;
    if (nremote == 0) {
      report_lost(seqno) ;
      return ;
    }
  }
  return ;
}


/* should allow lambda to be fractional */
static void send_remote_request(Seqno seqno) {
  SAin dst ;
  double delay ;
  char *name ;
  Item p ;

  p = iq_get_item(iq, seqno) ;
  assert (p != NULL) ;
  assert (p->type == IQ_PENDING) ;
  if (p->remote_try == 1) {
    dtime_set(seqno, Gettimeofday()) ;
  } else if (p->remote_try >= 10) {
    report_lost(seqno) ;
    return ;
  }
  p->remote_try++ ;

  if (nremote == 0) {
    send_local_request(seqno) ;
    return ;
  }

  name = choose_remote(&dst, &delay) ;
  p->remote_timer = timer_req("Remote request", delay, send_remote_request, seqno) ;
  if (rand_int(nlocal) < lambda) {
    if (DEBUG) {
      printf("Remote request: seqno=%lu dst=%s ntries: %lu wait: %.2fms Time: %.3f\n", 
	     seqno, name, p->remote_try, delay*1000, get_time()) ;
    }
    send_request(REMOTE_REQUEST, dst, seqno) ;
    s.remote_request++ ;
  }

  if (p->local_try == 0) {
    send_local_request(seqno) ;
  }
  return ;
}

static void form_search_request(Seqno seqno, SAin dst, SAin target) {
  char buf[1024] ;
  size_t offset ;
  Hdr hdr ;

  hdr.type = SEARCH_REQUEST ;
  hdr.name = myname ;
  hdr.addr = myaddr ;
  hdr.target = target ;
  hdr.seqno = seqno ;
  offset = 0 ;
  hdr_ms(buf, &offset, hdr) ;
  dst.sin_port = pt2pt_port ;
  Sendto(pt2pt_xmit_sock, buf, offset, &dst) ;
  return ;
}


static void send_search_request(Seqno seqno) {
  SAin dst, target ;
  char *name ;
  double delay ;
  UL ntries ;
  timer_id id ;

  if (nlocal == 1) return ;
  ntries = cnt_get(SEARCH_REQUEST, seqno, &target) ;
  cnt_incr(SEARCH_REQUEST, seqno, target) ;
  if (ntries <= 4) {
    /* don't search forever */
    name = choose_local(&dst, &delay) ;
    id = timer_req("Search request", delay, send_search_request, seqno) ;
    pending_set(SEARCH_REQUEST, seqno, id) ;
    if (DEBUG) {
      printf("Search request: seqno=%lu dst=%s ntries: %lu wait: %.2fms Time: %.3f\n", seqno, name, ntries, delay*1000, get_time()) ;
    }
    form_search_request(seqno, dst, target) ;
    s.search_request++ ;
  } else {
    cnt_del(SEARCH_REQUEST, seqno) ;
    pending_del(SEARCH_REQUEST, seqno) ;
  }
  return ;
}


static void cancel_request(Seqno seqno) {
  TV t ;
  double interval ;

  if (dtime_exist(seqno)) {
    t = dtime_get(seqno) ;
    interval = time_diff(Gettimeofday(), t) ;
    /* value is in tenth millisecond */
    intlst_append(s.delay, (int)(interval * 10000)) ;
    if (DEBUG) printf("delay for seqno %lu: %.3f\n", seqno, interval * 1000) ;
  } else {
  }
  return ;
}


void nack(Seqno hi) {
  Seqno ohi, seqno ;
  Item p ;

  ohi = iq->hi ;
  iq_set_hi(iq, hi) ;
  for (seqno = ohi; seqno < hi; seqno++) {
    p = iq_get_item(iq, seqno) ;
    p->type = IQ_PENDING ;
    p->local_try = p->remote_try = 0 ;
    p->waiting = false ;
    /* can delay sending a request if the network often re-orders packets */
    send_remote_request(seqno) ;
  }
  return ;
}


static void send_repair(Type type, SAin dst, Seqno seqno, char *msg, size_t size, TV ts, double delta) {
  char buf[1024] ;
  size_t offset ;
  Hdr hdr ;

  hdr.type = type ;
  hdr.name = myname ;
  hdr.addr = myaddr ;
  hdr.seqno = seqno ;
  hdr.ts = ts ;
  hdr.delta = delta ;
  offset = 0 ;
  hdr_ms(buf, &offset, hdr) ;
  dst.sin_port = pt2pt_port ;
  Sendmsg(pt2pt_xmit_sock, buf, offset, msg, size, &dst) ;
  return ;
}


static void relay_repair(Item p, Seqno seqno) {
  double delta ;
  
  if (p->waiting) {
    delta = time_diff(Gettimeofday(), p->recv_time) ;
    send_repair(REMOTE_REPAIR, p->dst, seqno, p->msg, p->size, p->ts, delta) ;
  }
  return ;
}


static void check_deliver(Iq iq) {
  Seqno seqno ;
  Item p ;

  for (seqno = iq->read; seqno < iq->hi; seqno++) {
    p = iq_get_item(iq, seqno) ;
    assert (p != NULL) ;
    if (p->type == IQ_DATA) {
      appl_deliver(p->msg, p->size) ;
      s.slow_read++ ;
    } else {
      assert (p->type == IQ_PENDING) ;
      break ;
    }
  }
  iq->read = seqno ;
  return ;
}


static void do_account_hier(void) {
  intlst_append(s.requests, s.remote_request) ;
  intlst_append(s.repairs, s.remote_repair) ;
  s.remote_request = s.remote_repair = 0 ;
  if ((s.requests)->len < s.npoints) {
    timer_sweep("RRMP", 1.0, do_account_hier) ;
  }
  return ;
}


static void recv_data(SAin addr, Seqno seqno, char *msg, size_t size) {
  Item p ;
  bool duplicate ;

  if (!addr_eq(addr, sender)) {
    fprintf(stderr, "Error: mult-sender not supported\n") ;
    exit(1) ;
  }

  if (iq->hi == 0) {
    /* do not go back for old msgs */
    iq->low = iq->read = iq->hi = seqno ;
    if (s.test_hier) {
      timer_sweep("RRMP", 1.0, do_account_hier) ;
    }
    s.local_request = s.remote_request = 0 ;
    s.local_repair = s.remote_repair = 0 ;
  }

  /* should remove the debugging support into another module */
  if (drop_recv(RRMP, DATA, seqno)) {
    if (DEBUG) {
      printf("RRMP: dropping msg %lu Time: %.3f\n", seqno, get_time()) ;
    }
    s.drop_msg++ ;
    free(msg) ;
    return ;
  }

  check_short_buf() ;
  if (seqno == iq->read && seqno == iq->hi) {
    fast_read(iq, seqno, msg, size) ;
    appl_deliver(msg, size) ;
    s.fast_read++ ;
    return ;
  }

  if (seqno >= iq->hi) {
    nack(seqno) ;
    iq_insert(iq, seqno, msg, size) ;
    return ;
  }

  p = slow_read(iq, seqno, msg, size, &duplicate) ;
  if (duplicate) {
    printf("duplicate data msg %lu\n", seqno) ;
  } else {
    printf("slow read data msg %lu Time: %.3f\n", seqno, get_time()) ;
    cancel_request(seqno) ;
    check_deliver(iq) ;
    relay_repair(p, seqno) ;
  }

  return ;
}


static void send_local_cast(Seqno seqno, char *msg, size_t size, double rtt, SAin target) {
  char buf[1024] ;
  size_t offset ;
  Hdr hdr ;

  hdr.type = LOCAL_CAST ;
  hdr.name = myname ;
  hdr.addr = myaddr ;
  hdr.target = target ;
  hdr.seqno = seqno ;
  /* all members in a local region can share RTT estimation to a remote member */
  hdr.rtt = rtt ;
  offset = 0 ;
  hdr_ms(buf, &offset, hdr) ;
  Sendmsg(cast_xmit_sock, buf, offset, msg, size, &grpaddr) ;
  return ;
}


static void recv_local_request(char *name, SAin dst, Seqno seqno, TV ts) {
  Item p ;
  char *msg ;
  size_t size ;

  /* should ignore local requests after a local cast */
  if (seqno >= iq->hi) {
    if (DEBUG) printf("local request with seqno %lu trigger nack Time: %.3f\n", seqno, get_time()) ;
    nack(++seqno) ;
    /* iq->hi = seqno ;*/
    return ;
  }

  if (seqno < iq->low) {
    /* already discarded in short-term buffer */
    msg = longbuf_get(seqno, &size) ;
    if (msg == NULL) {
      /* cannot answer this local request */
      printf("cannot honor local request %lu\n", seqno) ;
    } else {
      if (DEBUG) printf("send msg out of longbuf\n") ;
      send_repair(LOCAL_REPAIR, dst, seqno, msg, size, ts, 0) ;
      s.local_repair++ ;
    }
    return ;
  }

  p = iq_get_item(iq, seqno) ;
  assert (p != NULL) ;
  switch (p->type) {
    case IQ_DATA:
      /* msg is available in short-term buffer */
      if (DEBUG) {
	printf("Local repair: seqno=%lu dst=%s Time: %.3f\n", seqno, name, get_time()) ;
      }
      p->last = Gettimeofday() ;
      send_repair(LOCAL_REPAIR, dst, seqno, p->msg, p->size, ts, 0) ;
      s.local_repair++ ;
      return ;

    case IQ_PENDING:
    case IQ_LOST:
      return ;

    default:
      fprintf(stderr, "Error(recv_remote_request): unknown iq type\n") ;
      exit(1) ;
  }
}


static void recv_remote_request(char *name, SAin dst, Seqno seqno, TV ts) {
  Item p ;
  char *msg ;
  size_t size ;

  if (seqno >= iq->hi) {
    /* never heard of this msg before */
    nack(++seqno) ;
    /*iq->hi = seqno ;*/
    return ;
  }

  if (seqno < iq->low) {
    /* already discarded in short-term buffer */
    msg = longbuf_get(seqno, &size) ;
    if (msg == NULL) {
      cnt_incr(SEARCH_REQUEST, seqno, dst) ;
      send_search_request(seqno) ;
    } else {
      if (DEBUG) printf("send msg out of longbuf\n") ;
      send_repair(REMOTE_REPAIR, dst, seqno, msg, size, ts, 0) ;
      s.remote_repair++ ;
    }
    return ;
  }

  p = iq_get_item(iq, seqno) ;
  assert (p != NULL) ;
  switch (p->type) {
    case IQ_DATA:
      /* msg is available in short-term buffer */
      if (DEBUG) printf("Remote repair: seqno=%lu dst=%s Time: %.3f\n", seqno, name, get_time()) ;
      send_repair(REMOTE_REPAIR, dst, seqno, p->msg, p->size, ts, 0) ;
      s.remote_repair++ ;
      return ;

    case IQ_PENDING:
      p->waiting = true ;
      p->dst = dst ;
      p->ts = ts ;
      p->recv_time = Gettimeofday() ;
      return ;

    case IQ_LOST:
      /* I already gave up on the msg */
      return ;

    default:
      fprintf(stderr, "Error(recv_remote_request): unknown iq type\n") ;
      exit(1) ;
  }
}


static void recv_search_request(char *name, SAin addr, SAin target, Seqno seqno) {
  Item p ;
  char *msg ;
  size_t size ;
  
  p = iq_get_item(iq, seqno) ;
  if (p != NULL && p->type == IQ_DATA) {
    if (DEBUG) printf("Search done: seqno=%lu Time: %.3f\n", seqno, get_time()) ;
    send_repair(REMOTE_REPAIR, target, seqno, p->msg, p->size, time_of_zero(), 0) ;
    s.remote_repair++ ;
    send_search_done(seqno) ;
    return ;
  }

  if (p == NULL) {
    msg = longbuf_get(seqno, &size) ;
    if (msg == NULL) {
      cnt_incr(SEARCH_REQUEST, seqno, target) ;
      send_search_request(seqno) ;
    } else {
      send_repair(REMOTE_REPAIR, target, seqno, msg, size, time_of_zero(), 0) ;
      s.remote_repair++ ;
      send_search_done(seqno) ;
    }
  }
  return ;
}


static void recv_local_repair(char *name, SAin addr, Seqno seqno, char *msg, size_t size, TV ts) {
  double rtt ;
  Item p ;
  bool duplicate ;

  if (DEBUG) {
    printf("RRMP: recv local repair from %s: seqno=%lu Time: %.3f\n", name, seqno, get_time()) ;
  }
  s.recv_ret++ ;
  rtt = time_diff(Gettimeofday(), ts) ;
  if (DEBUG) printf("Local RTT to %s: %.3fms\n", name, 1000 * rtt) ;
  set_local_rtt(name, rtt) ;
  p = slow_read(iq, seqno, msg, size, &duplicate) ;
  if (duplicate) s.bad_ret++ ;
  else {
    cancel_request(seqno) ;
    check_deliver(iq) ;
    relay_repair(p, seqno) ;
  }
  return ;
}


static void check_local_cast(Seqno seqno) {
  UL cnt ;
  Item p ;
  double delay ;

  if (nlocal < lambda) cnt = nlocal ;
  else cnt = lambda ;
  p = iq_get_item(iq, seqno) ;
  if (rand_int(cnt) < 1) {
      if (DEBUG) printf("Local cast: seqno=%lu\n", seqno) ;
      send_local_cast(seqno, p->msg, p->size, p->rtt, p->target) ;
  } else {
    /* do a randomized back-off and try again */
    if (DEBUG) printf("Back off local cast: seqno=%lu\n", seqno) ;
    p->local_cast_pending = true ;
    delay = get_max_rtt() ;
    p->local_cast_timer = timer_req("Local cast", delay, check_local_cast, seqno) ;
  }
  return ;
}


static void recv_remote_repair(char *name, SAin addr, Seqno seqno, char *msg, size_t size, TV ts, double delta) {
  double rtt ;
  Item p ;
  bool duplicate ;

  if (DEBUG) {
    printf("RRMP: recv remote repair from %s: seqno=%lu Time: %.3f\n", name, seqno, get_time()) ;
  }
  s.recv_ret++ ;
  if (!time_is_zero(ts)) {
    rtt = time_diff(Gettimeofday(), ts) - delta ;
    if (DEBUG) printf("Remote RTT to %s: %.3fms\n", name, 1000 * rtt) ;
    set_remote_rtt(name, rtt) ;
  }
  p = slow_read(iq, seqno, msg, size, &duplicate) ;
  if (duplicate) s.bad_ret++ ;
  else {
    cancel_request(seqno) ;
    check_deliver(iq) ;
    relay_repair(p, seqno) ;
    p->rtt = rtt ;
    p->target = addr ;
    check_local_cast(seqno) ;
  }
  return ;
}


static void recv_local_cast(char *name, SAin addr, Seqno seqno, char *msg, size_t size, double rtt, SAin target) {
  Item p ;
  bool duplicate ;

  if (DEBUG) printf("RRMP: recv local cast from %s: seqno=%lu\n", name, seqno) ;
  /* should set "ignore" for this msg */
  s.recv_ret++ ;
  if (rtt > 0) {
    set_remote_rtt(get_remote_name(target), rtt) ;
  }
  if (seqno >= iq->hi) {
    nack(seqno) ;
    iq_insert(iq, seqno, msg, size) ;
    return ;
  }

  p = iq_get_item(iq, seqno) ;
  if (p != NULL && p->type == IQ_DATA) {
    if (p->local_cast_pending) {
      if (DEBUG) printf("RRMP: cancel local_cast for msg %lu\n", seqno) ;
      timer_cancel(p->local_cast_timer) ;
      p->local_cast_pending = false ;
    }
  }

  p = slow_read(iq, seqno, msg, size, &duplicate) ;
  if (duplicate) s.bad_ret++ ;
  else {
    cancel_request(seqno) ;
    check_deliver(iq) ;
    relay_repair(p, seqno) ;
  }

  return ;
}


static void recv_search_done(char *name, SAin addr, Seqno seqno) {
  if (pending_exist(SEARCH_REQUEST, seqno)) {
    pending_cancel(SEARCH_REQUEST, seqno) ;
    cnt_del(SEARCH_REQUEST, seqno) ;
  }
  return ;
}


void rrmp_cast(char *msg, size_t size) {
  static Seqno seqno = 0 ;
  Hdr hdr ;
  char buf[1024] ;
  size_t offset ;

  if (!am_sender) {
    fprintf(stderr, "need to specify as the sender in order to multicast data msg\n") ;
    exit(1) ;
  }

  if (iq->hi == 0) {
    if (s.test_hier) {
      timer_sweep("RRMP", 1.0, do_account_hier) ;
    }
    s.local_request = s.remote_request = 0 ;
    s.local_repair = s.remote_repair = 0 ;
  }
  /* save a local copy */
  fast_read(iq, seqno, msg, size) ;
  s.fast_read++ ;

  hdr.type = DATA ;
  hdr.name = myname ;
  hdr.addr = myaddr ;
  hdr.seqno = seqno++ ;
  offset = 0 ;
  hdr_ms(buf, &offset, hdr) ;
  Sendmsg(cast_xmit_sock, buf, offset, msg, size, &grpaddr) ;
  return ;
}


void rrmp_recv(char *buf, size_t offset, size_t size) {
  Hdr hdr ;
  char *msg ;

  hdr = hdr_um(buf, &offset) ;
  switch (hdr.type) {
    case DATA:
    case LOCAL_REPAIR:
    case REMOTE_REPAIR:
    case LOCAL_CAST:
      size -= offset ;
      msg = Malloc(size) ;
      memcpy(msg, buf+offset, size) ;
      break ;
  }

  switch (hdr.type) {
    case DATA:
      recv_data(hdr.addr, hdr.seqno, msg, size) ;
      break ;

    case LOCAL_REQUEST:
      recv_local_request(hdr.name, hdr.addr, hdr.seqno, hdr.ts) ;
      break ;

    case REMOTE_REQUEST:
      recv_remote_request(hdr.name, hdr.addr, hdr.seqno, hdr.ts) ;
      break ;

    case SEARCH_REQUEST:
      recv_search_request(hdr.name, hdr.addr, hdr.target, hdr.seqno) ;
      break ;

    case LOCAL_REPAIR:
      recv_local_repair(hdr.name, hdr.addr, hdr.seqno, msg, size, hdr.ts) ;
      break ;
      
    case REMOTE_REPAIR:
      recv_remote_repair(hdr.name, hdr.addr, hdr.seqno, msg, size, hdr.ts, hdr.delta) ;
      break ;

    case LOCAL_CAST:
      recv_local_cast(hdr.name, hdr.addr, hdr.seqno, msg, size, hdr.rtt, hdr.target) ;
      break ;

    case SEARCH_DONE:
      recv_search_done(hdr.name, hdr.addr, hdr.seqno) ;
      break ;

    default:
      printf("Error[RRMP]: Unknown msg type\n") ;
      exit(1) ;
  }
  return ;
}

