// ReliableNinjaRMI, by Zhongwei Li    (zhongwei@cs.cornell.edu)
//					    XiangJiang Ma  (xiangjiang@cs.cornell.edu)
//					    Yiwen Wang     (yiwen@cs.cornell.edu)
//
// Acknowledged:		
// NinjaRMI, by Matt Welsh (mdw@cs.berkeley.edu)
// See http://www.cs.berkeley.edu/~mdw/proj/ninja for details
/*
 * "Copyright (c) 1998 by The Regents of the University of California
 *  All rights reserved."
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */

package ninja.rmi.registry;

import java.rmi.registry.Registry;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Vector;
import java.util.Random;
import java.rmi.*;
import ninja.rmi.*;
import java.net.InetAddress;

import java.net.Socket;

/**
 * NewNinjaRegistryImpl is an implementation of the Registry interface, and
 * acts as a registry for remote objects, which enable multiple remote servers 
 * share the same registry name.
 */
public class NinjaRegistryImpl extends NinjaRemoteObject implements Registry {


  private Hashtable services;
  private InetAddress localhost;
  private String localhostname;
 
  public NinjaRegistryImpl() throws RemoteException {
    super((NinjaExportData)null); // Don't export ourselves yet
    NinjaExportData exportData = new NinjaExportData();
    exportData.port = Registry.REGISTRY_PORT;
    exportData.check_objid = false;
    this.exportObject(exportData); 
    
    services = new Hashtable();
    try {
      localhost = InetAddress.getLocalHost();
      localhostname = localhost.getHostName();
	  System.out.println("the registry is on host: " + localhostname + '\n');
    } catch (Exception e) {
      throw new RemoteException(e.getMessage());
    }
  }

   
  // The registry try to ping one of the server replicas with the corresponding name
  // (by building a socket), if the server is OK, then return its reference to client, otherwise,
  // unbind the server which is currently down.   -- Zhongwei
  public Remote lookup(String name) throws RemoteException {
    Vector remobj_v = (Vector)services.get(name);
	if (remobj_v == null) throw new RemoteException("Can't find "+name);
	else {
		boolean flag = false;
		int index = 0;;
		Random rand = new Random();  // random seed is based on the current time
		NinjaRemoteRef ref = null;
		Socket sock = null;
		while (sock == null)
		{
			int n = remobj_v.size();
			// server loading balance -- randomly choose the remote object.
			index = rand.nextInt(n);   // 0 <= index < n
			System.out.println("Get the " + index+"th remote object from vector!\n");
			ref = ((NinjaRemoteStub)remobj_v.get(index)).getNinjaRemoteRef();
			try {
			 sock = new Socket(ref.get_remotehost(), ref.get_remoteport());
			} catch (Exception e) {
				System.out.println("NinjaRemoteRef: Can't get socket to "+ref.get_remotehost()+":"+ref.get_remoteport() + '\n');
				System.out.println(e.getMessage());
				//unbind the unavailable remote object
				remobj_v.remove(index);  
				flag = true;
			}
		}
		if(flag)  // if the object vector is changed 
			services.put(name, remobj_v);
					
		System.out.println("Got service in RMIC, server is "+ref.get_remotehost()+":"+ref.get_remoteport());
		
		return (Remote)remobj_v.get(index);
	}
  }

public void bind(String name, Remote obj) throws RemoteException {
    allowBind();
	((NinjaRemoteStub)obj).getNinjaRemoteRef().service_name = name;
	Vector oldval = (Vector)services.get(name);
	if (oldval == null) {
      Vector v = new Vector();
	  v.add(obj);
	  services.put(name, v);
    }
	else {
		oldval.add(obj);
		services.put(name, oldval);
	}
  }	

// remove the whole element vector corresponding to a certain name
public void unbind(String name) throws RemoteException {
    allowBind();
    Vector val = (Vector)services.get(name);
    if (val == null) throw new RemoteException(name + " not bound");
	services.remove(name); 
	System.out.println("unbind the unavalaible server...\n");
 }

/* same as the bind() */
public void rebind(String name, Remote obj) throws RemoteException {
	System.out.println("rebinding to the registry...!\n");
	allowBind();
	// save the name of the remote object into the remote reference
	((NinjaRemoteStub)obj).getNinjaRemoteRef().service_name = name;
	
	System.out.println("bind service: " + ((NinjaRemoteStub)obj).getNinjaRemoteRef().service_name);
	
	Vector oldval = (Vector)services.get(name);
	if (oldval == null) {
      Vector v = new Vector();
	  v.add(obj);
	  services.put(name, v);
	  System.out.println("binding successfully!\n");
    }
	else {
		System.out.println("rebinding successfully!\n");
		oldval.add(obj);
		services.put(name, oldval);
	}
}

//new method:  remove a particular remote object from the registry hashtable
public void remove(String name, Remote obj) throws RemoteException {
	allowBind();
	Vector oldval = (Vector)services.get(name);
	if (oldval == null) throw new RemoteException(name + " not bound");
	else {
		oldval.remove(obj);
		if(oldval.isEmpty())
			services.remove(name);
		else
			services.put(name, oldval);
	}
  }

  public String[] list() throws RemoteException {
    int n = services.size();
    String thelist[] = new String[n];
    Enumeration e = services.keys();
    n = 0;
    while (e.hasMoreElements()) {
      thelist[n] = (String)e.nextElement();
      n++;
    }
    return thelist;
  }

  private void allowBind() throws RemoteException {
    String clientname;
    try {
      Thread t = Thread.currentThread();
      if (t instanceof ninja.rmi.Reliable_ServerThread) {
        clientname = ((ninja.rmi.Reliable_ServerThread)t).getClientHost();
      } else {
	throw new RemoteException("Registry must use Reliable transport");
      }
      
      InetAddress client = InetAddress.getByName(clientname);
	  System.out.println("client: " + client + '\n');
	  System.out.println("clientname: " + clientname + '\n');

      // Check the name _and_ the IP address - to resolve problems with,
      // e.g., 'asbestos' and 'asbestos.cs.cornell.edu' or multiple names/IPs
      // per host
	/*  
      if (!localhostname.equals(clientname) &&
	  !localhost.equals(client)) {
	throw new RemoteException("Ninja RmiRegistry: Localhost "+localhostname+" doesn't match client host "+clientname);
      }
	  */
    } catch (Exception e) {
      throw new RemoteException(e.getMessage());
    }
	  
  }
      

  public static void main(String args[]) {
    try {
      NinjaRegistryImpl reg = new NinjaRegistryImpl();
    } catch (Exception e) {
      System.out.println("Can't create registry: "+e.getMessage());
      System.exit(1);
    }
    System.out.println("Registry started.");
    
  }
  
}
