// NinjaRMI, by Matt Welsh (mdw@cs.berkeley.edu)
// See http://www.cs.berkeley.edu/~mdw/proj/ninja for details

/*
 * "Copyright (c) 1998 by The Regents of the University of California
 *  All rights reserved."
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */

package ninja.rmi.registry;

import java.rmi.registry.Registry;
import java.rmi.*;
import ninja.rmi.*;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.rmi.server.ObjID;
import java.rmi.server.RemoteStub;

// Class with static methods to hand us back a Registry from a hostname
// and a port

/**
 * NinjaLocateRegistry contains static methods used to obtain a
 * handle to a NinjaRegistryImpl at a given hostname and port.
 * Usage might be:
 * <tt>Registry reg = NinjaLocateRegistry.getRegistry(hostname, port);</tt>
 *
 * <p>The default port is Registry.REGISTRY_PORT.
 */
public class NinjaLocateRegistry {

  public NinjaLocateRegistry() {
  }

  /**
   * Obtains a handle to the registry at the local host with the given
   * port.
   */
  public static Registry getRegistry(int port) throws ClassNotFoundException, IllegalAccessException, InstantiationException, InvocationTargetException {
    return getRegistry("localhost", port);
  }

  /**
   * Obtains a handle to the registry at the given hostname with the
   * default port (Registry.REGISTRY_PORT).
   */
  public static Registry getRegistry(String hostname) throws ClassNotFoundException, IllegalAccessException, InstantiationException, InvocationTargetException {
    return getRegistry(hostname, Registry.REGISTRY_PORT);
  }

  /**
   * Obtains a handle to the registry at the given hostname and port number.
   */
  public static Registry getRegistry(String hostname, int port) throws ClassNotFoundException, IllegalAccessException, InstantiationException, InvocationTargetException {
    NinjaRemoteRef remref = new NinjaRemoteRef(hostname, port, new ObjID(), NinjaExportData.RMI_COMM_TYPE_RELIABLE, null);

    // XXX Tricky crap: Don't reference stub class directly here, so we
    // can compile ourselves
    Class stubclass = Class.forName("ninja.rmi.registry.NinjaRegistryImpl__RMIStub");

    // OK, now find the right constructor so we can shove the RemoteRef
    // into it

    int c, p;
    boolean found_constructor = false;
    NinjaRemoteStub stub = null;
    Constructor constarr[] = stubclass.getConstructors();
    for (c = 0; (c < constarr.length) && !found_constructor; c++) {
      Class paramarr[] = constarr[c].getParameterTypes();
      if ((paramarr.length == 1) &&
	  (paramarr[0].getName().equals("ninja.rmi.NinjaRemoteRef"))) {
	Object inargs[] = new Object[1];
	inargs[0] = (ninja.rmi.NinjaRemoteRef)remref;
	stub = (NinjaRemoteStub)constarr[c].newInstance(inargs);
	found_constructor = true;
      }
    }
    if (!found_constructor) {
      throw new ClassNotFoundException("Can't find NinjaRemoteRef constructor for NinjaRegistryImpl__RMIStub class");
    }
    return (Registry)stub;
  }
  
  /**
   * Obtains a handle to the registry(a new type of registry, which enable multiple remote object shared the same service name)
   *  at the given hostname and port number.	-- Zhongwei
   */
  public static Registry getNewRegistry(String hostname, int port ) throws ClassNotFoundException, IllegalAccessException, InstantiationException, InvocationTargetException {
    NinjaRemoteRef remref = new NinjaRemoteRef(hostname, port, new ObjID(), NinjaExportData.RMI_COMM_TYPE_RELIABLE, null);

    Class stubclass = Class.forName("ninja.rmi.registry.NewNinjaRegistryImpl__RMIStub");

    int c, p;
    boolean found_constructor = false;
    NinjaRemoteStub stub = null;
    Constructor constarr[] = stubclass.getConstructors();
    for (c = 0; (c < constarr.length) && !found_constructor; c++) {
      Class paramarr[] = constarr[c].getParameterTypes();
      if ((paramarr.length == 1) &&
	  (paramarr[0].getName().equals("ninja.rmi.NinjaRemoteRef"))) {
	Object inargs[] = new Object[1];
	inargs[0] = (ninja.rmi.NinjaRemoteRef)remref;
	stub = (NinjaRemoteStub)constarr[c].newInstance(inargs);
	found_constructor = true;
      }
    }
    if (!found_constructor) {
      throw new ClassNotFoundException("Can't find NinjaRemoteRef constructor for NinjaRegistryImpl__RMIStub class");
    }
    return (Registry)stub;
  }
  
}
