// NinjaRMI, by Matt Welsh (mdw@cs.berkeley.edu)
// See http://www.cs.berkeley.edu/~mdw/proj/ninja for details

/*
 * "Copyright (c) 1998 by The Regents of the University of California
 *  All rights reserved."
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */





package ninja.rmi;

import java.rmi.*;
import java.rmi.server.*;
import java.net.ServerSocket;
import java.net.Socket;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.OutputStream;
import java.io.InputStream;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.net.InetAddress;

/**
 * Reliable_ServerThread is a NinjaServerThread which deals with reliable
 * connections using TCP sockets. The getClientHost()/getClientPort()
 * methods are provided to allow threads to identify the client of an
 * RMI call in their thread.
 */
public class Reliable_ServerThread extends NinjaServerThread {
  private ServerSocket serverSocket;
  private Socket connSocket;
  private int port;
  private boolean singleOp = false;
  private boolean is_server;

  Reliable_ServerThread(NinjaServerRef serverref, int theport) throws Exception {
    super(serverref);
    is_server = true;
    serverSocket = new ServerSocket(theport);
    port = serverSocket.getLocalPort();
    this.setName("Reliable_ServerThread (toplevel) on port "+port);
  }

  // Constructor used to fork thread to deal with new incoming connection
  private Reliable_ServerThread(NinjaServerRef serverref, Socket theconnsock) throws Exception {
    super(serverref);
    is_server = false;
    connSocket = theconnsock;
    port = theconnsock.getPort();
    this.setName("Reliable_ServerThread for connection on port "+port);
  }
    
  /**
   * Returns the hostname of the client making the RMI call within this
   * thread.
   */
  public String getClientHost() throws ServerNotActiveException {
    if (connSocket == null) throw new ServerNotActiveException("Thread has no active client connection");
    return connSocket.getInetAddress().getHostName();
  }
  
  /**
   * Returns the remote port number of the client making the RMI call within
   * this thread.
   */
  public int getClientPort() throws ServerNotActiveException {
    if (connSocket == null) throw new ServerNotActiveException("Thread has no active client connection");
    return connSocket.getPort();
  }

  /**
   * Returns the port that the server is listening on for new connections.
   */
  public int getServerPort() throws ServerNotActiveException {
    if (serverSocket == null) throw new ServerNotActiveException("Thread has no active server socket");
    return port;
  }

  // Fork new thread to handle this connection
  private void relisten(Socket sock) throws Exception {
    Reliable_ServerThread thethread = new Reliable_ServerThread(servref, sock);
     ;
    thethread.start();
  }

  // Listen on serverSocket for connections
  public void run() {
     ;

    if (is_server) {

      // Listen on the socket for incoming connections

      Socket sock;

      while (true == true) {
	try {
	  // XXX mdw: This doesn't seem to return if there's any active
	  // client thread (which isn't sleeping or yielding). Yikes! 
	  sock = serverSocket.accept();
	} catch (Exception e) {
	   ;
	  return;
	}
	
	if (servref.exportData.callbacks != null) {
	  servref.exportData.callbacks.socket_created(sock.getInetAddress().getHostName(),
						      sock.getPort());
	}
	
	try {
	  sock.setTcpNoDelay(true);
	} catch (Exception e) {
	  // Do nothing
	}

	try {
	  relisten(sock);
	} catch (Exception e) {
	   ;
	  return;
	}
	
      }
      
    } else {

      // Handle a connection
      Thread.currentThread().yield();
      
      Socket sock = connSocket;
      BufferedInputStream bis;
      BufferedOutputStream bos;
      DataInputStream datais;
      DataOutputStream dataos;
      int i;
      short s;
      byte b;
      
      try {
	bis = new BufferedInputStream(sock.getInputStream());
	bos = new BufferedOutputStream(sock.getOutputStream());
	datais = new DataInputStream(bis);
	dataos = new DataOutputStream(bos);
	i = datais.readInt();
	s = datais.readShort();
	b = datais.readByte();
	
      } catch (Exception e) {
	 ;
	return;
      }
      
      if (i != 0x4e524d49 || s != 0x01) {
	 ;
	try {
	  sock.close();
	} catch (IOException e) {
	  // Do nothing
	}
      }
      
      if ((b != 0x4b) && (b != 0x4c)) {
	 ;
	try {
	  sock.close();
	} catch (IOException e) {
	  // Do nothing
	}
      }
      
      if (b == 0x4c) { this.singleOp = true; }
      
      // XXX XXX XXX mdw Need to send back 'protocol OK' and create
      // endpoint ID here, etc.
      
       ;

      while (true == true) {
	
	try {
	  b = datais.readByte();
        } catch (IOException e) {
	   ;

          if (servref.exportData.callbacks != null) {
            servref.exportData.callbacks.socket_destroyed(sock.getInetAddress().getHostName(), sock.getPort());
          }
	  return;

        } catch (Exception e) {
	   ;
	  
	  if (servref.exportData.callbacks != null) {
	    servref.exportData.callbacks.socket_destroyed(sock.getInetAddress().getHostName(),
							  sock.getPort());
	  }
	  return;
	}
	
      msg_switch: {
	  switch (b) {
	  case 0x50: // Call
	     ;
	    
	    NinjaRemoteCall call = new Reliable_RemoteCall(bis, bos);
	    ObjectInput callis;
	    ObjID theobj;
	    
	    try {
	      callis = call.getInputStream();
	       ;
	      theobj = ObjID.read(callis);
	       ;
	    } catch (IOException e) {
	       ;
	      
	      if (servref.exportData.callbacks != null) {
		servref.exportData.callbacks.socket_destroyed(sock.getInetAddress().getHostName(),
							      sock.getLocalPort());
	      }
	      return;
	    }
	    
	    // XXX If anything goes wrong at this point (except for sockets
	    // closing) need to shove an exception down the wire
	    
	    if (!servref.objid.equals(theobj)) {
	      // Do we care if the ObjID doesn't match?
	      if (servref.exportData.check_objid == true) {
		System.out.println("NinjaServerThread["+port+"]: ObjID doesn't match");
		ObjectOutput resultos;
		try {
		  resultos = call.getResultStream(false);
		  resultos.writeObject(new ServerException("NinjaRMI Server execption: Object ID on incoming call does not match that of server."));
		} catch (Exception e) {
		   ;
		}
		try {
		  sock.close();
		} catch (Exception e) {
		}
	      }
	    }
	    
	    int op;
	    long hash;
	    
	    try {
	      op = callis.readInt();
	      hash = callis.readLong();
	    } catch (IOException e) {
	       ;
	      break msg_switch;
	    }
	    
	     ;
	    try {
	      servref.skeleton.dispatch(servref.remote, call, op, hash);
	      dataos.flush();
	    } catch(Exception e) {
	       ;
	      
	      // OK, the server object threw an exception; we have to
	      // push it down the wire
	      ObjectOutput resultos;
	      try {
		resultos = call.getResultStream(false);
		resultos.writeObject(new ServerException("NinjaRMI Server threw exception: ", e));
	      } catch (Exception e2) {
		 ;
		try {
		  sock.close();
		} catch (Exception e3) {
		  // Do nothing
		}
	      }
	    }
	    
	    try {
	       ;
	      call.releaseOutputStream();
	    } catch (IOException e) {
	       ;
	    }
	    
	    break;
	    
	  case 0x52: // Ping
	     ;
	    try {
	      dataos.writeByte(0x53);  // Ping ack
	      dataos.flush();
	    } catch(Exception e) {
	      // Do nothing
	    }
	    break;
	    
	  case 0x54: // DgcAck
	    // XXX Don't know what to do with this
	     ;
	    break;
	    
	  default:
	     ;
	    break;
	    
	  }
	}
	
	if (this.singleOp) {
	  try {
	    sock.close();
	  } catch (IOException e) {
	    // Do nothing
	  }
	  return;
	}
	
      }
      
    }
    
  }

}
