// NinjaRMI, by Matt Welsh (mdw@cs.berkeley.edu)
// See http://www.cs.berkeley.edu/~mdw/proj/ninja for details

/*
 * "Copyright (c) 1998 by The Regents of the University of California
 *  All rights reserved."
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */





package ninja.rmi;

import java.rmi.server.RemoteObject;
import java.rmi.server.RemoteCall;
import java.rmi.RemoteException;
import java.rmi.Remote;
import java.rmi.server.Operation;
import java.rmi.server.ObjID;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.IOException;
import java.lang.ClassNotFoundException;
import java.rmi.server.RMISocketFactory;
import java.net.Socket;
import java.net.InetAddress;
import java.net.DatagramPacket;
import java.io.DataOutputStream;
import java.net.DatagramSocket;
import java.net.MulticastSocket;
import java.io.BufferedOutputStream;
import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ByteArrayInputStream;
import java.io.Serializable;


/**
 * NinjaRemoteRef is the client-side reference to a remote object.
 * The NinjaRemoteRef is contained within the client-side stub (which
 * has been obtained from the registry). Users shouldn't need to deal
 * with the NinjaRemoteRef at all; it is made public simply so that
 * the generated stubs can access it.
 *
 * <p>The idea is that the RMI server-side code instantiates the
 * stub and creates a NinjaRemoteRef and shoves it into the stub. The
 * stub is then handed to the registry, which passes it along to clients.
 * Therefore, the methods in NinjaRemoteRef need to be accessible to
 * the stub (to make remote calls) and to the Ninja RMI server code (to
 * create the NinjaRemoteRef when exporting a NinjaRemoteObject).
 */
public class NinjaRemoteRef implements Serializable {

  protected String remotehost;
  protected int remoteport; // The port to call on for this object
  protected ObjID objid;
  protected int commtype;
  
  private Socket sock = null;
  private BufferedInputStream bis = null;
  private BufferedOutputStream bos = null;
  private ByteArrayOutputStream baos = null;
	
  // name of remote interface which is binded with current ref object -- Zhongwei 
  public String service_name;	

  public NinjaRemoteRef() {
     ;
  }

  /**
   * Creates a NinjaRemoteRef which points to the object on the given
   * hostname and port with the given object ID. Used to manually bootstrap
   * a client-side reference to a remote object (that is, without going
   * through the registry).
   * @param host The hostname to contact to call on this remote object.
   * @param port The port to contact to call on this remote object.
   * @param oid The ObjID corresponding to the remote object.
   * @param comm_type One of NinjaExportData.RMI_COMM_TYPE_* values,
   *  specifying the communication semantics for the call.
   */
  public NinjaRemoteRef(String host, int port, ObjID oid, int comm_type, String name) {
     ;
    remotehost = host;
    remoteport = port;
    objid = oid;
    commtype = comm_type;
	service_name = name;  //Zhongwei
	
  }




  /**
   * Invokes the given method with the given params on the remote object.
   * Called by the client-side stub.
   */
  public Object invoke(Remote obj,
		       java.lang.reflect.Method method,
		       Object[] params,
		       long opnum) throws Exception {
     ;
    return null;
  }

  /**
   * Returns a NinjaRemoteCall, used by the stub to make method calls on the
   * remote object.
   */
  public NinjaRemoteCall newCall(int opnum, long hash) throws RemoteException {

     ;

    // XXX mdw Cleanup: Should have NinjaRemoteRef 'remember' the current
    // call, have the call create sockets etc. if necessary...yuk.
    
    if (commtype == NinjaExportData.RMI_COMM_TYPE_RELIABLE) {
      
      // Need to create a new socket to the server?
      if (sock == null) {
	
	try {
	  sock = new Socket(remotehost, remoteport);
	} catch (Exception e) {
	  throw new RemoteException("NinjaRemoteRef: Can't get socket to "+remotehost+":"+remoteport, e);
	}
	
	// Socket has been created, let's push the protocol header on down
	try {
	  bis = new BufferedInputStream(sock.getInputStream());
	  bos = new BufferedOutputStream(sock.getOutputStream());
	  DataOutputStream dataos = new DataOutputStream(bos);
	  dataos.writeInt(0x4e524d49);  // Header "NRMI"
	  dataos.writeShort(0x0001);    // Version
	  dataos.writeByte(0x4b);       // Protocol: Stream
	  //dataos.flush();
	  // XXX XXX mdw: Need to get back 'protocol OK' and endpt ID
	} catch (Exception e) {
	  throw new RemoteException("NinjaRemoteRef: Can't push header", e);
	}
      }
      
      // This will send the call header
      // Stub picks up by writing arguments
      try {
	return new Reliable_RemoteCall(bis, bos, objid, opnum, hash);
      } catch (Exception e) {
	throw new RemoteException("NinjaRemoteRef: Can't create new NinjaRemoteCall", e);
      }

    } else if (commtype == NinjaExportData.RMI_COMM_TYPE_UNRELIABLE_ONEWAY) {

      baos = new ByteArrayOutputStream();
      DataOutputStream dataos = new DataOutputStream(baos);
      try {
        dataos.writeInt(0x4e524d49);  // Header "NRMI"
        dataos.writeShort(0x0002);    // Version
        //dataos.flush();
        return new UnrelOw_RemoteCall(baos,objid,opnum,hash);
      } catch (Exception e) {
        throw new RemoteException("NinjaRemoteRef.newCall(): Can't create unreliable call", e);
      }

    } else if (commtype == NinjaExportData.RMI_COMM_TYPE_MULTICAST_ONEWAY) {
      
      baos = new ByteArrayOutputStream();
      DataOutputStream dataos = new DataOutputStream(baos);
      try {
        dataos.writeInt(0x4e524d49);  // Header "NRMI"
        dataos.writeShort(0x0002);    // Version
        //dataos.flush();
        return new McastOw_RemoteCall(baos,objid,opnum,hash);
      } catch (Exception e) {
        throw new RemoteException("NinjaRemoteRef.newCall(): Can't create multicast call", e);
      }



    } else {
      throw new RemoteException("NinjaRemoteRef.newCall(): invalid commtype "+commtype);
    }
      
  }

  /**
   * Invokes the method on the remote object for the given RemoteCall. Used
   * by the stub.
   */
  public void invoke(NinjaRemoteCall call) throws Exception {
     ;
    call.executeCall();

    // XXX Cleanup: This should be part of the RemoteCall...
    if (commtype == NinjaExportData.RMI_COMM_TYPE_UNRELIABLE_ONEWAY) {
      // Wrap the data up in a datagram packet and send it
      baos.flush();
      byte[] ba = baos.toByteArray();
      DatagramPacket dp = new DatagramPacket(ba,ba.length,
					     InetAddress.getByName(remotehost),
					     remoteport);
      DatagramSocket ds = new DatagramSocket();
      ds.send(dp);
      
    } else if (commtype == NinjaExportData.RMI_COMM_TYPE_MULTICAST_ONEWAY) {
      // Wrap the data up in a datagram packet and send it
      baos.flush();
      byte[] ba = baos.toByteArray();
      DatagramPacket dp = new DatagramPacket(ba,ba.length,
					     InetAddress.getByName(remotehost),
					     remoteport);
      MulticastSocket ds = new MulticastSocket();
      ds.joinGroup(InetAddress.getByName(remotehost));
      ds.send(dp);
    }
  }

  /**
   * Finishes the remote method invocation. Used by the stub.
   */
  public void done(NinjaRemoteCall call) throws RemoteException {
     ;
    try {
      call.done();
    } catch (IOException e) {
      throw new RemoteException(e.getMessage());
    }
  }

  /**
   * Used to obtain the classname of the remote reference for serialization
   * purposes. Used by the stub.
   */
  public String getRefClass(java.io.ObjectOutput out) {
     ;
    // XXX mdw: You must use the 'ninjarmic' script to modify the output of
    // rmic to add code to the generated Stub class to override
    // readObject/writeObject so that a fully-qualified classname is
    // read/written, rather than one indirected from RemoteRef.packagePrefix.
    return "ninja.rmi.NinjaRemoteRef";
  }

  /**
   * Returns a hashcode on the remote reference. Used by the stub.
   */
  public int remoteHashCode() {
     ;
    return objid.hashCode();
  }

  /**
   * Determines if two NinjaRemoteRef's are equal - used by the stub.
   */
  public boolean remoteEquals(NinjaRemoteRef obj) {
     ;

    if (obj instanceof NinjaRemoteRef) return objid.equals(((NinjaRemoteRef)obj).objid);
    else return false;
  }

  /**
   * Returns a string representing this remote reference. Used by the stub.
   */
  public String remoteToString() {
     ;
    return new String("NinjaRemoteRef at "+remotehost+":"+remoteport);
  }

  /**
   * writeExternal is called on the NinjaRemoteRef when the stub is
   * serialized over the network.
   */
  public void writeExternal(ObjectOutput out) throws IOException {
     ;
    out.writeUTF(this.remotehost);
    out.writeInt(this.remoteport);
    this.objid.write(out);
    out.writeInt(this.commtype);
    return;
  }

  /**
   * readExternal is called on the NinjaRemoteRef when the stub is
   * serialized over the network.
   */
  public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
     ;
    this.remotehost = in.readUTF();
    this.remoteport = in.readInt();
    this.objid = ObjID.read(in);
    this.commtype = in.readInt();
    return;
  }

  /* The following methods are used by the client, which can extract the
   * NinjaRemoteRef from a Stub using the 'getNinjaRemoteRef' method
   * added to the stub by ninjarmic.
   */

  /**
   * Returns the fully-qualified hostname of the machine which this
   * NinjaRemoteRef connects to.
   */
  public String get_remotehost() {
    return remotehost;
  }

  /**
   * Returns the port number of the machine which this NinjaRemoteRef
   * connects to. This could be a TCP or a UDP port, depending on the
   * value of get_commtype().
   */
  public int get_remoteport() {
    return remoteport;
  }

  /**
   * Returns the object ID which this NinjaRemoteRef refers to. The
   * object ID is a unique identifier for the remote object, and is
   * checked on each remote method invocation to ensure that the
   * correct client-side stub is being used to make remote calls.
   * Some remote objects do not require the object ID to match.
   */
  public ObjID get_objid() {
    return objid;
  }

  /**
   * Returns the communications type of the connection used by this
   * NinjaRemoteRef. The commtype fields are defined in NinjaExportData.
   * @see NinjaExportData
   */
  public int get_commtype() {
    return commtype;
  }

  
}
    
