// NinjaRMI, by Matt Welsh (mdw@cs.berkeley.edu)
// See http://www.cs.berkeley.edu/~mdw/proj/ninja for details

/*
 * "Copyright (c) 1998 by The Regents of the University of California
 *  All rights reserved."
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */





package ninja.rmi;

import java.rmi.Remote;
import java.rmi.RemoteException;
import java.rmi.server.RemoteStub;
import java.rmi.server.ServerNotActiveException;

// XXX Technically this should extend java.rmi.server.RemoteServer,
// but I see no point.

/**
 * NinjaRemoteObject is the class which all remotely-accessible objects
 * using NinjaRMI must extend. By subclassing NinjaRemoteObject, the
 * object is automatically "exported" for remote access. Clients wishing
 * to access the remote object must obtain a stub - this is generally
 * done only through the registry.
 *
 * Various methods of NinjaRemoteObject are provided to allow the
 * server-side object to obtain information on the clients accessing it.
 * 
 * @see NinjaRegistryImpl
 */
public class NinjaRemoteObject {
  private NinjaServerRef servref; // The server-side reference for this object

  /**
   * The constructor for a NinjaRemoteObject, when called with
   * no arguments, exports the object for remote access using the default
   * settings.
   */
  protected NinjaRemoteObject() throws RemoteException {
    servref = null;
    exportObject(new NinjaExportData());
  }
  
  /** ----- Zhongwei ------
   * The constructor for a NinjaRemoteObject, used for building reliable NinjaRMI system( 
   * the argument "serviceName" is what being binded in registry table), exports the object for remote access using the default
   * settings.
   */
  protected NinjaRemoteObject(String serviceName) throws RemoteException {
    servref = null;
	NinjaExportData exportData = new NinjaExportData();
	exportData.service_name = serviceName;
    exportObject(exportData);
  }

  /**
   * When a NinjaExportData is passed into the NinjaRemoteObject
   * constructor, the behavior of the remote object can be modified
   * (for example, the TCP port on which the object is exported can
   * be specified). If <tt>null</tt> is passed in, then the NinjaRemoteObject
   * will not be exported; the user must call exportObject() directly.
   *
   * @see NinjaExportData
   */ 
  protected NinjaRemoteObject(NinjaExportData exportData) throws RemoteException {
    // If 'null' is passed in, we don't export the object: Rather we expect
    // the subclass will do it directly (after constructing the exportData
    // itself).
    servref = null;
    if (exportData != null) exportObject(exportData);
  }

  /**
   * exportObject exports the NinjaRemoteObject for remote access.
   * Only call this function if you call the NinjaRemoteObject constructor
   * with an argument of <tt>null</tt>. The intent here is that the
   * user can construct a NinjaExportData structure before calling
   * exportObject - because superclass constructors must be the first
   * thing called in a constructor.
   * @param exportData The NinjaExportData structure specifying how the
   *  object should be exported.
   */
  public void exportObject(NinjaExportData exportData) throws RemoteException {
     ;
    servref = new NinjaServerRef();

    if (exportData == null) throw new RemoteException("exportObject: Can't export object without exportData defined!");
    
    try {
       ;
      servref.exportObject((Remote)this, exportData);
    } catch (Exception e) {
      throw new RemoteException("Can't export remote object", e);
    }
     ;
  }
    
  /**
   * Calling 'unexportObject' on a NinjaRemoteObject will cause the
   * object to be unexported for future incoming calls. Currently,
   * any calls in progress (i.e., any existing client sockets connecting
   * to the remote object) are not destroyed; only future calls are
   * made impossible.
   */
  public void unexportObject() throws RemoteException {
    if (servref != null) {
      servref.unexportObject();
    } else {
      throw new RemoteException("unexportObject: Can't unexport until you have exported!");
    }
  }

  // Obtain stub for this remote object - used by server-side code only
  protected NinjaRemoteStub getStub() {
    NinjaRemoteStub stub = servref.getStub();
    if (stub == null) {
       ;
    }
    return stub;
  }
}
