// NinjaRMI, by Matt Welsh (mdw@cs.berkeley.edu)
// See http://www.cs.berkeley.edu/~mdw/proj/ninja for details

/*
 * "Copyright (c) 1998 by The Regents of the University of California
 *  All rights reserved."
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */





package ninja.rmi;

import java.net.InetAddress;


/**
 * NinjaExportData contains information used to export a NinjaRemoteObject
 * for remote access. If no NinjaExportData is specified when the
 * NinjaRemoteObject is exported, then the defaults are used.
 * @see NinjaRemoteObject
 */
public class NinjaExportData {

  /**
   * The TCP port on which to export this object. This is generally
   * used only for the RMI registry (which needs to have a well-known
   * port number). If zero, then an anonymous port is used.
   */
  public int port;

  /**
   * The IP address on which this object will listen. Only meaningful
   * if comm_type is set to NinjaExportData.RMI_COMM_TYPE_MULTICAST_ONEWAY
   * (viz).
   */
  public InetAddress ipaddr;

  /**
   * If true, then the incoming object ID is checked for each method
   * invocation. If false, then this check is not made. The idea here
   * is that the client-side stub must have a matching object ID to
   * make calls on this remote object. Since the stub is generally
   * obtained through the registry, then the object ID should match.
   * However, if the client has a "stale" stub (say, the server has
   * crashed and restarted since the stub was obtained) then calls on
   * that stub will fail.
   *
   * <p>Since use of the NinjaRegistryImpl requires generation of stubs
   * "out of thin air", check_objid is set to <tt>false</tt> for the
   * registry itself. All other objects should set check_objid to
   * <tt>true</tt>, unless they don't care for some reason.
   *
   * <p>The default value is <tt>true</tt>.
   */
  public boolean check_objid; // If false, don't worry if incoming ObjID
                              // matches (e.g., for Registry)


  /**
   * Set to one of the RMI_COMM_TYPE_* values to specify the type of
   * communication semantics this object should support.
   * XXX To be done: Allow multiple settings of this field
   * (for now you can have multiple instances of different NinjaRemoteObjects
   * which all reference the same back-end object to share data).
   */
  public int comm_type;

  /**
   * The default setting for comm_type: Specifies a reliable connection
   * (uses TCP sockets).
   */
  public static final int RMI_COMM_TYPE_RELIABLE = 0x0;

  /**
   * Specifies an unreliable connection,
   * one-way (no return value is allowed; a RemoteException is thrown if
   * the method returns a value).
   */
  public static final int RMI_COMM_TYPE_UNRELIABLE_ONEWAY = 0x1;

  /**
   * Specifies a one-way multicast connection. The server <em>must</em>
   * specify NinjaExportData.port and NinjaExportData.ipaddr to define
   * which multicast group it will listen on.
   */
  public static final int RMI_COMM_TYPE_MULTICAST_ONEWAY = 0x2;
  
  // -- Zhongwei
  public String service_name;  



  /**
   * If the user creates a NinjaServerCallbacks object and references
   * it in the NinjaExportData, then the callbacks in that object will
   * be invoked when certain server-side events occur (such as socket
   * creation and deletion). This can be used to notify the object of
   * server-side events (such as client connect and disconnect).
   */
  public NinjaServerCallbacks callbacks;

  /**
   * The no-arg constructor for NinjaExportData creates a NinjaExportData
   * object with all parameters set to the default.
   */
  public NinjaExportData() {
    port = 0;
    ipaddr = null;
    check_objid = true;
    callbacks = null;
	service_name = null;	// Zhongwei
    comm_type = RMI_COMM_TYPE_RELIABLE;
  }
}
  
