
import java.awt.*;
import java.awt.event.*;
import java.net.*;
import java.io.*;
import java.util.*;
import JavaGroups.*;


class EventHandler extends WindowAdapter {
    Frame gui;
    
    public EventHandler(Frame g) {gui=g;}

    public void windowClosing(WindowEvent e) {
	gui.dispose();
	System.exit(0);
    }
}


class Request implements java.io.Serializable {
    public static final int STOP            = 0;
    public static final int ADDITION        = 1;
    public static final int SUBTRACTION     = 2;
    public static final int MULTIPLICATION  = 3;
    public static final int DIVISION        = 4;


    public int  type=ADDITION;
    public int  x=0;
    public int  y=0;
    public int  val=0;    


    Request(int type, int x, int y, int val) {
	this.type=type; this.x=x; this.y=y; this.val=val;
    }

    public static String Type2Str(int t) {
	switch(t) {
	case STOP:           return "STOP";
	case ADDITION:       return "ADDITION";
	case SUBTRACTION:    return "SUBTRACTION";
	case MULTIPLICATION: return "MULTIPLICATION";
	case DIVISION:       return "DIVISION";
	default:             return "<unknown>";
	}
    }

    public String toString() {
	return "[" + x + "," + y + ": " + Type2Str(type) + "(" + val + ")]";
    }
}



class MyCanvas extends Canvas {
    int           field_size=100;
    int           num_fields=4;
    int           x_offset=30;
    int           y_offset=30;

    final Font    def_font=new Font("Helvetica", Font.BOLD, 14);
    int[][]       array=null;      // state

    Dimension     off_dimension=null;
    Image         off_image=null;
    Graphics      off_graphics=null;
    final Font    def_font2=new Font("Helvetica", Font.PLAIN, 12);
    final Color   checksum_col=Color.blue;
    int           checksum=0;

    
    public MyCanvas(int num_fields, int field_size, int x_offset, int y_offset) {
	this.num_fields=num_fields; this.field_size=field_size;
	this.x_offset=x_offset; this.y_offset=y_offset;

	array=new int[num_fields][num_fields];
	setBackground(Color.white);
	setSize(2*x_offset + num_fields*field_size+30, y_offset + num_fields*field_size+50);

	for(int i=0; i < num_fields; i++)
	    for(int j=0; j < num_fields; j++)
		array[i][j]=0;
    }


    public void SetFieldSize(int fs) {field_size=fs;}
    public void SetNumFields(int nf) {num_fields=nf;}
    public void SetXOffset(int o)    {x_offset=o;}
    public void SetYOffset(int o)    {y_offset=o;}


    public int AddValueTo(int x, int y, int value) {
	synchronized(array) {
	    array[x][y]+=value;
	    repaint();
	    return array[x][y];
	}
    }

    public int SubtractValueFrom(int x, int y, int value) {
	synchronized(array) {
	    array[x][y]-=value;
	    repaint();
	    return array[x][y];
	}
    }

    public int MultiplyValueWith(int x, int y, int value) {
	synchronized(array) {
	    array[x][y]*=value;
	    repaint();
	    return array[x][y];
	}
    }

    public int DivideValueBy(int x, int y, int value) {
	if(value == 0)
	    return array[x][y];
	synchronized(array) {
	    array[x][y]/=value;
	    repaint();
	    return array[x][y];
	}
    }


    public void SetValueAt(int x, int y, int value) {
	synchronized(array) {
	    array[x][y]=value;
	}
	repaint();
    }



    public int GetValueAt(int x, int y) {
	synchronized(array) {
	    return array[x][y];
	}
    }


    public void Clear() {
	synchronized(array) {
	    for(int i=0; i < num_fields; i++)
		for(int j=0; j < num_fields; j++)
		    array[i][j]=0;
	    checksum=Checksum();
	    repaint();				      
	}
    }


    public int[][] GetState() {
	synchronized(array) {
	    return array;
	}
    }


    public int[][] GetCopyOfState() {
	int[][] retval=new int[num_fields][num_fields];

	synchronized(array) {	    
	    for(int i=0; i < num_fields; i++)
		for(int j=0; j < num_fields; j++)
		    retval[i][j]=array[i][j];
	    return retval;
	}
    }


    public void Update() {
	checksum=Checksum();
	repaint();
    }


    public void SetState(Object new_state) {

	if(new_state == null)
	    return;

	try {
	    int[][] new_array=(int[][])new_state;
	    synchronized(array) {
		Clear();

		for(int i=0; i < num_fields; i++)
		    for(int j=0; j < num_fields; j++)
			array[i][j]=new_array[i][j];
		checksum=Checksum();
		repaint();
	    }
	}
	catch(Exception e) {
	    System.err.println(e);
	    return;
	}
    }


    public int Checksum() {
	int retval=0;

	synchronized(array) {
	    for(int i=0; i < num_fields; i++)
		for(int j=0; j < num_fields; j++)
		    retval+=array[i][j];
	}
	return retval;
    }



    public void update(Graphics g) {
	Dimension d=getSize();

	if(off_graphics == null || 
	   d.width != off_dimension.width || 
	   d.height != off_dimension.height) {
	    off_dimension=d;
	    off_image=createImage(d.width, d.height);
	    off_graphics=off_image.getGraphics();
	}

	//Erase the previous image.
        off_graphics.setColor(getBackground());
        off_graphics.fillRect(0, 0, d.width, d.height);
        off_graphics.setColor(Color.black);
	off_graphics.setFont(def_font);
	DrawEmptyBoard(off_graphics);
	DrawNumbers(off_graphics);
	g.drawImage(off_image, 0, 0, this);
    }




    public void paint(Graphics g) {
	update(g);
    }

    


    /** Draws the empty board, no pieces on it yet, just grid lines */
    void DrawEmptyBoard(Graphics g) {
	int   x=x_offset, y=y_offset;
	Color old_col=g.getColor();

	g.setFont(def_font2);
	old_col=g.getColor();
	g.setColor(checksum_col);
	g.drawString(("Checksum: " + checksum), x_offset + field_size, y_offset - 20);
	g.setFont(def_font);
	g.setColor(old_col);

	for(int i=0; i < num_fields; i++) {
	    for(int j=0; j < num_fields; j++) {  // draws 1 row
		g.drawRect(x, y, field_size, field_size);		    
		x+=field_size;
	    } 
	    g.drawString(("" + (num_fields-i-1)), x+20, y+field_size/2);
	    y+=field_size;
	    x=x_offset;
	}

	for(int i=0; i < num_fields; i++) {
	    g.drawString((""+i), x_offset + i*field_size + field_size/2, y+30);
	}
    }



    void DrawNumbers(Graphics g) {
	Point        p;
	String       num;
	FontMetrics  fm=g.getFontMetrics();
	int          len=0;
	
	synchronized(array) {
	    for(int i=0; i < num_fields; i++)
		for(int j=0; j < num_fields; j++) {
		    num="" + array[i][j];
		    len=fm.stringWidth(num);
		    p=Index2Coord(i, j);
		    g.drawString(num, p.x - (len/2), p.y);
		}
	}
    }



    
    Point Coord2Index(int x, int y) {
	Point ret=new Point();
	
	ret.x=x_offset + (x * field_size);
	ret.y=y_offset + ((num_fields - 1 - y) * field_size);
	return ret;
    }
    

    Point Index2Coord(int i, int j) {
	int x=x_offset + i*field_size + field_size/2;

	// int y=y_offset + j*field_size + field_size/2;

	int y=y_offset + num_fields*field_size - j*field_size - field_size/2;
	
	return new Point(x, y);
    }

    
}














public class TotalOrderPush extends Frame implements MessageListener {
    final Font       def_font=new Font("Helvetica", Font.BOLD, 14);
    final Font       def_font2=new Font("Helvetica", Font.PLAIN, 12);
    MyCanvas         canvas;
    MenuBar          menubar=CreateMenuBar();
    Button           start=new Button("Start"), stop=new Button("Stop");
    Button           clear=new Button("Clear"), get_state=new Button("Get State");
    Button           quit=new Button("Quit");
    Panel            button_panel=new Panel();
    Thread           sender=null;
    Channel          channel;
    Dialog           error_dlg;
    long             timeout=0;
    int              field_size=0;
    int              num_fields=0;
    final int        x_offset=30;
    final int        y_offset=40;
    PullPushAdapter  pull=null;



    class SenderThread extends Thread {
	Request req;

	public void run() {
	    setName("SenderThread");

	    while(true) {
		try {
		    req=CreateRandomRequest();
		    channel.Send(new Message(null, null, Util.ObjectToByteBuffer(req)));
		    Util.Sleep(timeout);
		}
		catch(Exception e) {
		    Error(e.toString());
		    return;
		}
	    }
	}
    }



    void ProcessRequest(Request req) throws Exception {
	int x=req.x, y=req.y, val=req.val, result=0;

	if(req.type == Request.STOP) {
	    StopSender();
	    return;
	}
	
	switch(req.type) {
	case Request.ADDITION:
	    result=canvas.AddValueTo(x, y, val);
	    break;
	case Request.SUBTRACTION:
	    result=canvas.SubtractValueFrom(x, y, val);
	    break;
	case Request.MULTIPLICATION:
	    result=canvas.MultiplyValueWith(x, y, val);
	    break;
	case Request.DIVISION:
	    result=canvas.DivideValueBy(x, y, val);
	    break;
	}	
	canvas.Update();
    }


    
    public TotalOrderPush(String title, long timeout, int num_fields, int field_size, boolean ens) {
	Dimension s;

	this.timeout=timeout; this.num_fields=num_fields; this.field_size=field_size;
	setFont(def_font);

	try {
	    if(ens)
		channel=new EnsChannel();
	    else
		channel=new JChannel("UDP:DELAY(out_delay=100):PING:FD:GMS:STATE_TRANSFER");
	    channel.SetOpt(Channel.GET_STATE_EVENTS, new Boolean(true));
	    channel.Connect("TotalOrderGroup");
	}
	catch(Exception e) {
	    Error(e.toString());
	    System.exit(-1);
	}

	start.addActionListener(
				new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		StartSender();
	    }	    
	});

	stop.addActionListener(
				new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		try {
		    channel.Send(new Message(null, null, 
					     Util.ObjectToByteBuffer(new Request(Request.STOP, 
										 0, 0, 0))));
		}
		catch(Exception ex) {
		}
		
	    }	    
	});

	clear.addActionListener(
				new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		canvas.Clear();
	    }	    
	});



	get_state.addActionListener(
				    new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		boolean rc=channel.GetState(3000);
		if(rc == false)
		    Error("State could not be retrieved !");
	    }
	});


	quit.addActionListener(
				new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		channel.Disconnect();
		channel.Close();
		System.exit(0);
	    }	    
	});


	setTitle(title);
	addWindowListener(new EventHandler(this));
	setBackground(Color.white);
	setMenuBar(menubar);

	setLayout(new BorderLayout());
	canvas=new MyCanvas(num_fields, field_size, x_offset, y_offset);

	add("Center", canvas);
	button_panel.setLayout(new FlowLayout());
	button_panel.setFont(def_font2);
	button_panel.add(start);
	button_panel.add(stop);
	button_panel.add(clear);
	button_panel.add(get_state);
	button_panel.add(quit);
	add("South", button_panel);

	s=canvas.getSize();
	s.height+=100;
	setSize(s);

	pull=new PullPushAdapter(channel);
	pull.SetListener(this);
	boolean rc=channel.GetState(0);
    }



    public void Receive(Message msg) {
	try {
	    Request req=(Request)Util.ObjectFromByteBuffer(msg.GetBuffer());
	    ProcessRequest(req);
	}
	catch(Exception e) {
	    System.err.println(e);
	}	
    }


    
    public Serializable GetState() {
	int[][] copy_of_state=(int[][])canvas.GetCopyOfState();
	return copy_of_state;
    }



    public void SetState(Serializable state) {
	if(state != null)
	    canvas.SetState(state);
    }



    void StartSender() {
	if(sender == null) {
	    sender=new SenderThread();
	    sender.start();
	}
    }

    void StopSender() {
	if(sender != null) {
	    sender.stop();
	    sender=null;
	}
    }

    

    MenuBar CreateMenuBar() {
	MenuBar   ret=new MenuBar();
	Menu      file=new Menu("File");
	MenuItem  quit=new MenuItem("Quit");

	ret.setFont(def_font2);
	ret.add(file);

	file.addSeparator();
	file.add(quit);


	quit.addActionListener(				      
			       new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		System.exit(1);
	    }});
	return ret;
    }


    public void Error(String msg) {
	Button  ok=new Button("Ok");
	Label   l=new Label(msg);

	error_dlg=new Dialog(this, msg, true);
	error_dlg.setLocation(90, 150);
	error_dlg.setSize(420, 100);
	error_dlg.setLayout(new BorderLayout());
	error_dlg.setFont(def_font2);

	ok.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		error_dlg.dispose();
	    }
	});

	error_dlg.add("Center", l);
	error_dlg.add("South", ok);
	error_dlg.show();
    }



    Request CreateRandomRequest() {
	Request ret=null;
	int     op_type=(int)(((Math.random() * 10) % 4)+1);  // 1 - 4
	int     x=(int)((Math.random() * num_fields * 2) % num_fields);
	int     y=(int)((Math.random() * num_fields * 2) % num_fields);
	int     val=(int)((Math.random() * num_fields * 200) % 10);

	ret=new Request(op_type, x, y, val);
	return ret;
    }




    public static void main(String[] args) {
	TotalOrderPush  g;
	String          arg;
	long            timeout=1000;
	int             num_fields=4;
	int             field_size=80;
	boolean         use_ensemble=false;

	for(int i=0; i < args.length; i++) {
	    arg=args[i];
	    if(arg.equals("-timeout")) {
		timeout=new Long(args[++i]).longValue();
		continue;
	    }
	    if(arg.equals("-num_fields")) {
		num_fields=new Integer(args[++i]).intValue();
		continue;
	    }
	    if(arg.equals("-field_size")) {
		field_size=new Integer(args[++i]).intValue();
		continue;
	    }
	    if(arg.equals("-ens")) {
		use_ensemble=true;
		continue;
	    }
	    if(arg.equals("-help")) {
		System.out.println("\nTotalOrderPush [-timeout <value>] [-num_fields <value>]"+
				   "[-field_size <value>]\n");
		return;
	    }
	}


	try {
	    g=new TotalOrderPush("Total Order Demo on " + InetAddress.getLocalHost().getHostName(), 
				 timeout, num_fields, field_size, use_ensemble);
	    g.show();	    
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }


  
}

