package JavaGroups.JavaStack.Protocols;


import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;





class RpcGmsHeader implements java.io.Serializable {
    public ViewId   view_id=null;
    public boolean  handle=false;
    
    public RpcGmsHeader(ViewId v, boolean h) {
	view_id=v;
	handle=h;
    }
    
    public String toString() {
	return "[RpcGMS: view_id=" + view_id + ", handle=" + handle + "]";
    }
}






/**

 */
public class RpcGMS extends RpcProtocol implements Runnable {
    private RpcGmsImpl          impl=RpcClientGmsImpl.CreateInstance(this);
    public  Properties          props=null;
    public  Address             local_addr=null;
    public  String              group_addr=null;
    public  Membership          members=new Membership();
    public  ViewId              view_id=null;
    public  long                ltime=0; 
    public  long                initial_mbrs_timeout=5000;
    public  long                join_timeout=5000;
    public  long                join_retry_timeout=2000;
    public  long                leave_timeout=2000;
    public  boolean             trace=false;
    public  Integer             impl_mutex=new Integer(0);  // synchronizes event entry into impl
    public  Integer             view_mutex=new Integer(0);  // synchronizes view installations
    private Queue               event_queue=new Queue();    // stores SUSPECT, MERGE events
    private Thread              evt_thread=null;



    public String   GetName()          {return "RpcGMS";}


    public void SetImpl(RpcGmsImpl new_impl) {
	synchronized(impl_mutex) {
	    impl.Deactivate();
	    new_impl.Activate();
	    impl=new_impl;
	    System.out.println("Changed role to " + new_impl.getClass().getName());
	}
    }

    /* ---------------------------------- Transport protocol ----------------------------- */
    public void Send(Message msg) throws Exception {
	Address dest=(Address)msg.GetDest();
	
	if(dest == null || dest.IsMulticastAddress()) {       // group multicast
	    msg.AddHeader(new RpcGmsHeader(view_id, true));
	    //System.out.println("RpcGMS.Send(): view_id is " + view_id);
	}
	else
	    msg.AddHeader(new RpcGmsHeader(view_id, false));  // don't bother with unicast msgs

	PassDown(new Event(Event.MSG, msg));
    }
    /* ------------------------------- End of Transport protocol ------------------------- */


    /**
       Computes the next view. Returns a copy that has <code>suspected_mbrs</code> removed
       and <code>new_mbrs</code> added.
     */
    public View GetNextView(Vector new_mbrs, Vector suspected_mbrs) {
	Vector      mbrs;
	long        vid=0;
	View        v;
	Membership  tmp_mbrs;

	synchronized(view_mutex) {
	    vid=Math.max(view_id.GetId(), ltime) + 1;
	    ltime=vid;
	    tmp_mbrs=members.Copy();
	    tmp_mbrs.Merge(new_mbrs, suspected_mbrs);
	    mbrs=(Vector)tmp_mbrs.GetMembers().clone();	    
	    v=new View(local_addr, vid, mbrs);
	    return v;
	}
    }




    /**
       Run view update protocol to install a new view in all members (this involves casting the new
       view to all members).
     */
    public synchronized void CastViewChange(View v) {
	ViewId tmp_id;
	Rsp    rsp;

	if(v != null) {
	    tmp_id=new ViewId((Address)v.GetCreator(), v.GetId());

	    System.out.println("==> 1. Casting VIEW_CHANGE(" + tmp_id + ")");

	    RspList rsp_list=CallRemoteMethods(v.GetMembers(), "HandleViewChange",
					       tmp_id, v.GetMembers(), MessageProtocol.ALL, 0);
					       //tmp_id, v.GetMembers(), MessageProtocol.ALL, 6000);

	    System.out.println("DONE, responses are:");
	    for(int i=0; i < rsp_list.size(); i++) {
		rsp=(Rsp)rsp_list.elementAt(i);
		System.out.println(rsp);
	    }

	    return;
	}

	synchronized(view_mutex) {
	    if(members.size() == 0) {
		System.err.println("RpcGMS.CastViewChange(): am not going to cast empty view !");
		return;
	    }
	    v=GetNextView(null, null); // no new or suspected members; just create a new view
	    tmp_id=new ViewId((Address)v.GetCreator(), v.GetId());
	}

	System.out.println("==> 2. Casting VIEW_CHANGE(" + tmp_id + ")");
	CallRemoteMethods(members.GetMembers(), "HandleViewChange", tmp_id, 
			  members.GetMembers(), MessageProtocol.ALL, 0);
    }



    



    /**
       Assigns the new ltime. Installs view and view_id. Changes role to coordinator if necessary.
       Sends VIEW_CHANGE event up and down the stack.
     */
    public void InstallView(ViewId new_view, Vector mbrs) {
	Object coord;
	int    rc;

	synchronized(view_mutex) {                    // serialize access to views
	    ltime=Math.max(new_view.GetId(), ltime);  // compute Lamport's logical time
	    System.out.println("==> Received VIEW_CHANGE(" + new_view + ")");

	    /* Check for self-inclusion: if I'm not part of the new membership, I just discard it.
	       This ensures that messages sent in view V1 are only received by members of V1 */
	    if(CheckSelfInclusion(mbrs) == false) {
		System.err.println("RpcGMS.InstallView(): I'm not member of " + mbrs + ", discarding");
		return;
	    }

	    if(view_id != null && new_view != null) {
		rc=new_view.Compare(view_id);  // rc should always be a positive number
		if(rc <= 0) {  // don't accept view id lower than our own
		    Util.Print("**** ViewChange, rc is " + rc + ": received view <= current " +
			       "view; discarding it ! ***");
		    Util.Print("view_id: " + view_id + "\nnew_view: " + new_view);
		    return;
		}
	    }
	    else {
		System.err.println("RpcGMS.InstallView(): view_id was null ! Setting it to an initial " +
				   "value from " + new_view);
		view_id=new_view;
	    }

	    if(new_view.GetCoordAddress() != null)
		view_id.SetCoordAddress(new_view.GetCoordAddress());

	    view_id.SetId(new_view.GetId());

	    if(mbrs != null && mbrs.size() > 0)
		members.Set(mbrs);

	    // Send VIEW_CHANGE event up and down the stack:	    
	    Event view_event=new Event(Event.VIEW_CHANGE, MakeView(members.GetMembers()));
	    PassUp(view_event);
	    PassDown(view_event); // needed e.g. by failure detector or UDP

	    coord=DetermineCoordinator();
	    if(coord != null && coord.equals(local_addr) && !(coord.equals(new_view.GetCoordAddress())))
		SetImpl(RpcCoordGmsImpl.CreateInstance(this));
	}
    }



    protected Address DetermineCoordinator() {
	synchronized(members) {
	    return members != null && members.size() > 0 ? (Address)members.elementAt(0) : null;
	}
    }



    /** Returns true if local_addr is member of mbrs, else false */
    protected boolean CheckSelfInclusion(Vector mbrs) {
	Object mbr;
	if(mbrs == null)
	    return false;
	for(int i=0; i < mbrs.size(); i++) {
	    mbr=mbrs.elementAt(i);
	    if(mbr != null && local_addr.equals(mbr))
		return true;
	}
	return false;
    }


    public View MakeView(Vector mbrs) {
	Object  coord=null;
	long    id=0;

	if(view_id != null) {
	    coord=view_id.GetCoordAddress();
	    id=view_id.GetId();
	}
	return new View(coord, id, mbrs);
    }


    public View MakeView(Vector mbrs, ViewId vid) {
  	Object  coord=null;
  	long    id=0;
	
  	if(vid != null) {
  	    coord=vid.GetCoordAddress();
  	    id=vid.GetId();
  	}
  	return new View(coord, id, mbrs);
    }




    /* ------------------------- Request handler methods ----------------------------- */

    public void Join(Address mbr) {
	synchronized(impl_mutex) {
	    impl.Join(mbr);
	}
    }

    public void Leave(Address mbr) {
	synchronized(impl_mutex) {
	    impl.Leave(mbr);
	}
    }

    public void Suspect(Address mbr) {
	synchronized(impl_mutex) {
	    impl.Suspect(mbr);
	}
    }

    public void Merge(Vector new_mbrs) {
	synchronized(impl_mutex) {
	    impl.Merge(new_mbrs);
	}
    }

    public JoinRsp HandleJoin(Address mbr) {
	return impl.HandleJoin(mbr);	
    }

    public boolean HandleLeave(Address mbr) {
	return impl.HandleLeave(mbr);	
    }

    public void HandleViewChange(ViewId new_view, Vector mbrs) {
	impl.HandleViewChange(new_view, mbrs);	
    }

    public void HandleMerge(Vector new_mbrs, Vector suspects, long other_ltime) {
	impl.HandleMerge(new_mbrs, suspects, other_ltime);	
    }

    public void HandleSuspect(Address mbr) {
	impl.HandleSuspect(mbr);	
    }


    public void CastView(View v) {
	impl.CastView(v);
    }

    /* --------------------- End of Request handler methods -------------------------- */




    
    /**
       <b>Callback</b>. Called by superclass when event may be handled.<p>
       <b>Do not use <code>PassUp</code> in this method as the event is passed up
       by default by the superclass after this method returns !</b>
       @return boolean Defaults to true. If false, event will not be passed up the stack.
     */
    public boolean HandleUpEvent(Event evt) {
	RpcGmsHeader  hdr;
	ViewId        vid;
	Object        obj, sender;
	Message       m;


	switch(evt.GetType()) {

  	case Event.MSG:

	    m=(Message)evt.GetArg();
	    obj=m.PeekHeader();
	    if(obj != null && obj instanceof RpcGmsHeader) {
		hdr=(RpcGmsHeader)m.RemoveHeader();
		
		if(hdr.handle == false) { // unicast messages are not checked for ViewIds
		    return true;
		}


		vid=hdr.view_id;
		// +++ implement +++
		// 1. Check view id. If same as our current view -> return true, else return false. 
		//    If false is returned, the msg will be discarded (not passed up the stack)
		// 2. Check whether sender is in our view (*only for mcast messages* !).
		//    If not -> return false, else true

		
		sender=m.GetSrc();

		// Clients accept all messages, coordinator and participants only those from members
		// in their current view. Client are determined by their null membership
		if(members.size() > 0) { //if not a client, drop msgs from non-members
		    if(!members.Contains((Address)sender)) {
			System.out.println("RpcGMS.HandleUpEvent(MSG): mbrship " + members + 
					   " does not contain " + sender + "; message was dropped !");
			return false;
		    }
		}

		//System.out.println("Received msg with VID " + vid);
	    }

  	    return true;
	    


	case Event.SET_LOCAL_ADDRESS:
	    local_addr=(Address)evt.GetArg();
	
	    Util.Print("\n----------------------------------------------------------\n" +
		       "RpcGMS: my address is " + local_addr + 
		       "\n----------------------------------------------------------");
	    
	    return true;                         // pass up

	case Event.SUSPECT:
	    PassUp(evt);
	    try {event_queue.Add(evt);}
	    catch(Exception e) {}
	    return false;                        // don't pass up: was already passed up

	case Event.MERGE:
	    try {event_queue.Add(evt);}
	    catch(Exception e) {}
	    return false;                        // don't pass up
	}

	return impl.HandleUpEvent(evt);
    }




    /**
       <b>Callback</b>. Called by superclass when event may be handled.<p>
       <b>Do not use <code>PassDown</code> in this method as the event is passed down
       by default by the superclass after this method returns !</b>
       @return boolean Defaults to true. If false, event will not be passed down the stack.
    */
    public boolean HandleDownEvent(Event evt) {	
	Address dest;
	Message msg;

	switch(evt.GetType()) {


	case Event.MSG:

	    // +++ implement +++
	    // Add current view id, will be checked by HandleUpEvent()
	    msg=(Message)evt.GetArg();
	    dest=(Address)msg.GetDest();
	    
	    if(dest == null || dest.IsMulticastAddress())
		msg.AddHeader(new RpcGmsHeader(view_id, true));
	    else
		msg.AddHeader(new RpcGmsHeader(view_id, false));  // don't bother with unicast msgs
	    return true;

		
	case Event.CONNECT:
	    try {
		group_addr=(String)evt.GetArg();
	    }
	    catch(ClassCastException cce) {
		System.err.println("RpcGMS.HandleDownEvent(CONNECT): group address must " +
				   "be a string (group name) to make sense");
	    }
	    PassDown(evt);
	    impl.Join(local_addr);
	    PassUp(new Event(Event.CONNECT_OK));
	    StartEventHandlerThread();
	    return false;                        // don't pass down: was already passed down
		
	case Event.DISCONNECT:
	    impl.Leave((Address)evt.GetArg());
	    PassUp(new Event(Event.DISCONNECT_OK));
	    StopEventHandlerThread();
	    return true;                         // pass down
	}	

	return impl.HandleDownEvent(evt);
    }



    /** Setup the Protocol instance acording to the configuration string */
    public boolean SetProperties(Properties props) {
	String     str;
	this.props=props;

	str=props.getProperty("trace");
	if(str != null) {
	    trace=new Boolean(str).booleanValue();
	    props.remove("trace");
	}

 	str=props.getProperty("initial_mbrs_timeout");   // time to wait for initial member retrieval
 	if(str != null) {
 	    initial_mbrs_timeout=new Long(str).longValue();
	    props.remove("initial_mbrs_timeout");
	}

 	str=props.getProperty("join_timeout");           // time to wait for JOIN
 	if(str != null) {
 	    join_timeout=new Long(str).longValue();
	    props.remove("join_timeout");
	}

 	str=props.getProperty("join_retry_timeout");     // time to wait between JOINs
 	if(str != null) {
 	    join_retry_timeout=new Long(str).longValue();
	    props.remove("join_retry_timeout");
	}

 	str=props.getProperty("leave_timeout");           // time to wait until coord responds to LEAVE req.
 	if(str != null) {
 	    leave_timeout=new Long(str).longValue();
	    props.remove("leave_timeout");
	}

	if(props.size() > 0) {
	    System.err.println("RpcGMS.SetProperties(): the following properties are not recognized:");
	    props.list(System.out);
	    return false;
	}
	return true;
    }



    public void run() {
	Event evt;

	while(true) {
	    try {
		evt=(Event)event_queue.Remove();
		switch(evt.GetType()) {
		case Event.SUSPECT:
		    impl.Suspect((Address)evt.GetArg());
		    break;
		case Event.MERGE:
		    impl.Merge((Vector)evt.GetArg());
		    break;
		default:
		    System.out.println("RpcGMS.run(): event handler thread encountered event of type " +
				       Event.Type2String(evt.GetType()) + ": not handled by me !");
		    break;
		}
	    }
	    catch(QueueClosed closed) {
		break;
	    }
	}
    }



    /* ------------------------------- Private Methods --------------------------------- */

    private void StartEventHandlerThread() {
	if(evt_thread == null) {
	    evt_thread=new Thread(this, "RpcGMS.EventHandlerThread");
	    evt_thread.start();
	}
    }


    private void StopEventHandlerThread() {
	if(evt_thread != null) {
	    evt_thread.stop();
	    evt_thread=null;
	    event_queue.Reset();
	}
    }


}
