package JavaGroups.JavaStack.Protocols;

import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;



/**
   Finite state machine for point-to-point connections between 2 peers (on top of a connection-less
   transport). Provides lossless, FIFO message reception plus flow control. Implemented using the 
   State pattern (Gamma et al.): each state of a connection is modeled as a separate class, when a 
   state change occurs, the <code>state</code> variable is set to a different instance. Each event
   is delegated to the state instance, which acts differently depending on its class.<p>
   In the ESTABLISHED state, sliding windows are used to provide lossless FIFO, flow-controlled
   delivery. Depending on the protocol configuration, either an ACK or NAK based scheme is used.<p>
   PT2PT acts like a UDP channel (but with retransmission and timeouts). However, when a peer is not
   up when sending data, this is only reported as a <code>SUSPECT</code> event (<code>Send</code> is
   asynchronous and therefore cannot throw an exception). In multicast mode, that member will be 
   excluded from the group.
 */

public class Pt2ptFsm {




    /** Sends a SYN with a proposed sequence number to a peer */
    class SendSynCommand implements Command {
	Object dest;
	long   proposed_seqno;

	SendSynCommand(Object dest, long proposed_seqno) {
	    this.dest=dest; this.proposed_seqno=proposed_seqno;
	}

	public boolean Execute() {
	    Message      syn=new Message(dest, null, null);
	    Pt2ptHeader  hdr=new Pt2ptHeader(Pt2ptHeader.SYN, proposed_seqno);
	    
	    syn.AddHeader(hdr);
	    protocol.PassDown(new Event(Event.MSG, syn));
	    return true;
	}
    }


    /** Sends back an ACK and a SYN in one packet */
    class SendAckSynCommand implements Command {
	Object dest;
	long   ack_no=0;
	long   proposed_seqno=0;

	public SendAckSynCommand(Object dest, long ack_no, long proposed_seqno) {
	    this.dest=dest; this.ack_no=ack_no; this.proposed_seqno=proposed_seqno;
	}

	public boolean Execute() {
	    Message      ack_syn_msg=new Message(dest, null, null);
	    Pt2ptHeader  hdr=new Pt2ptHeader(Pt2ptHeader.ACK_SYN, ack_no, proposed_seqno);

	    System.out.println("SendAckSyn: hdr is " + hdr);
	    
	    ack_syn_msg.AddHeader(hdr);
	    protocol.PassDown(new Event(Event.MSG, ack_syn_msg));
	    return true;
	}
    }



    /** Reset the state to CLOSED */
    class ResetCommand implements Command {

	public boolean Execute() {
	    synchronized(msg_queue) {msg_queue.removeAllElements();}
	    SetState(new Closed());
	    connection_ex=new ChannelException("Connection to peer refused");
	    synchronized(connect_mutex) {
		connect_mutex.notifyAll();
	    }
	    return true;
	}

    }



    class RetransmissionThread extends Thread {
	Command  retransmit_command, abort_command=null;
	long     timeout=2000;
	int      num_tries=0;  // 0 == forever

	public RetransmissionThread(Command c, long timeout, int num_tries, Command abort_command) {
	    retransmit_command=c; this.abort_command=abort_command;
	    this.timeout=timeout; this.num_tries=num_tries;
	    setName("Pt2ptFsm.RetransmissionThread");
	}

	public void run() {
	    if(num_tries == 0) {
		while(true) {
		    try {
			sleep(timeout);
			if(retransmit_command != null)
			    retransmit_command.Execute();
		    }
		    catch(Exception e) {
		    }
		}
	    }
	    else {
		while(num_tries > 0) {
		    try {
			System.out.println("Sleep(" + timeout + ")");
			sleep(timeout);
			if(retransmit_command != null)
			    retransmit_command.Execute();
		    }
		    catch(Exception e) {
		    }
		    num_tries--;
		}
		if(abort_command != null)
		    abort_command.Execute();
	    }
	}
    }





    
    interface State {
	public void Down(Message msg);
	public void Up(Pt2ptHeader hdr, Message msg);
    }




    

    /** -----------------------------------------------------------------------------
	                             CLOSED state
	-----------------------------------------------------------------------------
    */
    class Closed implements State {

	private void Connect(Object dest) throws ChannelException {
	    own_seq=GetInitialSeqNo();
	    
	    System.out.println("(" + this.hashCode() + ") --> SYN(" + own_seq + ")");
	    
	    send_syn_command=new SendSynCommand(dest, own_seq);
	    send_syn_command.Execute();
	    
	    SetState(new SynSent());

	    synchronized(connect_mutex) {
		try {
		    connect_mutex.wait();
		}
		catch(Exception e) {}
		if(connection_ex != null) {
		    ChannelException new_ex=connection_ex;
		    connection_ex=null;  // reset exception
		    throw new_ex;
		}
	    }
	}
	
	

	/** 
	    Queue msg (outgoing). Send a SYN message with a proposed seq_no. 
	    Change state to SynSent.
	 */
	public void Down(Message msg) {
	    Message      syn;
	    Pt2ptHeader  hdr;

	    synchronized(msg_queue) {
		msg_queue.addElement(msg);
	    }

	    try {
		Connect(msg.GetDest());
	    }
	    catch(Exception e) {
		protocol.PassUp(new Event(Event.SUSPECT, msg.GetDest()));
		System.err.println(e);
	    }
	}



	/**
	   When SYN request is received: send back ACK_SYN message (ack plus own SYN request.
	 */
	public void Up(Pt2ptHeader hdr, Message msg) {
	    if(hdr.type != Pt2ptHeader.SYN) {
		System.err.println("Pt2ptFsm.SynSent.Up(): expected SYN, but received " +
				   hdr.type + "(" + Pt2ptHeader.Type2Str(hdr.type) + ")");
		return;		
	    }

	    /* Get peer's sequence number */
	    peer_seq=hdr.syn_seq;
	    System.out.println("(" + this.hashCode() + ") <-- SYN(" + peer_seq + ")");

	    System.out.println("own_seq (initial)=" + own_seq);

	    /* Compute own's sequence number */
	    if(own_seq == 0) {
		own_seq=GetInitialSeqNo();
		System.out.println("own_seq=" + own_seq);
	    }
	    System.out.println("(" + this.hashCode() + ") --> ACK_SYN(" + peer_seq + ", " + 
			       own_seq + ")");   
	    send_ack_syn_command=new SendAckSynCommand(msg.GetSrc(), peer_seq, own_seq);
	    send_ack_syn_command.Execute();

	    SetState(new SynReceived());
	}


	public String toString() {return "CLOSED";}

    }




    /** ------------------------------------------------------------------------------------------
       Initial SYN has been sent. Waits until an ACK_SYN is received. Retransmits SYN periodically 
       until ACK_SYN is received. When ACK_SYN is received, retransmission is stopped, and state
       set to ESTABLISHED.
       -------------------------------------------------------------------------------------------
     */
    class SynSent implements State {
	RetransmissionThread r=null;

	public SynSent() {
	    if(r == null) {
		r=new RetransmissionThread(send_syn_command, retransmission_timeout, 
					   3, new ResetCommand());
		r.start();
	    }
	}



	public void Down(Message msg) {
	    if(state instanceof Closed) {
		System.err.println("Down(): state is closed; msg discarded");
		return;
	    }

	    synchronized(msg_queue) {
		if(state instanceof Established && msg_queue.size() == 0) {
		    state.Down(msg);
		    return;
		}
		System.out.println("msg_queue.addElement()");
		msg_queue.addElement(msg);
	    }
	}


	/** 
	    Receive ACK_SYN. Send ACK
	*/
	public void Up(Pt2ptHeader hdr, Message msg) {
	    Message      ack_syn;
	    Pt2ptHeader  down_hdr;

	    if(hdr.type != Pt2ptHeader.ACK_SYN) {
		System.err.println("Pt2ptFsm.SynSent.Up(): expected ACK_SYN, but received " +
				   hdr.type + "(" + Pt2ptHeader.Type2Str(hdr.type) + ")");
		return;
	    }

	    /* check whether our seq_no is ack'ed correctly */
	    if(hdr.ack_seq != own_seq) {
		System.err.println("Pt2ptFsm.SynSent.Up(): proposed sequence number " +
				   own_seq + " was not ack'ed correctly (ack_seq=" +
				   hdr.ack_seq + ")");
		return;
	    }


	    System.out.println("(" + this.hashCode() + ") <-- ACK_SYN(" + hdr.ack_seq + ", " +
			       hdr.syn_seq + ")");

	    /* get peer's seq_no */
	    peer_seq=hdr.syn_seq;
	    


	    /* Send ACK */
	    ack_syn=new Message(msg.GetSrc(), null, null);  // empty message
	    down_hdr=new Pt2ptHeader(Pt2ptHeader.ACK, peer_seq);
	    ack_syn.AddHeader(down_hdr);

	    System.out.println("(" + this.hashCode() + ") --> ACK(" + peer_seq + ")");

	    protocol.PassDown(new Event(Event.MSG, ack_syn));
	    
	    if(r != null) {  // stop retransmission of SYN
		r.stop();
		r=null;
	    }
	    SetState(new Established());
	}

	public String toString() {return "SYN_SENT";}
    }





    /** -----------------------------------------------------------------------------
	                            SYN_RECEIVED state
	-----------------------------------------------------------------------------
    */
    class SynReceived implements State {
	public void Down(Message msg) {
	    System.err.println("Pt2ptFsm.SynReceived.Down(): no message can be sent " +
			       "in this state; msg is " + msg);	    
	}





	/**
	   If SYN: retransmit ACK_SYN
	   If ACK: goto ESTABLISHED
	 */
	public void Up(Pt2ptHeader hdr, Message msg) {
	    switch(hdr.type) {
	    case Pt2ptHeader.SYN:   // retransmit request, peer did apparently not receive ACK_SYN

		System.out.println("(" + this.hashCode() + ") <-- SYN(" + hdr.syn_seq + ")");

		send_ack_syn_command.Execute();
		break;
	    case Pt2ptHeader.ACK:

		System.out.println("(" + this.hashCode() + ") <-- ACK(" + hdr.ack_seq + ")");

		SetState(new Established());
		break;
	    default:
		System.err.println("Pt2ptFsm.SynReceived.Up(): received unexpected header type " +
				   Pt2ptHeader.Type2Str(hdr.type));
		break;
	    }
	}

	public String toString() {return "SYN_RECEIVED";}

    }





    /** -----------------------------------------------------------------------------
	                           ESTABLISHED state
	-----------------------------------------------------------------------------  */
    class Established implements State {
	Retransmitter retransmitter=new Retransmitter();

	class Retransmitter implements AckSenderWindow.RetransmitCommand {

	    public void Retransmit(long seq_no, Message msg, int num_tries) {
		// msg already has header

		if(num_tries > 3) {
		    System.out.println("Pt2ptFsm.Established.Down(): msg has been retransmitted " +
				       num_tries + " times; closing connection to " +
				       msg.GetDest());
		    System.out.println("Sending SUSPECT msg");
		    protocol.PassUp(new Event(Event.SUSPECT, msg.GetDest()));
		    SetState(new Closed());
		}
		else		    
		    protocol.PassDown(new Event(Event.MSG, msg));
	    }
	    
	}



	/** Send/deliver messages in the queues */
	public Established() {
	    Message      m;
	    Pt2ptHeader  hdr;	    

	    ack_receiver=new AckReceiverWindow(peer_seq);
	    ack_sender=new AckSenderWindow(retransmitter, own_seq);
	    ack_sender.Start();

	    synchronized(msg_queue) {

		System.out.println("Sending " + msg_queue.size() + " messages");

		for(int i=0; i < msg_queue.size(); i++) {
		    m=(Message)msg_queue.elementAt(i);
		    Down(m);
		}
		msg_queue.removeAllElements();
	    }


	    connection_ex=null;
	    synchronized(connect_mutex) {
		connect_mutex.notifyAll();   // wakes up Connect()
	    }

	}




	public void Down(Message msg) {
	    Pt2ptHeader hdr=new Pt2ptHeader(Pt2ptHeader.DATA, own_seq);
	    
	    System.out.println("(" + this.hashCode() + ") --> DATA(" + hdr.syn_seq + ")");

	    msg.AddHeader(hdr);
	    ack_sender.Add(own_seq, msg);
	    own_seq++;
	    protocol.PassDown(new Event(Event.MSG, msg));
	}



	public void Up(Pt2ptHeader hdr, Message msg) {
	    Message up_msg;


	    if(hdr.type == Pt2ptHeader.DATA_ACK) {
		System.out.println("(" + this.hashCode() + ") <-- DATA_ACK(" + hdr.ack_seq + ")");
		ack_sender.Ack(hdr.ack_seq);         // advances sender window
	    }
	    else if(hdr.type == Pt2ptHeader.DATA) {
		System.out.println("(" + this.hashCode() + ") <-- DATA(" + hdr.ack_seq + ")");
		ack_receiver.Add(hdr.syn_seq, msg);

		// Send DATA_ACK:
		Message data_ack=new Message(msg.GetSrc(), null, null);
		data_ack.AddHeader(new Pt2ptHeader(Pt2ptHeader.DATA_ACK, hdr.syn_seq));
		System.out.println("(" + this.hashCode() + ") --> DATA_ACK(" + hdr.syn_seq + ")");
		protocol.PassDown(new Event(Event.MSG, data_ack));

		while((up_msg=ack_receiver.Remove()) != null)  // this ensures FIFO, lossless del.
		    protocol.PassUp(new Event(Event.MSG, up_msg));
	    }
	    else
		System.err.println("Pt2ptFsm.Established.Up(): received " + 
				   Pt2ptHeader.Type2Str(hdr.type) + " (" + hdr.type + 
				   " ); does not understand");
	}



	public String toString() {return "ESTABLISHED";}
    }











    /** Returns random initial sequence number between 1 and 100 */
    long GetInitialSeqNo() {
	return Math.abs(rand.nextLong() % 99) + 1;
    }






    long                  own_seq=0;
    long                  peer_seq=0;
    long                  retransmission_timeout=2000; // 2 secs before transmission
    Vector                msg_queue=new Vector();
    State                 state=new Closed();          // initial state
    Protocol              protocol=null;
    Random                rand=new Random(this.hashCode());
    SendSynCommand        send_syn_command=null;
    SendAckSynCommand     send_ack_syn_command=null;
    AckSenderWindow       ack_sender=null;
    AckReceiverWindow     ack_receiver=null;
    Integer               connect_mutex=new Integer(0);
    ChannelException      connection_ex=null;



    public Pt2ptFsm(Protocol prot) {protocol=prot;}
    
    
    void SetState(State new_state) {
	synchronized(state) {
	    System.out.println("Pt2ptFsm.SetState(): changing state from " + state + " to " +
			       new_state);
	    state=new_state;
	}
    }

    void Destroy() {
	if(ack_sender != null) {
	    ack_sender.Stop();
	    ack_sender=null;
	}
    }



    
    /** Add a header ! */
    void Down(Message msg) {
	state.Down(msg);
    }
    
    
    /** Remove the header ! */
    void Up(Pt2ptHeader hdr, Message msg) {
	state.Up(hdr, msg);
    }
    

   
}
