package JavaGroups.JavaStack.Protocols;

import java.io.*;
import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;




class PingHeader implements Serializable {
    public static final int GET_MBRS_REQ=1;   // arg = null
    public static final int GET_MBRS_RSP=2;   // arg = PingRsp(local_addr, coord_addr)

    public int    type=0;
    public Object arg=null;

    public PingHeader(int type, Object arg) {
	this.type=type; this.arg=arg;
    }

    public String toString() {
	return "[PING: type=" + Type2Str(type) + ", arg=" + arg + "]";
    }

    String Type2Str(int t) {
	switch(t) {
	case GET_MBRS_REQ: return "GET_MBRS_REQ";
	case GET_MBRS_RSP: return "GET_MBRS_RSP";
	default:           return "<unkown type (" + t + ")>";
	}
    }
}




/**
   The PING protocol layer retrieves the initial membership (used by the GMS when started
   by sending event FIND_INITIAL_MBRS down the stack). We do this by mcasting PING
   requests to an IP MCAST address (or, if gossiping is enabled, by contacting the gossip
   server). The responses should allow us to determine the coordinator whom we have to
   contact, e.g. in case we want to join the group.  When we are a server (after having
   received the BECOME_SERVER event), we'll respond to PING requests with a PING
   response.<p> The FIND_INITIAL_MBRS event will eventually be answered with a
   FIND_INITIAL_MBRS_OK event up the stack.  
*/
public class PING extends Protocol {
    boolean         trace=false;
    Vector          members=new Vector(), initial_members=new Vector();
    Address         local_addr=null;
    String          group_addr=null;
    String          groupname=null;
    long            timeout=2000;
    long            num_initial_members=2;
    String          gossip_host=null;
    int             gossip_port=0;
    GossipClient    gossip=null;
    boolean         is_server=false;



    public String  GetName() {return "PING";}



    public Vector ProvidedUpServices() {
	Vector ret=new Vector();
	ret.addElement(new Integer(Event.FIND_INITIAL_MBRS));
	return ret;
    }


    public boolean SetProperties(Properties props) {
	String     str;

	this.props=props;
	str=props.getProperty("trace");
	if(str != null) {
	    trace=new Boolean(str).booleanValue();
	    props.remove("trace");
	}

	str=props.getProperty("timeout");              // max time to wait for initial members
	if(str != null) {
	    timeout=new Long(str).longValue();
	    props.remove("timeout");
	}

	str=props.getProperty("num_initial_members");  // wait for at most n members
	if(str != null) {
	    num_initial_members=new Integer(str).intValue();
	    props.remove("num_initial_members");
	}

	str=props.getProperty("gossip_host");
	if(str != null) {
	    gossip_host=new String(str);
	    props.remove("gossip_host");
	}

	str=props.getProperty("gossip_port");
	if(str != null) {
	    gossip_port=new Integer(str).intValue();
	    props.remove("gossip_port");
	}

	if(gossip_host != null && gossip_port != 0)
	    gossip=new GossipClient(gossip_host, gossip_port);

	if(props.size() > 0) {
	    System.err.println("PING.SetProperties(): the following properties are not recognized:");
	    props.list(System.out);
	    return false;
	}
	return true;
    }





    public void Up(Event evt) {
	Message      msg, rsp_msg;
	Object       obj;
	PingHeader   hdr, rsp_hdr;
	PingRsp      rsp;
	Object       coord;
	Address      sender;
	boolean      contains;
	Vector       tmp;


	switch(evt.GetType()) {

	case Event.MSG:
	    msg=(Message)evt.GetArg();
	    obj=msg.PeekHeader();
	    	    if(obj == null || !(obj instanceof PingHeader)) {
		PassUp(evt);
		return;
	    }
	    hdr=(PingHeader)msg.RemoveHeader();
	    
	    switch(hdr.type) {

	    case PingHeader.GET_MBRS_REQ:   // return Rsp(local_addr, coord)
		if(!is_server) {
		    //System.err.println("PING.Up(GET_MBRS_REQ): did not return a response " +
		    //	       "as I'm not a server yet !");
		    return;
		}
		synchronized(members) {
		    coord=members.size() > 0 ? members.firstElement() : local_addr;
		}
		rsp_msg=new Message(msg.GetSrc(), null, null);
		rsp_hdr=new PingHeader(PingHeader.GET_MBRS_RSP, new PingRsp(local_addr, coord));
		rsp_msg.AddHeader(rsp_hdr);
		PassDown(new Event(Event.MSG, rsp_msg));
		return;

	    case PingHeader.GET_MBRS_RSP:   // add response to vector and notify waiting thread
		rsp=(PingRsp)hdr.arg;		
		synchronized(initial_members) {
		    initial_members.addElement(rsp);
		    initial_members.notify();
		}
		return;

	    default:
		System.err.println("PING.Up(): got PING header with unknown type (" + 
				   hdr.type + ")");
		return;
	    }
	    

	case Event.SET_LOCAL_ADDRESS:
	    PassUp(evt);
	    local_addr=(Address)evt.GetArg();
	    if(gossip != null)
		gossip.SetLocalAddress(local_addr);
	    break;

	default:
	    PassUp(evt);            // Pass up to the layer above us
	    break;
	}
    }





    public void Down(Event evt) {
	Message      msg;
	PingHeader   hdr;
	long         time_to_wait, start_time;
	Vector       gossip_rsps=new Vector();

	switch(evt.GetType()) {

	case Event.FIND_INITIAL_MBRS:   // sent by GMS layer, pass up a GET_MBRS_OK event
	    if(gossip != null) {
		gossip_rsps=gossip.Get(group_addr);
		
		
		if(gossip_rsps != null && gossip_rsps.size() > 0) {
		    // Set a temporary membership in the UDP layer, so that the following multicast
		    // will be sent to all of them
		    Event view_event=new Event(Event.TMP_VIEW, MakeView(gossip_rsps));
		    PassDown(view_event); // needed e.g. by failure detector or UDP	
		}
		else {
		    PassUp(new Event(Event.FIND_INITIAL_MBRS_OK, null));
		    return;
		}
		Util.Sleep(500);
	    }
	    
	    
	    // 1. Mcast GET_MBRS_REQ message
	    
	    hdr=new PingHeader(PingHeader.GET_MBRS_REQ, null);
	    msg=new Message(null, null, null);
	    msg.AddHeader(hdr);
	    PassDown(new Event(Event.MSG, msg));
	    
	    
	    // 2. Wait 'timeout' ms or until 'num_initial_members' have been retrieved
	    synchronized(initial_members) {
		initial_members.removeAllElements();
		start_time=System.currentTimeMillis();
		time_to_wait=timeout;
		
		while(initial_members.size() < num_initial_members && time_to_wait > 0) {
		    try {initial_members.wait(time_to_wait);} catch(Exception e) {}
		    time_to_wait-=System.currentTimeMillis() - start_time;
		}
	    }
	    
	    // 3. Send response
	    PassUp(new Event(Event.FIND_INITIAL_MBRS_OK, initial_members));
	    break;

	    
	case Event.STOP:
	    PassDown(evt);
	    if(gossip != null)
		gossip.Stop();
	    break;
	    
	case Event.TMP_VIEW:
	case Event.VIEW_CHANGE:	    
	    Vector tmp;
	    if((tmp=(Vector)((View)evt.GetArg()).GetMembers()) != null) {
		synchronized(members) {
		    members.removeAllElements();
		    for(int i=0; i < tmp.size(); i++)
			members.addElement(tmp.elementAt(i));
		}
	    }
	    PassDown(evt);
	    break;

	case Event.BECOME_SERVER: // called after client has join and is fully working group member
	    PassDown(evt);
	    is_server=true;
	    break;

	case Event.CONNECT:
	    group_addr=(String)evt.GetArg();
	    PassDown(evt);
	    if(gossip != null) {
		gossip.SetGroupAddress(group_addr);
		gossip.Start();
	    }
	    break;

	case Event.DISCONNECT:
	    if(gossip != null)
		gossip.Stop();
	    PassDown(evt);
	    break;
	    
	default:
	    PassDown(evt);          // Pass on to the layer below us
	    break;
	}
    }



    /* -------------------------- Private methods ---------------------------- */


    private View MakeView(Vector mbrs) {
	Object  coord=null;
	long    id=0;
	ViewId  view_id=new ViewId(local_addr);

	coord=view_id.GetCoordAddress();
	id=view_id.GetId();

	return new View(coord, id, mbrs);
    }




}
