package JavaGroups.JavaStack;


import java.util.*;
import JavaGroups.*;
import JavaGroups.Algorithms.*;






/**
   Based on Protocol, but incorporates RequestCorrelator and GroupRequest: the latter can 
   be used to mcast messages to all members and receive their reponses.<p>
   A protocol based on this template can send messages to all members and receive all, a single,
   n, or none responses. Requests directed towards the protocol can be handled by overriding
   method <code>Handle</code>.<p>
   Requests and responses are in the form of <code>Message</code>s, which would typically need to
   contain information pertaining to the request/response, e.g. in the form of objects contained
   in the message. To use remote method calls, use <code>RpcProtocol</code> instead.<p>
   Typical use of of a <code>MessageProtocol</code> would be when a protocol needs to interact with
   its peer protocols at each of the members' protocol stacks. A simple protocol like fragmentation,
   which does not need to interact with other instances of fragmentation, may simply subclass
   <code>Protocol</code> instead.
 */
public abstract class MessageProtocol extends Protocol implements Transport, RequestHandler {
    public static final int FIRST       = 1;
    public static final int ALL         = 2;
    public static final int NONE        = 3;   // don't wait for response, asynchronous mcast
    public static final int MAJORITY    = 4;   // majority of all non-faulty members
    
    RequestCorrelator       _corr=null;
    GroupRequest            _req=null;
    
    
    

    



   /**
       Cast a message to all members, and wait for <code>mode</code> responses. The responses are
       returned in a response list, where each response is associated with its sender.<p>
       Uses <code>GroupRequest</code>.
       @param dests The members to which then message is to be sent
       @param msg The message to be sent to n members
       @param mode The number of responses to wait for:
                   <ol>
		   <li>FIRST: return when 1 response has been received
		   <li>ALL: wait for all responses (minus responses from suspected members)
		   <li>NONE: don't wait for any responses (asynchronous, one-way message send)
		   <li>MAJORITY: wait for a majority of the members to respond. If a members
		                 crashes during the wait, the majority is recomputed.
		   </ol>
       @param timeout If 0: wait forever. Otherwise, wait for <code>mode</code> responses 
                      <em>or</em> timeout time.
       @return RspList A list of responses. Each response is an <code>Object</code> and associated
                       to its sender.
   */    
    public RspList CastMessage(Vector dests, Message msg, int mode, long timeout) {
	int     rsp_mode=TranslateMode(mode);
	
	// This marks message as sent by us ! (used in Up()
	// msg.AddHeader(new MsgProtHeader(GetName()));   ++ already done by RequestCorrelator

	_req.Reset(msg, dests, mode, timeout, 0);
	_req.Execute();
	return _req.GetResults();
    }





    
    /**
       Sends a message to a single member (destination = msg.dest) and returns the response. 
       The message's destination must be non-zero !
    */
    public Object SendMessage(Message msg, int mode, long timeout) throws Timeout, Suspected {
	Vector    mbrs=new Vector();
	RspList   rsp_list=null;
	Object    dest=msg.GetDest();
	Rsp       rsp;
	Object    retval=null;

	if(dest == null) {
	    System.out.println("MessageProtocol.SendMessage(): the message's destination is null ! " +
			       "Cannot send message !");
	    return null;
	}

	
	mbrs.addElement(dest);   // dummy membership (of destination address)	
	_req.Reset(msg, mbrs, GroupRequest.GET_FIRST, timeout, 0);
	_req.Execute();
	rsp_list=_req.GetResults();
	
	if(rsp_list.size() == 0) {
	    System.err.println("MessageProtocol.SendMessage(): response list is empty !");
	    return null;	    
	}
	if(rsp_list.size() > 1)
	    System.err.println("MessageProtocol.SendMessage(): response list contains " +
			       "more that 1 response; returning first response !");
	rsp=(Rsp)rsp_list.elementAt(0);
	if(rsp.WasSuspected())
	    throw new Suspected(dest);
	if(!rsp.WasReceived())
	    throw new Timeout();
	return rsp.GetValue();
    }







    /**
       Processes a request destined for this layer. The return value is sent as response.
    */
    public Object Handle(Message req) {
	System.out.println("MessageProtocol.Handle(): this method should be overridden !");
	return null;
    }




    
    
    /* ------------------------- Transport interface ---------------------------- */
    
    /**
       Add header and pass down. Only used for sending responses (by RequestCorrelator).
    */
    public void Send(Message msg) throws Exception {
	PassDown(new Event(Event.MSG, msg));
    }
    
    public Object Receive(long timeout) throws Exception {
	throw new Exception("MessageProtocol.Receive(): this method must not be used !");
    }
    
    /* ---------------------- End of Transport interface ------------------------ */
    
    
    

    
    public final void Up(Event evt) {
	Message  msg;
	Object   hdr;
	
	switch(evt.GetType()) {

	case Event.START:
	    Start();
	    HandleUpEvent(evt);
	    PassUp(evt);
	    break;
	    
	default:
	    if(HandleUpEvent(evt))
		_corr.Receive(evt);
	    break;
	}
    }





    public final void Down(Event evt) {
	Message msg;

	switch(evt.GetType()) {
	case Event.STOP:
	    Stop();
	    HandleDownEvent(evt);
	    PassDown(evt);
	    break;

	default:
	    if(HandleDownEvent(evt))
		PassDown(evt);
	    break;
	}
    }


    /**
       Handle up event. Return false if it should not be passed up the stack.
     */
    protected boolean HandleUpEvent(Event evt) {
	// override in subclasses
	return true;
    }

    /**
       Handle down event. Return false if it should not be passed down the stack.
     */
    protected boolean HandleDownEvent(Event evt) {
	// override in subclasses
	return true;
    }






    protected void Start() {
	if(_corr == null) {
	    _corr=new RequestCorrelator(GetName(), this, this);
	    _req=new GroupRequest(null, _corr, null, GroupRequest.GET_ALL);
	}
    }


    protected void Stop() {
	if(_corr != null) {
	    _corr.Stop();
	    _corr=null;
	}
    }


    int TranslateMode(int mode) {
	switch(mode) {
	case FIRST:     return GroupRequest.GET_FIRST;
	case ALL:       return GroupRequest.GET_ALL;
	case NONE:      return GroupRequest.GET_NONE;
	case MAJORITY:  return GroupRequest.GET_MAJORITY;
	}
	return GroupRequest.GET_ALL;
    }


}
