package JavaGroups;

import java.io.*;
import java.util.*;
import JavaGroups.*;


/**
   A channel represents a group communication endpoint (like BSD datagram sockets). A
   client joins a group by connecting the channel to a group address and leaves it by
   disconnecting. Messages sent over the channel are received by all group members that
   are connected to the same group (that is, all member that have the same group
   address).<p>

   The FSM for a channel is roughly as follows: a channel is created
   (<em>unconnected</em>). The channel is connected to a group
   (<em>connected</em>). Messages can now be sent and received. The channel is
   disconnected from the group (<em>unconnected</em>). The channel could now be connected to a 
   different group again. The channel is closed (<em>closed</em>).<p>

   Only a single sender is allowed to be connected to a channel at a time, but there can be
   more than one channel in an application.<p>

   Messages can be sent to the group members using the <em>Send</em> method and messages
   can be received using <em>Receive</em> (pull approach).<p>

   A channel instance is created using either a <em>ChannelFactory</em> or the public
   constructor. Each implementation of a channel must provide a subclass of
   <code>Channel</code> and an implementation of <code>ChannelFactory</code>.  <p>
   Various degrees of sophistication in message exchange can be achieved using building
   blocks on top of channels, e.g. light-weight groups, synchronous message invocation,
   or remote method calls. Channels are on the same abstraction level as sockets, and
   should really be simple to use. Higher-level abstractions are all built on top of
   channels.

   @author  Bela Ban
   @see     java.net.DatagramPacket
   @see java.net.MulticastSocket */
public abstract class Channel implements Transport {

    public static final int BLOCK                 =  0;
    public static final int VIEW                  =  1;
    public static final int SUSPECT               =  2;
    public static final int LOCAL                 =  3;
    public static final int GET_STATE_EVENTS      =  4;



    /** Hidden from clients */
    protected Channel() throws ChannelException {}


    /**
       Creates a channel (initially not connected).

       @param properties Properties of the channel. This argument serves a generic option (hint)
	      for subclasses as to how to create the channel (parameterization). Interpreted by
	      the channel subclasses. May be for example a <code>String</code> denoting the QoS
	      required by the underlying transport, or <code>null</code> to use default
	      properties.  
	      
       @exception ChannelException Channel creation failed.

    */
    protected Channel(Object properties) throws ChannelException {}




    /** 
	Connects the channel to a group. The client is now able to receive group
	messages, views and block events (depending on the options set) and to send
	messages to (all or single) group members.  This is a null operation if already
	connected.<p>

	All channels with the same group address form a group, that means all messages
	sent to the group will be received by all channels connected to the same group
	address.<p>

	The group address is opaque, and can only be interpreted by a channel
	implementation, i.e. the implementation decides what to do with the object.

	@exception ChannelClosed The channel is closed and therefore cannot be used any longer.
	                         A new channel has to be created first.
	
	@see Channel.Disconnect */
    abstract public void    Connect(Object group_address) throws ChannelClosed;





    /** Disconnects the channel from the current group (if connected), leaving the group. 
	It is a null operation if not connected. It is a null operation if the channel is closed.

	@see Channel.Connect */
    abstract public void    Disconnect();


    /**
       Destroys the channel and its associated resources (e.g. the protocol stack). After a channel
       has been closed, invoking methods on it throws the <code>ChannelClosed</code> exception 
       (or results in a null operation). It is a null operation if the channel is already closed.<p>
       If the channel is connected to a group, <code>Disconnect</code> will be called first.
     */
    abstract public void Close();




    /** Sends a message to a destination. The message contains
	<ol>
	<li>a destination address (Object). A <code>null</code> address sends the message 
	    to all group members.
	<li>a source address. Can be left empty. Will be filled in by the protocol stack.
	<li>a byte buffer. The message contents.
	<li>several additional fields. They can be used by application programs (or patterns). E.g.
            a message ID, a <code>oneway</code> field which determines whether a response is 
	    expected etc.
	</ol>
	@param msg The message to be sent. Destination and buffer should be set. A null destination
	           means to send to all group members.

	@exception ChannelNotConnected The channel must be connected to send messages.

	@exception ChannelClosed The channel is closed and therefore cannot be used any longer.
	                         A new channel has to be created first.

    */
    abstract public void    Send(Message msg) throws ChannelNotConnected, ChannelClosed;






    /** Receives a message, a view change or a block event. By using <code>SetOpt</code>, the
	type of objects to be received can be determined (e.g. not views and blocks, just
	messages).

	The possible types returned can be:
	<ol>
	<li><code>Message</code>. Normal message
	<li><code>View</code>. A view change.
	<li><code>BlockEvent</code>. A block event indicating an impending view change.
	<li><code>SuspectEvent</code>. A notification of a suspected member.
	<li><code>GetStateEvent</code>. The current state of the application should be
	                                returned using <code>ReturnState</code>.
	<li><code>SetStateEvent</code>. The state of a single/all members as requested previously
	                                by having called <codee>Channel.GetState(s).
	</ol>
	The <code>instanceof</code> operator can be used to discriminate between different types 
	returned.
	@param timeout Value in milliseconds. Value <= 0 means wait forever
	@return A Message, View, BlockEvent,SuspectEvent, GetStateEvent object or
	        SetStateEvent, depending on what is on top of the internal queue.

	@exception ChannelNotConnected The channel must be connected to receive messages.

	@exception ChannelClosed The channel is closed and therefore cannot be used any longer.
	                         A new channel has to be created first.

	@exception Timeout Thrown when a timeout has occurred.
    */
    abstract public Object  Receive(long timeout) throws ChannelNotConnected, ChannelClosed, Timeout;




    /** Returns the next message, view, block, suspect or other event <em>without removing
	it from the queue</em>.
	@param timeout Value in milliseconds. Value <= 0 means wait forever
	@return A Message, View, BlockEvent, SuspectEvent, GetStateEvent or SetStateEvent object,
	        depending on what is on top of the internal queue.

	@exception ChannelNotConnected The channel must be connected to receive messages.

	@exception ChannelClosed The channel is closed and therefore cannot be used any longer.
	                         A new channel has to be created first.

	@exception Timeout Thrown when a timeout has occurred.

	@see Receive
    */
    abstract public Object  Peek(long timeout) throws ChannelNotConnected, ChannelClosed, Timeout;





    /** Gets the current view. This does <em>not</em> retrieve a new view, use
	<code>Receive</code> to do so. The view may only be available after a successful
	<code>Connect</code>. The result of calling this method on an unconnected channel
	is implementation defined (may return null). Calling it on a channel that is not
	enabled to receive view events (via <code>SetOpt</code>) returns
	<code>null</code>. Calling this method on a closed channel returns a null view.

	@return The current view.  */
    abstract public View    GetView();





    /** 
	Returns the channel's own address. The result of calling this method on an unconnected
	channel is implementation defined (may return null). Calling this method on a closed
	channel returns null.

	@return The channel's address. Generated by the underlying transport, and opaque.
	        Addresses can be used as destination in the <code>Send</code> operation.  
    */
    abstract public Object  GetLocalAddress();




    /**
       Returns the group address of the group of which the channel is a member. This is
       the object that was the argument to <code>Connect</code>. Calling this method on a closed 
       channel returns <code>null</code>.

       @return The group address */
    abstract public Object  GetGroupAddress();








    /** 
	Sets an option. The following options are currently recognized:
	<ol>
	<li><code>BLOCK</code> (0). Turn the reception of BLOCK events on/off (value is Boolean).
	    Default is off. If set to on, receiving VIEW events will be set to on, too.
	<li><code>VIEW</code>  (1). Turn the reception of VIEW events on/off (value is Boolean).
	    Default is on.
	<li><code>SUSPECT</code> (2). Turn the reception of SUSPECT events on/off (value is Boolean).
	    Default is on.
	<li><code>LOCAL</code> (3). Receive its own broadcast messages to the group 
	          (value is Boolean). Default is on.
	<li><code>GET_STATE_EVENTS</code> (4). Turn the reception of GetState events on/off 
	          (value is Boolean). Default is off, which means that no other members can
		  ask this member for its state (null will be returned).
	</ol>
	This method can be called on an unconnected channel. Calling this method on a 
	closed channel has no effect.
     */
    abstract public void    SetOpt(int option, Object value);





    /** 
	Gets an option. This method can be called on an unconnected channel.  Calling this 
	method on a closed channel returns <code>null</code>.

	@param option  The option to be returned.
	@return The object associated with an option.  
    */
    abstract public Object  GetOpt(int option);





    /** Called to acknowledge a Block (callback in <code>MembershipListener</code> or 
	<code>BlockEvent</code> received from call to <code>Receive</code>). 
	After sending BlockOk, no messages should be sent until a new view has been received.
	Calling this method on a closed channel has no effect.
    */
    abstract public void    BlockOk();



    /**
       Retrieve the state of the group. Will usually contact the oldest group member to get
       the state. When the method returns true, a <code>SetStateEvent</code> will have been
       added to the channel's queue, causing <code>Receive</code> to return the state in one of
       the next invocations. If false, no state will be retrieved by <code>Receive</code>.
       @param timeout Milliseconds to wait for the response (0 = wait indefinitely).
       @return boolean True if the state was retrieved successfully, otherwise false.
     */
    abstract public boolean GetState(long timeout);


    /**
       Retrieve all states of the group members. Will contact all group members to get
       the states. When the method returns true, a <code>SetStateEvent</code> will have been
       added to the channel's queue, causing <code>Receive</code> to return the states in one of
       the next invocations. If false, no states will be retrieved by <code>Receive</code>.
       @param timeout Milliseconds to wait for the response (0 = wait indefinitely).
       @return boolean True if the state was retrieved successfully, otherwise false.
    */

    abstract public boolean  GetAllStates(long timeout);



    /**
       Called by the application is response to receiving a <code>GetState</code> object when
       calling <code>Receive</code>.
       @param state The state of the application as a Serializable (to send over the network).
     */
    public void             ReturnState(Serializable state) {
	; // should be overridden by subclasses
    }
}
