package JavaGroups.Algorithms;


import java.util.*;
import JavaGroups.*;



/**
   Sends a message to all members of the group and waits for all responses (or timeout). Returns a
   boolean value (success or failure). Results (if any) can be retrieved when done.<p>
   When started an array of responses, correlating to the membership, is created. Each response
   is added to the corresponding field in the array. When all fields have been set, the algorithm
   terminates.
   This algorithm can optionally use a suspicion service (failure detector) to detect (and 
   exclude from the membership) fauly members. If no suspicion service is available, timeouts 
   can be used instead (see <code>Execute</code>). When done, a list of suspected members 
   can be retrieved.<p>
   Because a channel might deliver requests, and responses to <em>different</em> requests, the 
   <code>GroupRequest</code> class cannot itself receive and process requests/responses from the
   channel. A mechanism outside this class has to do this; it has to determine what the responses
   are for the message sent by the <code>Execute</code> method and call <code>ReceiveResponse</code>
   to do so.<p>
   <b>Requirements</b>: lossless delivery, e.g. acknowledgement-based message confirmation.
*/ 

public class GroupRequest implements RspCollector, Command {
    public static final int      GET_FIRST        = 1; // return only first response
    public static final int      GET_ALL          = 2; // return all responses
    public static final int      GET_MAJORITY     = 3; // return majority (of all non-faulty members)
    public static final int      GET_ABS_MAJORITY = 4; // return majority (of all members, may block)
    public static final int      GET_N            = 5; // return n responses (may block)
    public static final int      GET_NONE         = 6; // return no response

    private final short          NOT_RECEIVED=0;
    private final short          RECEIVED=1;
    private final short          SUSPECTED=2;
    

    private   Object             membership[]=null;
    private   Object             responses[]=null;
    private   short              received[]=null;
    private   Vector             suspects=new Vector();
    protected Message            request_msg=null;
    protected RequestCorrelator  corr=null;
    protected int                rsp_mode=GET_ALL;
    protected boolean            done=false;
    protected Integer            rsp_mutex=new Integer(0);
    protected long               timeout=0;
    protected int                expected_mbrs=0;






    /**
       @param m The message to be sent
       @param corr The request correlator to be used. A request correlator sends requests tagged with
                   a unique ID and notifies the sender when matching responses are received. The 
		   reason <code>GroupRequest</code> uses it instead of a <code>Transport</code> is
		   that multiple requests/responses might be sent/received concurrently.
       @param members The initial membership. This value reflects the membership to which the request
                      is sent (and from which potential responses are expected). Is reset by Reset().
       @param rsp_mode How many responses are expected. Can be
                       <ol>
		       <li><code>GET_ALL</code>: wait for all responses from non-suspected members. 
		           A suspicion service might warn 
		           us when a member from which a response is outstanding has crashed, so it can
			   be excluded from the responses. If no suspision service is available, a 
			   timeout can be used (a value of 0 means wait forever). <em>If a timeout of 
			   0 is used, no suspicion service is available and a member from which we 
			   expect a response has crashed, this methods blocks forever !</em>.
		       <li><code>GET_FIRST</code>: wait for the first available response.
		       <li><code>GET_MAJORITY</code>: wait for the majority of all responses. The
		           majority is re-computed when a member is suspected.
		       <li><code>GET_ABS_MAJORITY</code>: wait for the majority of 
		                                          <em>all</em> members.
		           This includes failed members, so it may block if no timeout is specified.
		       <li><code>GET_N</CODE>: wait for N members. 
		                               Return if n is >= membership+suspects.
		       <li><code>GET_NONE</code>: don't wait for any response. Essentially send an
		            asynchronous message to the group members.
		       </ol>
       @param expected_mbrs Number of responses expected (used with GET_ALL).
     */
    public GroupRequest(Message m, RequestCorrelator corr, Vector members, int rsp_mode) {
	request_msg=m;
	this.corr=corr;
	this.rsp_mode=rsp_mode;
	Reset(members);
	suspects.removeAllElements();
    }


    /**
       @param timeout Time to wait for responses (ms). A value of <= 0 means wait indefinitely
                      (e.g. if a suspicion service is available; timeouts are not needed).
    */
    public GroupRequest(Message m, RequestCorrelator corr, Vector members,
			int rsp_mode, long timeout, int expected_mbrs) {
	this(m, corr, members, rsp_mode);
	if(timeout > 0)
	    this.timeout=timeout;
	this.expected_mbrs=expected_mbrs;
    }






    /** Sends the message. Returns when n responses have been received, or when a 
	timeout has occurred. <em>n</em> can be the first response, all responses, or a
	majority of the responses.
     */
    public synchronized boolean Execute() {
	boolean retval;

	if(corr == null) {
	    System.err.println("GroupRequest.Execute(): request correlator is null !");
	    return false;
	}
	synchronized(rsp_mutex) {
	    done=false;	
	    retval=DoExecute(timeout);
	    done=true;
	    return retval;
	}
    }



    /**
       Resets the group request, so it can be reused for another execution.
     */
    public void Reset(Message m, int mode, long timeout) {
	synchronized(rsp_mutex) {
	    done=false;
	    request_msg=m;
	    rsp_mode=mode;
	    this.timeout=timeout;
	}
    }



    public void Reset(Message m, final Vector members, int rsp_mode,
		      long timeout, int expected_rsps) {
	synchronized(rsp_mutex) {
	    Reset(m, rsp_mode, timeout);
	    Reset(members);
	    suspects.removeAllElements();
	    this.expected_mbrs=expected_mbrs;
	}
    }



    public void Reset(Vector members) {
	if(members != null) {
	    int size=members.size();
	    membership=new Object[size];
	    responses=new Object[size];
	    received=new short[size];
	    for(int i=0; i < size; i++) {
		membership[i]=members.elementAt(i);
		responses[i]=null;
		received[i]=NOT_RECEIVED;
	    }
	}
	else {
	    if(membership != null) {
		for(int i=0; i < membership.length; i++) {
		    responses[i]=null;
		    received[i]=NOT_RECEIVED;
		}
	    }
	}
    }






    /* ---------------------- Interface RspCollector -------------------------- */

    /**
       <b>Callback</b> (called by RequestCorrelator).
       Adds a response to the response table. When all responses have been received, 
       <code>Execute</code> returns. 
     */
    public void ReceiveResponse(Message m) {
	Object   sender=m.GetSrc();
	Object   val=null, mbr;

	if(done) {
	    System.err.println("GroupRequest.ReceiveResponse(): command is done; " +
			       "cannot add response !");
	    return;
	}
	
	if(suspects != null && suspects.size() > 0 && suspects.contains(sender)) {
	    System.err.println("Received response from suspected member " + sender +
			       "; discarding !");
	    return;
	}
	
	if(m.GetBuffer() != null) {
	    try {
		val=Util.ObjectFromByteBuffer(m.GetBuffer());
	    }
	    catch(Exception e) {
		System.err.println("GroupRequest.ReceiveResponse(): " + e);
	    }
	}

	synchronized(rsp_mutex) {
	    for(int i=0; i < membership.length; i++) {
		mbr=membership[i];
		if(mbr.equals(sender)) {
		    if(received[i] == NOT_RECEIVED) {
			responses[i]=val;
			received[i]=RECEIVED;
			rsp_mutex.notify();       // wakes up Execute()
			break;
		    }
		}
	    }
	}

	// PrintReceived();
	    
    }





    
    /** <b>Callback</b> (called by RequestCorrelator).
	Report to <code>GroupRequest</code> that a member is reported as faulty (suspected).
	This method would probably be called when getting a suspect message from a failure detector
	(where available). It is used to exclude faulty members from the response list.
     */
    public void Suspect(Object suspected_member) {
	Object   mbr;

	synchronized(rsp_mutex) {  // modify 'suspects' and 'responses' array
	    for(int i=0; i < membership.length; i++) {
		mbr=membership[i];
		if(mbr.equals(suspected_member)) {
		    if(!suspects.contains(suspected_member)) {
			suspects.addElement(suspected_member);
			responses[i]=null;
			received[i]=SUSPECTED;
			rsp_mutex.notify();
			break;
		    }
		}
	    }
	}

	// PrintReceived();
    }

    /* -------------------- End of Interface RspCollector ----------------------------------- */






    /** Returns the results as a RspList.
     */
    public RspList GetResults() {
	RspList  retval=new RspList();
	Object   sender, result;

	synchronized(rsp_mutex) {
	    for(int i=0; i < membership.length; i++) {
		sender=membership[i];

		switch(received[i]) {
		case SUSPECTED:
		    retval.AddSuspect(sender);
		    break;
		case RECEIVED:
		    retval.AddRsp(sender, responses[i]);
		    break;
		case NOT_RECEIVED:
		    retval.AddNotReceived(sender);
		    break;
		}
	    }
	    return retval;
	}
    }


    public String toString() {
	StringBuffer ret=new StringBuffer();
	ret.append("[GroupRequest:\n");
	ret.append("members: ");

	for(int i=0; i < membership.length; i++)
	    ret.append(membership[i] + " ");

	ret.append("\nresponses: ");
	for(int i=0; i < responses.length; i++)
	    ret.append(responses[i] + " ");

	if(suspects.size() > 0)
	    ret.append("\nsuspects: " + suspects);

	ret.append("\nrequest_msg: " + request_msg);
	ret.append("\nrsp_mode: " + rsp_mode);
	ret.append("\ndone: " + done);
	ret.append("\ntimeout: " + timeout);
	ret.append("\nexpected_mbrs: " + expected_mbrs);
	ret.append("\n]");
	return ret.toString();
    }

    
    public int     GetNumSuspects()   {return suspects.size();}
    public Vector  GetSuspects()      {return suspects;}
    public boolean IsDone()           {return done;}
    









    /* --------------------------------- Private Methods -------------------------------------*/

    protected int DetermineMajority(int i) {
	return i < 2 ? i : (i/2) + 1;
    }



    protected boolean DoExecute(long timeout) {
	boolean   retval=false;
	long      start_time=0;
	long      req_id=System.currentTimeMillis();
	Object    mbr, suspect;
	
	Reset(null);   // clear 'responses' array

	if(suspects != null) {  // mark all suspects in 'received' array
	    for(int i=0; i < suspects.size(); i++) {
		suspect=suspects.elementAt(i);
		for(int j=0; j < membership.length; j++) {
		    mbr=membership[j];
		    if(mbr.equals(suspect)) {
			received[j]=SUSPECTED;
			break;
		    }
		}
	    }
	}

	try {	    
	    if(rsp_mode == GET_NONE) {  // send oneway
		corr.SendRequest(req_id, request_msg, null);
		return true;
	    }
	    else
		corr.SendRequest(req_id, request_msg, this);
	}
	catch(Exception e) {
	    System.err.println("GroupRequest: " + e);
	    corr.Done(req_id);
	    return false;
	}
	
	if(timeout <= 0) {
	    while(true) {                     	    /* Wait for responses: */
		if(GetResponses()) {
		    corr.Done(req_id);
		    return true;
		}
		try {rsp_mutex.wait();}
		catch(Exception e) {}
	    }
	}
	else {
	    start_time=System.currentTimeMillis();
	    while(timeout > 0) {	            /* Wait for responses: */
		if(GetResponses()) {
		    corr.Done(req_id);
		    return true;
		}
		timeout=timeout-(System.currentTimeMillis()-start_time);
		if(timeout > 0) {
		    try {rsp_mutex.wait(timeout);}
		    catch(Exception e) {}
		}
	    }
	    corr.Done(req_id);
	    return false;
	}
    }




    protected boolean GetResponses() {
	int num_not_received=GetNum(NOT_RECEIVED);
	int num_received=GetNum(RECEIVED);
	int num_suspected=GetNum(SUSPECTED);
	int num_total=membership.length;
	int majority=DetermineMajority(num_total);

	switch(rsp_mode) {
	case GET_FIRST:
	    if(num_received > 0)
		return true;
	    if(num_suspected >= num_total)  // e.g. 2 members, and both suspected
		return true;		
	    break;
	case GET_ALL:
	    if(num_not_received > 0)
		return false;
	    return true;
	case GET_MAJORITY:
	    if(num_received + num_suspected >= majority)
		return true;
	    break;
	case GET_ABS_MAJORITY:
	    if(num_received >= majority)
		return true;
	    break;
	case GET_N:
	    if(expected_mbrs >= num_total) {
		rsp_mode=GET_ALL;
		return GetResponses();
	    }
	    if(num_received >= expected_mbrs)
		return true;
	    if(num_received + num_not_received < expected_mbrs) {
		if(num_received + num_suspected >= expected_mbrs)
		    return true;
		return false;
	    }
	    return false;
	case GET_NONE:
	    return true;
	default:
	    System.err.println("GroupRequest.Execute(): rsp_mode " + rsp_mode + " unknown !");
	    break;
	}
	return false;
    }




    /** Return number of elements of a certain type in array 'received'. Type can be RECEIVED,
	NOT_RECEIVED or SUSPECTED */
    int GetNum(int type) {
	int retval=0;
	for(int i=0; i < received.length; i++)
	    if(received[i] == type)
		retval++;
	return retval;
    }

    void PrintReceived() {
	for(int i=0; i < received.length; i++) {
	    System.out.println(membership[i] + ": " + 
			       (received[i] == NOT_RECEIVED ? "NOT_RECEIVED" : 
			       received[i] == RECEIVED     ? "RECEIVED" : "SUSPECTED"));
	}
    }
    
}
