package JavaGroups;


import java.util.*;

/**
   ACK-based sliding window for a sender. Messages are added to the window according to message
   IDs. When an ACK is received, the corresponding entry is marked and the window advanced as
   much as possible. The addition of a message to a full window will block until a slot becomes
   free (when receiving an ACK). A retransmission thread continously iterates over the entries,
   retransmitting messages for which no ACK has been received in <code>timeout</code> time.
 */
public class AckSenderWindow implements Runnable {
    int                  window_size=100;
    long                 low_mark=0;                 // keeps track of the lowest seq_no
    Thread               retransmit_thread=null;
    long                 timeout=2000;
    RetransmitCommand    retransmit_command=null;
    Vector               msgs=new Vector();


    public interface RetransmitCommand {
	public void Retransmit(long seq_no, Message msg, int num_tries);
    }


    class Entry {	
	long      seq_no=0;
	long      wakeup_at=System.currentTimeMillis() + timeout;
	Message   msg;
	int       num_tries=0;
	boolean   ack_received=false;

	Entry(long seq_no, Message msg) {
	    this.seq_no=seq_no;
	    this.msg=msg;
	}

	public String toString() {
	    StringBuffer ret=new StringBuffer();
	    ret.append(seq_no);
	    if(ack_received)
		ret.append("+");
	    else
		ret.append("-");
	    return ret.toString();
	}
    }


    /**
       Creates a new instance. Thre retransmission thread has to be started separately with
       <code>Start</code>.
       @param com If not null, its method <code>Retransmit</code> will be called when a message
                  needs to be retransmitted (called by the retransmitter thread).
       @param low_mark The next sequence number to expect ACKs for.
     */
    public AckSenderWindow(RetransmitCommand com, long low_mark) {
	retransmit_command=com;
	this.low_mark=low_mark;
    }
    

    /**
       Creates a new instance. Thre retransmission thread has to be started separately with
       <code>Start</code>.
       @param window_size Determines how many messages can be added to the window before it blocks.
                          The caller will be unblocked when a message is removed.
       @param com If not null, its method <code>Retransmit</code> will be called when a message
                  needs to be retransmitted (called by the retransmitter thread).
       @param low_mark The next sequence number to expect ACKs for.
     */
    public AckSenderWindow(int window_size, RetransmitCommand com, long low_mark) {
	this(com, low_mark);
	this.window_size=window_size;
    }


    public long GetLowMark()              {return low_mark;}
    public long GetTimeout()              {return timeout;}
    public void SetTimeout(long timeout)  {this.timeout=timeout;}


    /**
       Starts the retransmission thread (will not be started automatically !). The thread scans over
       the messages sent and requests retransmission for those that haven't received an ACK for
       a certain period of time. 
     */
    public void Start() {
	if(retransmit_thread == null) {
	    retransmit_thread=new Thread(this, "AckSenderWindow.RetransmitThread");
	    retransmit_thread.start();
	}
    }


    public void Stop() {
	if(retransmit_thread != null) {
	    retransmit_thread.stop();
	    retransmit_thread=null;
	}
    }


    /** 
	Add a new message. <code>seq_no</code> should always be 1 higher than the previous
	one. Messages will be ordered according to seq_no. When the number of entries exceeds
	the maximum window size, the caller will be blocked until an ACK has been received
	(removing 1 message).
    */
    public void Add(long seq_no, Message msg) {
	Entry     e;
	boolean   inserted=false;


	synchronized(msgs) {

	    /* Check whether window size allows addition */
	    while(msgs.size() >= window_size) {
		try {
		    msgs.wait();
		}
		catch(Exception ex) {}
	    }
	    
	    if(msgs.size() == 0) {
		msgs.addElement(new Entry(seq_no, msg));
		msgs.notifyAll();
		return;
	    }
	    
	    for(int i=0; i < msgs.size(); i++) {
		e=(Entry)msgs.elementAt(i);
		
		if(seq_no == e.seq_no)  // discard duplicates
		    return;
		
		if(seq_no > e.seq_no)
		    continue;
		else {
		    msgs.insertElementAt(new Entry(seq_no, msg), i);
		    inserted=true;
		    break;
		}
	    }
	    
	    if(!inserted)
		msgs.addElement(new Entry(seq_no, msg));

	    System.out.println("Add: " + toString() + ", low_mark=" + low_mark);

	    msgs.notifyAll();
	}
    }





    /**
       Marks one message as received. Removes as many messages as possible, 
       sets <code>low_mark</code> correspondingly.
     */
    public void Ack(long seq_no) {
	Entry    e;
	boolean  found=false;

	synchronized(msgs) {
	    for(int i=0; i < msgs.size(); i++) {
		e=(Entry)msgs.elementAt(i);
		if(e.seq_no == seq_no) {
		    found=true;
		    e.ack_received=true;		    
		    break;
		}
	    }

	    if(!found)
		return;

	    while(msgs.size() > 0) {  // remove as many messages as possible
		e=(Entry)msgs.elementAt(0);
		if(e.ack_received) {
		    low_mark=e.seq_no;
		    msgs.removeElementAt(0);
		}
		else
		    break;
	    }
	    
	    msgs.notifyAll();  // wake up a) caller on Add and b) retransmission thread
	}
    }





    /**
       Iterate over msgs. For entries that haven't received an ACK within <code>timeout</code>
       time, issue a retransmission command.
     */
    public void run() {
	Entry   e;
	long    current_time=0, min_time, time_to_wait, already_waited, earliest_wakeup;

	System.out.println("retransmitter was started");

	while(true) {
	    synchronized(msgs) {
		while(msgs.size() < 1) {
		    try {
			msgs.wait();
		    }
		    catch(Exception ex) {}
		}
		while(msgs.size() > 0) {
		    current_time=System.currentTimeMillis();

		    earliest_wakeup=current_time + timeout + 1000;  // greater than any timeout value

		    for(int i=0; i < msgs.size(); i++) {
			e=(Entry)msgs.elementAt(i);
			if(e.ack_received) // don't retransmit a msg that has already been ACK'ed
			    continue;


			if(e.wakeup_at <= System.currentTimeMillis()) {
			    e.num_tries++;
			    e.wakeup_at=System.currentTimeMillis() + timeout;
			    if(retransmit_command != null)
				retransmit_command.Retransmit(e.seq_no, e.msg, e.num_tries);
			}

			earliest_wakeup=Math.min(earliest_wakeup, e.wakeup_at);
		    }
		    if(earliest_wakeup > 0) {
			time_to_wait=earliest_wakeup-System.currentTimeMillis();
			if(time_to_wait < 0)
			    time_to_wait=0;
			try {
			    //System.out.println("time_to_wait=" + time_to_wait);
			    msgs.wait(time_to_wait); // gives others a chance to add / remove messages
			}
			catch(Exception ee) {}
		    }
		}
	    }
	}   
    }



    public String toString() {
	StringBuffer retval=new StringBuffer();
	retval.append(msgs);
	return retval.toString();
    }




    public static void main(String[] args) {

	class Retr implements RetransmitCommand {

	    public void Retransmit(long seq_no, Message msg, int num_tries) {
		System.out.println("--> Retransmit(" + seq_no + ", " + num_tries + ")");
	    }	    

	}

	class Adder extends Thread {
	    AckSenderWindow s;
	    Message         mm=new Message();

	    public Adder(AckSenderWindow s) {this.s=s;}

	    public void run() {
		for(int i=1; i < 100; i++) {
		    s.Add(i, mm);
		    System.out.println("Add(" + i + ")");
		    try {
			sleep(100);
		    }
		    catch(Exception e) {
			System.err.println(e);
		    }
		}
	    }
	}



	class Acker extends Thread {
	    AckSenderWindow s;
	    Message         mm=new Message();
	    Random          rand=new Random(System.currentTimeMillis());
	    long            sleep_time=0;

	    public Acker(AckSenderWindow s) {this.s=s;}

	    public void run() {
		for(int i=1; i < 100; i++) {
		    System.out.println("ACK(" + i + ")");
		    s.Ack(i);
		    try {
			if(i > 50)
			    sleep_time=Math.abs(rand.nextLong() % 10) + 100;
			else
			    sleep_time=Math.abs(rand.nextLong() % 1000) + 100;
			System.out.println("Acker: sleeping for " + sleep_time);
			sleep(sleep_time);
		    }
		    catch(Exception e) {
			System.err.println(e);
		    }
		}
	    }
	}



	AckSenderWindow  sender=new AckSenderWindow(5, new Retr(), 0);
	Message          m=new Message();


	sender.Start();
	Thread adder=new Adder(sender);
	Thread acker=new Acker(sender);
	
	adder.start(); acker.start();
	try {
	    adder.join();
	    acker.join();
	}
	catch(Exception e) {
	    System.err.println(e);
	}
	sender.Stop();
    }

}
