
import java.util.*;
import JavaGroups.*;



/**
 * Example of a replicated quote server using JavaGroups and Ensemble (ejava). The server
 * maintains state which consists of a list of quotes and their corresponding values. When it
 * is started, it tries to reach other quote servers to get its initial state. If it does not
 * receive any response after 5 seconds, it assumes it is the first server and starts
 * processing requests. When it receives a view notification it checks whether there are more 
 * members in the view than in its previous view. If this is the case, it broadcasts a 
 * request for the state to all members. Integration of different states is simply the union
 * of all states (with the danger of overwriting mutually inconsistent state).<p>
 * This mechanism allows both for fast discovery of initial servers, and - in the case of
 * partition merging - for reintegration of existing servers. Broadcasting the state request
 * upon every view change (only when new members are joined) causes potentially a lot of 
 * network traffic, but it is assumes that there will not be more than 5 quote servers at most.
 */

public class QuoteServer implements MembershipListener {
    private Hashtable     stocks=new Hashtable();
    private Dispatcher    disp=new Dispatcher(new JChannelFactory(), null);
    private final String  channel_name="Quotes";
    private int           num_members=1;


    class ViewThread extends Thread {
	private int new_size=1;

	public ViewThread(int new_size) {this.new_size=new_size;}
	    
	public void run() {
	    try {
		if(new_size < num_members || new_size < 2) {
		    num_members=new_size;
		    return;
		}
		num_members=new_size;
		GetStateFromOthers();
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	}

    }


    

    protected synchronized void GetStateFromOthers() throws Exception {
	int        num_members=0;
	Hashtable  tmp_state;
	Vector     other_states;
	
	// Get current state (if available) from other QuoteServers:
	System.out.println("Trying to get state from other quote servers");
	num_members=disp.GetNumMembers(channel_name);
	other_states=disp.SendGetN(channel_name, 
				   "GetAllStocks", // name of method in QuoteServer
				   null,           // no arguments
				   2,              // expect 2 other members
				   5000);          // wait for 5 secs maximum
	
	if(other_states != null && other_states.size() > 0) {
	    // there were other QuoteServers around, integrate their state
	    for(int i=0; i < other_states.size(); i++) {
		tmp_state=(Hashtable)other_states.elementAt(i);
		Integrate(tmp_state);
	    }
	    System.out.println("Integrated state from " + other_states.size() + 
			       " other quote servers");
	    System.out.println("I have " + stocks.size() + " stocks");
	}
	else
	    System.out.println("No other quote servers were found");
    }
    

    private void Integrate(Hashtable state) {
	String key;	
	for(Enumeration e=state.keys(); e.hasMoreElements();) {
	    key=(String)e.nextElement();
	    stocks.put(key, state.get(key));  // just overwrite
	}
    }


    public void ViewAccepted(View new_view) {
	System.out.println("Accepted view (" + new_view.Size() + new_view.GetMembers() +
			   ")");

	ViewThread t=new ViewThread(new_view.Size());
	t.start();
    }

    public void Suspect(Object suspected_mbr) {}
    
    public void Block() {}


    public void Start() {
	disp.Join(channel_name, this);
	disp.SetMembershipListener(channel_name, this);

	System.out.println("\nQuote Server started at " + new Date());
	System.out.println("Joined channel '" + channel_name + "' (" + 
			   disp.GetNumMembers(channel_name) + " members)");

	try { 	    
	    GetStateFromOthers();
	}
	catch(Exception e) {
	    System.err.println(e);
	    return;
	}
	System.out.println("Ready to serve requests");
    }


    /* Quote methods: */

    public float GetQuote(String stock_name) throws Exception {
	System.out.print("Getting quote for " + stock_name + ": ");
	Float retval=(Float)stocks.get(stock_name);
	if(retval == null) {
	    System.out.println("not found");
	    throw new Exception("Stock " + stock_name + " not found");
	}
	System.out.println(retval.floatValue());
	return retval.floatValue();
    }

    public void SetQuote(String stock_name, Float value) {
	System.out.println("Setting quote for " + stock_name + ": " + value);
	stocks.put(stock_name, value);
    }


    public Hashtable GetAllStocks() {
	System.out.print("GetAllStocks: ");
	PrintAllStocks();
	return stocks;
    }
    
    public void PrintAllStocks() {
	System.out.println(stocks);
    }


    public static void main(String args[]) {
	QuoteServer server=new QuoteServer();
	server.Start();
    }

}
