
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.io.*;
import JavaGroups.*;



/**
 * Application shows virtual synchrony. A thread in each process randomly chooses square
 * and computes a random color (red/green/blue tuple) for it. It then multicasts the
 * square's location (x,y) and new color to all members of the group. When the message is
 * received (method call SetColor), the corresponding square is set to the new color and
 * repainted.<p>
 * Note that a process must not update its local grid before or after multicasting the update as
 * this could lead to inconsistencies ! Therefore, the Grid demo requires loopback to be
 * enabled, i.e. multicasts to a groups always have to also be received by its sender !
 */

public class GridSimple extends Frame implements WindowListener, ActionListener, Runnable {
    private Panel                  main_panel, button_panel, bottom_panel;
    private Button                 start_button, stop_button, check_button;
    private Label                  checksum_label=new Label("Checksum: ");
    private final Font             default_font=new Font("Helvetica",Font.PLAIN,12);
    private boolean                started=false;
    private int                    NUM=4;
    private ColorComp[][]          grid;
    private Thread                 mythread;
    private Random                 random=new Random(System.currentTimeMillis());
    private int                    sleeptime=100;  // ms


 
    public GridSimple(int num, int sleeptime) {
	NUM=num;
	this.sleeptime=sleeptime;
	grid=new ColorComp[NUM][NUM];    
	main_panel=new Panel();
	main_panel.setLayout(new GridLayout(NUM, NUM));
	button_panel=new Panel();
	bottom_panel=new Panel();
	bottom_panel.setLayout(new BorderLayout());
	start_button=new Button("Start");
	start_button.setFont(default_font);
	start_button.addActionListener(this);
	
	stop_button=new Button("Stop");
	stop_button.setFont(default_font);
	stop_button.addActionListener(this);
	
	check_button=new Button("Checksum");
	check_button.setFont(default_font);
	check_button.addActionListener(this);

	button_panel.add("South", start_button);
	button_panel.add("South", stop_button);
	button_panel.add("South", check_button);
	
	bottom_panel.add("Center", checksum_label);
	bottom_panel.add("South", button_panel);
	checksum_label.setFont(default_font);

	setLayout(new BorderLayout());
	add("Center", main_panel);
	add("South",  bottom_panel);
	SetStarted(false);
		
	for(int x=0; x < NUM; x++)
	    for(int y=0; y < NUM; y++) {
		grid[x][y]=new ColorComp(Color.blue);
		main_panel.add(grid[x][y]);
	    }
    }

    



    private Color SelectColor() {
	int red=(Math.abs(random.nextInt()) % 255);
	int green=(Math.abs(random.nextInt()) % 255);
	int blue=(Math.abs(random.nextInt()) % 255);
	return new Color(red, green, blue);
    }



    public void run() {
	Color new_color;
	int x, y;

	while(started) {
	    x=Math.abs(random.nextInt() % NUM);
	    y=Math.abs(random.nextInt() % NUM);
	    new_color=SelectColor();

	    try {
		SetColor(x, y, new_color);
		Thread.currentThread().sleep(sleeptime);
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	}
	SetStarted(false);
    }



	
    public void go() {
	try {
	    setSize(300,350);
 	    addWindowListener(this);
	    setVisible(true);
	    setTitle("Color Grid");
	    setBackground(Color.white);
	}
	catch(Exception e) {
	    System.err.println(e);
	    return;
	}
    }




    public static void main(String[] args) {
	GridSimple      grid=null;
	String          arg, next_arg;
	int             num=4, sleeptime=100;

	try {
	    for(int i=0; i < args.length; i++) {
		arg=args[i];
		if(arg.equals("-help")) {
		    System.out.println("GridSimple [-num <number>] [-sleep <time in ms>]");
		    return;
		}
		else if(arg.equals("-num")) {
		    num=new Integer(args[++i]).intValue();
		    continue;
		}		
		else if(arg.equals("-sleep")) {
		    sleeptime=new Integer(args[++i]).intValue();
		    continue;
		}
	    }
	}
	catch(Exception e) {
	    System.out.println("GridSimple [-num <number>] [-sleep <time (in ms)>]");
	    return;
	}

	grid=new GridSimple(num, sleeptime);
	grid.go();
    }





    public void SetColor(int x, int y, Color new_color) {
	synchronized(grid) {
	    grid[x][y].SetColor(new_color);	    
	}
    }


    public void Start() {
	if(started)
	    return;
	SetStarted(true);
	checksum_label.setText("Running");
	mythread=new Thread(this);	
	mythread.start();
    }


    void SetStarted(boolean flag) {
	started=flag;
	if(started) {
	    stop_button.setEnabled(true);
	    start_button.setEnabled(false);
	}
	else {
	    stop_button.setEnabled(false);
	    start_button.setEnabled(true);
	}
    }


    public void Stop() {
	if(started) {
	    SetStarted(false);
	    if(mythread != null) {
		try {
		    mythread.join();
		}
		catch(Exception e) {
		    System.err.println(e);
		}
		mythread.stop();
		mythread=null;
	    }	    
	}
	ComputeChecksum();
    }



    public void ComputeChecksum() {
	long   checksum=0;
	Color  col;
	synchronized(grid) {
	    for(int x=0; x < NUM; x++)
		for(int y=0; y < NUM; y++)
		    checksum+=grid[x][y].GetChecksum();
	}
	checksum_label.setText("Checksum: " + checksum);
    }



    public void windowActivated(WindowEvent e) {}
    public void windowClosed(WindowEvent e) {}
    public void windowClosing(WindowEvent e) {
	setVisible(false);
	System.exit(0);
    }
    public void windowDeactivated(WindowEvent e) {}
    public void windowDeiconified(WindowEvent e) {}
    public void windowIconified(WindowEvent e) {}
    public void windowOpened(WindowEvent e) {}


    public void actionPerformed(ActionEvent e) {
	String     command=e.getActionCommand();	
	if(command == "Start")
	    Start();
	else if(command == "Stop")
	    Stop();
	else if(command == "Checksum")
	    ComputeChecksum();
	else
	    System.out.println("Unknown action");
    }


}

