package JavaGroups;

import java.lang.reflect.Modifier;
import java.io.*;
import java.util.*;
import java.awt.*;



/**
 * A <code>MethodInvoker</code> can be used to register an object so that its methods
 * can be invoked from a client using a <a href=JavaGroups.channel.RemoteMethodCall.html>
 * RemoteMethodCall</a> object and is typically used in the server role. Sample code is
 * shown below:<p>
 * <pre>
 *
 *import java.util.*;
 *import JavaGroups.channel.*;<br>
 *
 *public class MethodInvokerTest {
 *    private Channel channel;
 *    private MethodInvoker inv;<br>
 *    
 *    public void Start() throws Exception {
 *        channel=new EnsChannel("MethodInvokerTest", null);
 *        channel.Connect(1000);
 *        <b>inv=new MethodInvoker(channel, this);</b>
 *    }<br>
 * 
 *    public Date <b>GetCurrentDate</b>() {return new Date();}<br>
 *
 *    public float <b>Divide</b>(Integer x, Integer y) {
 *        return (float)x.intValue()/y.intValue();
 *    }<br>
 *
 *    public void <b>ThrowMe()</b> throws Exception {
 *        System.out.println("ThrowMe(): throwing exception");
 *        throw new Exception("hello, this is a dummy exception from ThrowMe");
 *    }<br>
 *
 *    public long <b>GetCurrentTimeMillis</b>() {return System.currentTimeMillis();}<br>
 *
 *    public static void main(String args[]) {
 *        MethodInvokerTest t=new MethodInvokerTest();
 *        try {
 *	    t.Start();
 *        }
 *        catch(Exception e) {
 *            System.err.println(e);
 *        }
 *    }<br>
 *}
 * </pre>
 * <p>
 * The code first creates a channel object and connects to it. This allows clients which connect
 * to a channel with the same name to multicast method calls to all channels with the same name.
 * Then a <code>MethodInvoker</code> object is created. The first argument is a transport used
 * to receive messages and send responses. The second is the object on which the method calls 
 * will be invoked.<p>
 * Methods to be invoked by <code>MethodInvoker</code> have to have Objects because
 * primitive parameters (e.g. 'int') are currently not supported as Class objects for primitive 
 * types cannot be serialized. As consequence, methods to be invoked remotely (via 
 * <code>MethodInvoker</code>) have to have Objects as parameters, rather than
 * primitive types. Return values, however, can be primitive types.<p>
 * Note that a server role can at any time assume a client role by creating and sending remote
 * method calls.
 * @see RemoteMethodCall
 * @see Dispatcher
 */
public class MethodInvoker implements MessageListener {

    class TargetItem {
	Object    target=null;
	Hashtable funclets=null;

	TargetItem(Object target, String funclet_name, Object funclet) {
	    this.target=target;
	    if(funclet_name != null)
		RegisterFunclet(funclet_name, funclet);
	}

	void RegisterFunclet(String funclet_name, Object funclet) {
	    if(funclets == null)
		funclets=new Hashtable();
	    funclets.put(funclet_name, funclet);
	}

	void UnregisterFunclet(String funclet_name) {
	    if(funclets != null) funclets.remove(funclet_name);
	}

	Object GetFunclet(String funclet_name) {
	    return funclets != null ? funclets.get(funclet_name) : null;
	}

	public String toString() {
	    StringBuffer ret=new StringBuffer();
	    ret.append("[" + target);
	    if(funclets != null)
		ret.append(funclets);
	    ret.append("]");
	    return ret.toString();
	}
    }


    protected Transport        transport;
    protected PullPushAdapter  adapter=null;
    protected Vector           targets=new Vector();
    protected MethodLookup     method_lookup=null;
    protected boolean          hold_requests=false;
    protected Vector           request_queue=new Vector();



    private TargetItem FindTargetItem(Object target) {
	TargetItem item;
	for(int i=0; i < targets.size(); i++) {
	    item=(TargetItem)targets.elementAt(i);
	    if(target == item.target)
		return item;
	}
	return null;
    }





    /**
     * Creates a new MethodInvoker and (optionally) a PullPushAdapter and directs messages
     * received by the PullPushAdapter to the MethodInvoker.
     * @param t Transport over which to send responses
     * @param create_pull_push_adapter If true, a new instance of PullPushAdapter is created
     *                                 and messages received by it redirected to the 
     *                                 MethodInvoker. Setting this parameter to false allows to
     *                                 create one's own PullPushAdapter (to invoke Receive).
     */
    public MethodInvoker(Transport t, boolean create_pull_push_adapter) {
	transport=t;
	if(create_pull_push_adapter) {
	    adapter=new PullPushAdapter(transport);
	    adapter.SetListener(this);
	}
    }


    /**
     * Creates a new MethodInvoker and (optionally) a PullPushAdapter and directs messages
     * received by the PullPushAdapter to the MethodInvoker.
     * @param t Transport over which to send responses
     * @param target Adds the object to the MethodInvoker's target set. Any time a message is
     *               received, it will be dispatched (in the form of a method invocation)
     *               to all objects in the target set.
     * @param create_pull_push_adapter If true, a new instance of PullPushAdapter is created
     *                                 and messages received by it redirected to the 
     *                                 MethodInvoker. Setting this parameter to false allows to
     *                                 create one's own PullPushAdapter (to invoke Receive).
     */
    public MethodInvoker(Transport t, Object target, boolean create_pull_push_adapter) {
	this(t, create_pull_push_adapter);
	AddTargetObject(target);
    }

    /**
     * Allows to set a method resolution method which will be executed by the underlying
     * MethodCall object to resolve the method. If not set, MethodCall's internal lookup
     * method will be used.
     */
    public void SetMethodLookup(MethodLookup lookup) {method_lookup=lookup;}


    /**
     * Every request received after this call will be stored in a queue instead of being 
     * invoked on the target object(s). Method <code>ReleaseRequests</code> will invoke all
     * queued requests on the target object(s) and reset the queuing flag.
     */
    public synchronized void HoldRequests() {
	hold_requests=true;
    }


    /**
     * Removes all requests from the queue, invokes them on the target object(s) and resets the
     * queueing flag. After this call, no requests will be queued any more until a call to
     * <code>HoldRequests</code> is made.
     */
    public synchronized void ReleaseRequests() {
	synchronized(request_queue) {
	    for(int i=0; i < request_queue.size(); i++)
		Dispatch((Message)request_queue.elementAt(i));
	    request_queue.removeAllElements();
	    hold_requests=false;
	}
    }


    /**
     * A funclet handles requests on behalf of the target object. All methods (MethodCalls)
     * containing a non-empty <code>funclet_name</code> field will be handled by 
     * <code>funclet</code> instead of <code>target</code>. Both funclets and their methods
     * need to be public, otherwise an <code>IllegalAccessException</code> will be thrown at
     * runtime.
     */
    public void AddFunclet(Object target, String funclet_name, Object funclet) {
	
	if(funclet != null && !Modifier.isPublic(funclet.getClass().getModifiers())) {
	    System.err.println("MethodInvoker.AddFunclet(): cannot add funclet " + funclet_name +
			       " as it is not declared public !");
	    return;
	}

	synchronized(targets) {
	    TargetItem item=FindTargetItem(target);

	    if(item == null)
		targets.addElement(new TargetItem(target, funclet_name, funclet));
	    else
		item.RegisterFunclet(funclet_name, funclet);
	}
    }


    public void AddTargetObject(Object target) {
	TargetItem item;

	synchronized(targets) {
	    item=FindTargetItem(target);
	    if(item == null)
		targets.addElement(new TargetItem(target, null, null));
	}
    }


    public void RemoveTargetObject(Object target) {
	TargetItem item;

	synchronized(targets) {
	    item=FindTargetItem(target);
	    if(item != null)
		targets.removeElement(item);
	}
    }

    public void RemoveAllTargetObjects() {
	synchronized(targets) {
	    targets.removeAllElements();
	}
    }



    void Dispatch(Message msg) {
	MethodCall            method_call;
	ByteArrayInputStream  in_stream;
	ObjectInputStream     in;
	byte                  buf[]=null, rsp_buf[]=null;
	Object                response, target, sender=null;
	Message               rsp;
	Vector                arguments;
	TargetItem            target_item;
	String                funclet_name;

	buf=msg.GetBuffer();
	sender=msg.GetSrc();

	if(buf == null || buf.length < 1) {
	    System.err.println("MethodInvoker.Dispatch(): message buffer is empty");
	    return;
	}
	
	try {
	    method_call=(MethodCall)Util.ObjectFromByteBuffer(buf);
	}
	catch(Exception e) {
	    System.err.println(e);
	    return;
	}
	if(method_call == null) {
	    System.err.println("MethodInvoker.Dispatch(): failed to create MethodCall object");
	    return;
	}

	// Invoke method on all target objects and return each return using the transport
	for(int i=0; i < targets.size(); i++) {
	    target_item=(TargetItem)targets.elementAt(i);

	    if((funclet_name=method_call.GetFuncletName()) != null) {
		target=target_item.GetFunclet(funclet_name);
		if(target == null) {
		    System.err.println("MethodInvoker.Dispatch(): no funclet found for " +
				       "funclet_name " + funclet_name);
		    target=target_item.target;
		}
	    }
	    else
		target=target_item.target;
	    
	    response=method_call.Invoke(target, method_lookup);
	    if(msg.IsOneway())
		continue;

	    if(sender == null) {
		System.err.println("MethodInvoker.Dispatch(): cannot send response" +
				   " as sender address is null");
		continue;
	    }	    

	    if(response != null) {
		if(!(response instanceof Serializable)) {
		    System.err.println("MethodInvoker.Dispatch(): return value (type="+ 
				       response.getClass().getName()  +")cannot be "+ 
				       "serialized as it does not implement Serializable");
		    continue;
		}		
		try {
		    rsp_buf=Util.ObjectToByteBuffer((Serializable)response);
		}
		catch(Exception e) {
		    System.err.println(e);
		    continue;
		}
	    }
	    rsp=msg.MakeReply();
	    rsp.SetBuffer(rsp_buf);
	    try {
		transport.Send(rsp);
	    }
	    catch(Exception e) {
		System.err.println(e);
		continue;
	    }
	}	
    }


    public synchronized void Receive(Message msg) {
	if(hold_requests) {
	    request_queue.addElement(msg);
	    System.out.println("MethodInvoker.Receive(): queued message #" + msg.GetId());
	    return;
	}
	Dispatch(msg);
    }


    public static void main(String args[]) {
	String s1="Bela Ban", s2="Janet Fechner";
	Integer i1=new Integer(22), i2=new Integer(33);
	MethodInvoker m=new MethodInvoker(null, false);

	m.AddTargetObject(s2);
	m.AddFunclet(s1, "Foo", i1);
	m.AddFunclet(s1, "Bar", i2);

	m.RemoveTargetObject(s1);

	System.out.println(m.targets);
    }

}
