package JavaGroups.JavaStack.Protocols;

import java.io.*;
import java.util.*;
import JavaGroups.*;
import JavaGroups.Subject;
import JavaGroups.Observer;
import JavaGroups.Algorithms.*;
import JavaGroups.JavaStack.*;
import JavaGroups.Gui.*;




class StateTransferRequest implements Serializable {
    static final int MAKE_COPY    = 1;  // arg = originator of request
    static final int RETURN_STATE = 2;  // arg = orginator of request

    int    type=0;
    Object arg;


    public StateTransferRequest(int type, Object arg) {
	this.type=type; this.arg=arg;
    }

    public int    GetType() {return type;}
    public Object GetArg()  {return arg;}

    public String toString() {
	return "[StateTransferRequest: type=" + Type2Str(type) + ", arg=" + arg + "]";
    }

    static String Type2Str(int t) {
	switch(t) {
	case MAKE_COPY:     return "MAKE_COPY";
	case RETURN_STATE:  return "RETURN_STATE";
	default:            return "<unknown>";
	}
    }
}



/**
   State transfer layer. Upon receiving a GET_STATE event from JChannel, a MAKE_COPY message is 
   sent to all members. When the originator receives MAKE_COPY, it queues all messages to the 
   channel.
   When another member receives the message, it asks the JChannel to provide it with a copy of
   the current state (GetStateEvent is received by application, ReturnState() sends state down the
   stack). Then the current layer sends a unicast RETURN_STATE message to the coordinator, which
   returns the cached copy.
   When the state is received by the originator, the GET_STATE sender is unblocked with a
   GET_STATE_OK event up the stack (unless it already timed out).
 */

public class STATE_TRANSFER extends Protocol implements Transport, RequestHandler, Subject {
    boolean               trace=false;
    Object                local_addr=null;
    Vector                members=new Vector();
    Message               m=null;
    boolean               is_server=false;
    Object                cached_state=null;
    Integer               state_xfer_mutex=new Integer(0); // get state from appl (via channel).
    long                  timeout_get_appl_state=5000;
    long                  timeout_return_state=5000;
    Queue                 evt_queue=new Queue();
    boolean               queueing=false;
    RequestCorrelator     corr=null;
    Vector                observers=new Vector();
    QueueViewer           queue_viewer=null; //new QueueViewer("STATE_TRANSFER", up_queue, down_queue);



    public STATE_TRANSFER() {
	if(queue_viewer != null)
	    queue_viewer.show();
    }


    
    /** All protocol names have to be unique ! */
    public String  GetName() {return "STATE_TRANSFER";}


    
    
    public boolean SetProperties(Properties props) {
	String     str;
	
	this.props=props;
	str=props.getProperty("trace");
	if(str != null) {
	    trace=new Boolean(str).booleanValue();
	    props.remove("trace");
	}

	// Milliseconds to wait for application to provide requested state, events are
	// STATE_TRANSFER up and STATE_TRANSFER_OK down
	str=props.getProperty("timeout_get_appl_state");
	if(str != null) {
	    timeout_get_appl_state=new Long(str).longValue();
	    props.remove("timeout_get_appl_state");
	}

	// Milliseconds to wait for 1 or all members to return its/their state. 0 means wait
	// forever. States are retrieved using GroupRequest/RequestCorrelator
	str=props.getProperty("timeout_return_state");
	if(str != null) {
	    timeout_return_state=new Long(str).longValue();
	    props.remove("timeout_return_state");
	}

	if(props.size() > 0) {
	    System.err.println("STATE_TRANSFER.SetProperties(): the following " +
			       "properties are not recognized:");
	    props.list(System.out);
	    return false;
	}
	return true;
    }
    
    
    
    
    
    
    public void Up(Event evt) {
	Message              msg;
	Object               obj, sender;

	if(queue_viewer != null)
	    queue_viewer.Up(evt);

	
	switch(evt.GetType()) {

	case Event.START:
	    corr=new RequestCorrelator(GetName(), this, this, false);  // not multithreaded !
	    break;
	    
	case Event.BECOME_SERVER:
	    is_server=true;
	    break;

	case Event.SET_LOCAL_ADDRESS:
	    local_addr=evt.GetArg();
	    break;

	case Event.TMP_VIEW:
	case Event.VIEW_CHANGE:
	    Vector new_members=(Vector)((View)evt.GetArg()).GetMembers();
	    synchronized(members) {
		members.removeAllElements();
		if(new_members != null && new_members.size() > 0)
		    for(int k=0; k < new_members.size(); k++)
			members.addElement(new_members.elementAt(k));
	    }
	    break;
	    
	case Event.MSG:
	    msg=(Message)evt.GetArg();

	    if(trace) {
		// debugging code +++++++
		try {  
		    Object arg=Util.ObjectFromByteBuffer(msg.GetBuffer());
		    if(arg != null)
			System.out.println("STATE_TRANSFER.Up(): received " + arg);
		}
		catch(Exception e) {
		    System.out.println("TRANSFER.Up(exception): evt is " + evt);
		}
		// debugging code +++++++
	    }


	    if(corr.Receive(msg)) // correlator received it, we don't need to pass it up
		return;
	    break;                // else pass it up

	case Event.SUSPECT:
	    corr.Suspect(evt.GetArg());
	    break;

	} // end switch


	synchronized(evt_queue) {
	    if(queueing) {
		try {
		    if(trace) System.out.println("-- msg QUEUED");
		    evt_queue.Add(evt);
		}
		catch(QueueClosed closed) {
		    return;
		}
		return;             // do not pass up
	    }
	    else {
		if(queue_viewer != null) queue_viewer.PassUp(evt);
		PassUp(evt);            // Pass up to the layer above us	    
	    }
	}



    }




    public void Down(Event evt) {
	Integer        i;
	Object         coord, state;
	
	if(queue_viewer != null)
	    queue_viewer.Down(evt);

	switch(evt.GetType()) {

	case Event.STOP:
	    if(corr != null) {
		corr.Stop();
		corr=null;
	    }
	    break;

	case Event.TMP_VIEW:
	case Event.VIEW_CHANGE:
	    Vector new_members=(Vector)((View)evt.GetArg()).GetMembers();
	    synchronized(members) {
		members.removeAllElements();
		if(new_members != null && new_members.size() > 0)
		    for(int k=0; k < new_members.size(); k++)
			members.addElement(new_members.elementAt(k));
	    }
	    break;

	case Event.GET_STATE:       // generated by JChannel.GetState()
	    coord=DetermineCoordinator();
	    
	    if(coord == null || coord.equals(local_addr)) {
		if(trace) System.out.println("++Coordinator is null");
		
		if(queue_viewer != null) queue_viewer.PassUp(evt);
		PassUp(new Event(Event.GET_STATE_OK, null));

		synchronized(evt_queue) {
		    if(queueing) {
			if(trace) System.out.println("Stopping queueing");
			queueing=false;
			ReplayEventQueue();
		    }
		}
		return;             // don't pass down any further !
	    }

	    //System.out.println("Sending MAKE_COPY");
	    SendMakeCopyMessage();  // multicast MAKE_COPY to all members (we will receive it too !)
	    //System.out.println("MAKE_COPY -- Done");

	    i=(Integer)evt.GetArg();
	    if(i == null) i=new Integer(0);

	    state=i.intValue() == 1 ? GetStateFromOthers() : GetStateFromCoord();




	    /* Pass up the state to the application layer (insert into JChannel's event queue */
	    if(queue_viewer != null) queue_viewer.PassUp(new Event(Event.STATE_RECEIVED, state));
	    PassUp(new Event(Event.GET_STATE_OK, state));

	    /* Now stop queueing */
	    synchronized(evt_queue) {
		queueing=false;
		ReplayEventQueue();
	    }
	    return;                 // don't pass down any further !
	    
	case Event.GET_APPLSTATE_OK:
	    synchronized(state_xfer_mutex) {
		cached_state=evt.GetArg();
		state_xfer_mutex.notify();
	    }
	    return;                 // don't pass down any further !
	    
	}

 
	if(queue_viewer != null) queue_viewer.PassDown(evt);
	PassDown(evt);              // pass on to the layer below us
    }







    /* ---------------------- Interface RequestHandler -------------------------- */
    public Object Handle(Message msg) {
	StateTransferRequest req;

	try {
	    req=(StateTransferRequest)Util.ObjectFromByteBuffer(msg.GetBuffer());

	    switch(req.GetType()) {
	    case StateTransferRequest.MAKE_COPY:
		MakeCopy(req.GetArg());
		return null;
	    case StateTransferRequest.RETURN_STATE:
		return cached_state;
	    default:
		System.err.println("STATE_TRANSFER.Handle(): type " + req.GetType() + 
				   "is unknown in StateTransferRequest !");
		return null;
	    }
	}
	catch(Exception e) {
	    System.err.println("STATE_TRANSFER.Handle(): " + e);
	    return null;
	}
    }
    /* ------------------- End of Interface RequestHandler ---------------------- */




    
    /* ----------------------- Interface Transport ------------------------------ */

    public void Send(Message msg) throws Exception {


	if(queue_viewer != null) queue_viewer.PassDown(new Event(Event.MSG, msg));
	PassDown(new Event(Event.MSG, msg));
    }

    public Object Receive(long timeout) throws Exception {
	throw new Exception("STATE_TRANSFER.Receive(): should not be called !");
    }
    /* -------------------- End of Interface Transport -------------------------- */



    /* ------------------------- Interface Subject ------------------------------ */
    public void Attach(Observer o) {
	observers.addElement(o);
    }

    public void Detach(Observer o) {
	observers.removeElement(o);
    }

    public void Notify() {
	for(int i=0; i < observers.size(); i++)
	    ((Observer)observers.elementAt(i)).Update();
    }

    /* --------------------- End of Interface Subject ---------------------------- */







    Object GetStateFromCoord() {
	Vector                dests=new Vector();
	Message               msg;
	StateTransferRequest  r=new StateTransferRequest(StateTransferRequest.RETURN_STATE,
							 local_addr);
	RspList               rsp_list;
	Rsp                   rsp;
	Object                coord;
	GroupRequest          req;
	int                   num_tries=0;


	try {
	    msg=new Message(null, null, Util.ObjectToByteBuffer(r));
	}
	catch(Exception e) {
	    System.err.println("STATE_TRANSFER.GetStateFromCord(): " + e);
	    return null;
	}

	while(members.size() > 1 && num_tries++ < 3) {  // excluding myself
	    coord=DetermineCoordinator();
	    if(coord == null)
		return null;
	    msg.SetDest(coord);
	    dests.removeAllElements();
	    dests.addElement(coord);
	    req=new GroupRequest(msg, corr, dests, GroupRequest.GET_FIRST, timeout_return_state, 0);
	    req.Execute();
	    rsp_list=req.GetResults();
	    for(int i=0; i < rsp_list.size(); i++) {  // get the first non-suspected result
		rsp=(Rsp)rsp_list.elementAt(i);
		if(rsp.WasReceived())
		    return rsp.GetValue();
	    }
	    Util.Sleep(1000);
	}

	return null;
    }






    Vector GetStateFromOthers() {
	Vector                dests=new Vector();
	Message               msg;
	StateTransferRequest  r=new StateTransferRequest(StateTransferRequest.RETURN_STATE, 
							 local_addr);
	RspList               rsp_list;
	Vector                retval=new Vector();
	GroupRequest          req;


	for(int i=0; i < members.size(); i++)
	    if(!local_addr.equals(members.elementAt(i)))
		dests.addElement(members.elementAt(i));
		
	if(dests.size() == 0)
	    return null;
	
	msg=new Message();
	try {
	    msg.SetBuffer(Util.ObjectToByteBuffer(r));
	}
	catch(Exception e) {}

	req=new GroupRequest(msg, corr, dests, GroupRequest.GET_ALL, timeout_return_state, 0);
	req.Execute();
	rsp_list=req.GetResults();
	return rsp_list.GetResults();
    }





    void SendMakeCopyMessage() {
	GroupRequest           req;
	Message                msg=new Message();
	StateTransferRequest   r=new StateTransferRequest(StateTransferRequest.MAKE_COPY,
							  local_addr);
	Vector                 dests=new Vector();


	for(int i=0; i < members.size(); i++)  // don't add myself twice in dests !
	    if(!local_addr.equals(members.elementAt(i)))
		dests.addElement(members.elementAt(i));
	
	if(dests.size() == 0)
	    return;

	try {
	    msg.SetBuffer(Util.ObjectToByteBuffer(r));
	}
	catch(Exception e) {}

	req=new GroupRequest(msg, corr, dests, GroupRequest.GET_ALL, timeout_return_state, 0);
	req.Execute();
    }



    /** Return the first element of members which is not me. Otherwise return null. */
    Object DetermineCoordinator() {
	Object ret=null;
	if(members != null && members.size() > 1) {
	    for(int i=0; i < members.size(); i++)
		if(!local_addr.equals(members.elementAt(i)))
		    return members.elementAt(i);
	}
	return ret;
    }



    /**
       If server, ask application to send us a copy of its state (STATE_TRANSFER up,
       STATE_TRANSFER down). If client, start queueing events. Queuing will be stopped when
       state has been retrieved (or not) from single or all member(s).
     */
    void MakeCopy(Object sender) {
	if(sender.equals(local_addr)) { // was sent by us, has to start queueing
	    synchronized(evt_queue) {
		queueing=true;
	    }
	    if(trace) System.out.println("Started queuing");
	}
	else {               // only retrieve state from appl when not in client state anymore
	    if(is_server) {  // get state from application and store it locally
		synchronized(state_xfer_mutex) {
		    cached_state=null;


		    if(queue_viewer != null) 
			queue_viewer.PassUp(new Event(Event.GET_APPLSTATE, local_addr));
		    PassUp(new Event(Event.GET_APPLSTATE, local_addr));
		    try {
			state_xfer_mutex.wait(timeout_get_appl_state); // wait for STATE_TRANSFER_OK
		    }
		    catch(Exception e) {}
		}
	    }
	}
    }
    


    /**
       Remove all elements from queue and pass them up.
     */
    void ReplayEventQueue() {
	Event e;

	if(trace) System.out.println("Replaying event queue:");

	synchronized(evt_queue) {
	    try {




		while(evt_queue.Size() > 0) {
		    e=(Event)evt_queue.Remove();


		    if(queue_viewer != null) queue_viewer.PassUp(e);
		    PassUp(e);


		    if(trace) {
			// +++++++ debugging code
			System.out.println("Passed up queued event " + 
					   Event.Type2String(e.GetType()));
			
			if(e.GetType() == Event.MSG) {
			    try {
				Message m=(Message)e.GetArg();
				
				Object o=Util.ObjectFromByteBuffer(((Message)m).GetBuffer());
				System.out.println(o);
			    }
			    catch(Exception ex) {
				System.err.println(ex);
			    }
			}		    
			// +++++++ debugging code
		    }

		}





	    }
	    catch(QueueClosed closed) {
		return;
	    }
	}
    }


}
