package JavaGroups.JavaStack.Protocols;

import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;



/**
   Client part of GMS. Whenever a new member wants to join a group, it starts in the CLIENT role.
   No multicasts to the group will be received and processed until the member has been joined and
   turned into a SERVER (either coordinator or participant, mostly just participant). This class
   only implements <code>Join</code> (called by clients who want to join a certain group, and
   <code>ViewChange</code> which is called by the coordinator that was contacted by this client, to
   tell the client what its initial membership is.
 */
public class RpcClientGmsImpl extends RpcGmsImpl {
    static RpcClientGmsImpl  impl=null;
    Integer                  view_mutex=new Integer(0);
    Vector                   initial_mbrs=new Vector();
    Integer                  view_installation_mutex=new Integer(0);
    long                     expected_vid=-1;        // returned by JOIN, VIEW must match this one
    Object                   expected_creator=null;  // returned by JOIN, VIEW must match this one


    
    private RpcClientGmsImpl() {}

    public static RpcClientGmsImpl CreateInstance(RpcGMS gms) {
	if(impl == null)
	    impl=new RpcClientGmsImpl();
	impl.gms=gms;
	return impl;
    }





    /**
       Will generate a CONNECT_OK event. Determines the coordinator and sends a unicast 
       Join() message to it. If successful, we wait for a ViewChange (can time out). 
       If view change is received, impl is changed to an instance of RpcParticipantGmsImpl. 
       Otherwise, we continue trying to send Join() messages to	the coordinator, 
       until we succeed (or there is no member in the group. In this case, we create
       our own singleton group).
       @param mbr Our own address (assigned through SET_LOCAL_ADDRESS)
    */
    public void Join(Address mbr) {
	Object    coord=null;
	JoinRsp   join_rsp=null;

	while(true) {
	    FindInitialMembers();
	    if(initial_mbrs.size() < 1) {
		gms.view_id=new ViewId(mbr);       // create singleton view with mbr as only member
		gms.members.Add(mbr);
		gms.PassDown(new Event(Event.TMP_VIEW, 
				       gms.MakeView(gms.members.GetMembers(), gms.view_id)));
		gms.SetImpl(RpcCoordGmsImpl.CreateInstance(gms));           // become coordinator
		break;
	    }

	    coord=FindCoord(initial_mbrs);
	    if(coord == null) {
		System.err.println("RpcClientGmsImpl.Join(): could not determine coordinator " +
				   "from responses " + initial_mbrs);
		continue;
	    }

	    synchronized(view_installation_mutex) {
		expected_vid=-1;
		expected_creator=null;
		try {
		    System.out.println("--> Sending JOIN request to " + coord + ", timeout=" +
				       gms.join_timeout);
		    join_rsp=(JoinRsp)gms.CallRemoteMethod(coord, "HandleJoin", mbr, 
							   MessageProtocol.FIRST, gms.join_timeout);
		    System.out.println("JOIN_RSP is " + join_rsp);
		}
		catch(Timeout t) {
		    System.out.println("JOIN timed out !");
		    continue;
		}
		catch(Exception e) {
		    System.err.println("RpcClientGmsImpl.Join()" + e);
		    continue;
		}
		if(join_rsp == null) {
		    System.out.println("RpcClientGmsImpl.Join(): join response is null !");
		    Util.Sleep(gms.join_retry_timeout);
		    continue;
		}
		if(join_rsp.type == JoinRsp.JOIN_OK) {
		    View tmp_view=join_rsp.view;
		    if(tmp_view != null) {
			expected_vid=tmp_view.id;           // later ViewChange() must match this !
			expected_creator=tmp_view.creator;  // later ViewChange() must match this !
			gms.members.Set((Vector)join_rsp.view.GetMembers().clone());
			gms.Down(new Event(Event.TMP_VIEW, tmp_view));
		    }

		    // waits until new view is sent by coord, or timeout
		    try {
			view_installation_mutex.wait(gms.join_timeout);
		    }
		    catch(Exception e) {}

		    if(gms.view_id == null) {  // no view change was received
			System.err.println("RpcClientGmsImpl.Join(): JOIN_OK was " +
					   "received, but VIEW_CHANGE timed out ! Continuing... ");
			continue;
		    }
		    gms.SetImpl(RpcParticipantGmsImpl.CreateInstance(gms));  // become participant
		    break;
		}
		else if(join_rsp.type == JoinRsp.JOIN_FAILED) {
		    Util.Sleep(gms.join_retry_timeout);
		    continue;
		}
		else if(join_rsp.type == JoinRsp.ALREADY_JOINED) {
		    if(join_rsp.view != null) {
			gms.view_id=new ViewId((Address)join_rsp.view.GetCreator(), 
					       join_rsp.view.GetId());
			gms.members.Set(join_rsp.view.GetMembers());
			gms.PassDown(new Event(Event.VIEW_CHANGE, join_rsp.view));
			gms.PassUp(new Event(Event.VIEW_CHANGE, join_rsp.view));
		    }
		    break;
		}
		else {
		    System.out.println("RpcClientGmsImpl.Join(): join response " +
				       "type " + join_rsp.type + " not known !");
		    continue;
		}
	    } // end synchronized
	}  // end while
	gms.PassUp(new Event(Event.BECOME_SERVER));
	gms.PassDown(new Event(Event.BECOME_SERVER));
    }



    public void Leave(Address mbr) {
	WrongMethod("Leave");
	return;
    }



    public void Suspect(Address mbr) {
	//WrongMethod("Suspect");
	return;
    }



    public void Merge(Vector new_mbrs) {
	WrongMethod("Merge");
	return;
    }



    public JoinRsp HandleJoin(Address mbr) {
	WrongMethod("HandleJoin");
	return new JoinRsp(JoinRsp.JOIN_FAILED, null);  // I'm a client, can't handle Join() requests !	
    }



    /** Returns false. Clients don't handle Leave() requests */
    public boolean HandleLeave(Address mbr) {
	WrongMethod("HandleLeave");
	return false;
    }



    public void HandleViewChange(ViewId new_view, Vector mems) {
	synchronized(view_installation_mutex) {

	    if(expected_vid == -1 || expected_creator == null)
		return;

	    if(new_view.GetId() == expected_vid && expected_creator.equals(new_view.GetCoordAddress())) {
		gms.InstallView(new_view, mems);
		view_installation_mutex.notify();
	    }
	    else {
		System.err.println("RpcClientGmsImpl.ViewChange(): view did not match expected view:" +
				   " expected_vid=" + expected_vid + ", new_view_id=" +
				   new_view.GetId());
	    }
	}	
    }




    /** Returns immediately. Clients don't handle Merge() requests */
    public void HandleMerge(Vector new_mems, Vector suspects, long other_ltime) {
	WrongMethod("HandleMerge");
	return;
    }



    /** Returns immediately. Clients don't handle Suspect() requests */
    public void HandleSuspect(Address mbr) {
	WrongMethod("HandleSuspect");
	return;
    }



    /* We don't need to handle down events */
    //public boolean HandleDownEvent(Event evt) {}



    public boolean HandleUpEvent(Event evt) {
	Vector tmp;

	switch(evt.GetType()) {

	case Event.FIND_INITIAL_MBRS_OK:
	    tmp=(Vector)evt.GetArg();

	    System.out.println("FIND_OK: members are " + tmp);
	    
	    synchronized(initial_mbrs) {
		if(tmp != null && tmp.size() > 0)
		    for(int i=0; i < tmp.size(); i++)
			initial_mbrs.addElement(tmp.elementAt(i));
		initial_mbrs.notify();
	    }
	    return false;  // don't pass up the stack
	}

	return true;
    }





    /* --------------------------- Private Methods ------------------------------------ */


    




    /**
       Pings initial members. Removes self before returning vector of initial members. 
       Uses IP multicast or gossiping, depending on parameters.
     */
    void FindInitialMembers() {
	PingRsp ping_rsp;

	synchronized(initial_mbrs) {
	    initial_mbrs.removeAllElements();
	    gms.PassDown(new Event(Event.FIND_INITIAL_MBRS));
	    try {initial_mbrs.wait(gms.initial_mbrs_timeout);}
	    catch(Exception e) {}

	    for(int i=0; i < initial_mbrs.size(); i++) {
		ping_rsp=(PingRsp)initial_mbrs.elementAt(i);
		if(ping_rsp.own_addr != null && gms.local_addr != null && 
		   ping_rsp.own_addr.equals(gms.local_addr)) {
		    initial_mbrs.removeElementAt(i);
		    break;
		}
	    }
	}
    }



    Object FindCoord(Vector mbrs) {
	PingRsp mbr;
	if(mbrs == null || mbrs.size() < 1)
	    return null;

	for(int i=0; i < mbrs.size(); i++) {
	    mbr=(PingRsp)mbrs.elementAt(i);
	    if(mbr.coord_addr != null)
		return mbr.coord_addr;
	}
	return null;
    }

    
}
