package JavaGroups.JavaStack.Protocols;

import java.io.*;
import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;




class MnakHeader implements Serializable {
    public long    seq_no=0 ;               // sequence number of message (retransmit_no is -1)
    public long    retransmit_no=0 ;        // seq_no requested to be resent (seq_no is -1)
    public Object  sender=null;             // the sender of the message in case of a retransmit


    /** Either seq_no or retransmit_no has to be 0 */
    public MnakHeader(long seq_no, long retransmit_no, Object sender) {
	this.seq_no=seq_no;
	this.retransmit_no=retransmit_no;
	this.sender=sender;
    }

    public String toString() {
	StringBuffer ret=new StringBuffer();
	ret.append("[MNAK: ");
	ret.append("seq_no=" + seq_no + ", retransmit_no=" + retransmit_no + ", sender=" + sender);
	ret.append("]");
	return ret.toString();
    }
}



class MsgEntry {
    public long     seq_no=0;
    public Message  msg=null;

    public MsgEntry(long seq_no, Message msg) {
	this.seq_no=seq_no; this.msg=msg;
    }
}



public class MNAK extends Protocol implements SlidingWindow.RetransmitCommand {
    Hashtable  msgs_sent=new Hashtable();            // seq_no       -- Message
    Hashtable  msgs_received=new Hashtable();        // src (Object) -- SlidingWindow
    Hashtable  msgs_received_trash=new Hashtable();  // src (Object) -- Vector of MsgEntry's
    long       my_seq_no=1;                          // start with 1
    boolean    trace=false;
    Object     local_addr=null;
    

    /** 
	Method called by sliding window when it need to 'fill' a gap between sequence numbers.
	@param retransmit_no The sequence number of the message that is missing
	@param sender The original sender of the message. This and <code>retransmit_no</code> identify
	              every message uniquely.
	@param mcast_to_group Mcast (instead of ucast) the message to all group members. 
	                      This happens when 2 successive ucasts were not successful
	@param suspect Suspect <code>sender</code>. This happens when even the mcast was not 
	               successful. (no group member had a copy of the message). Highly unlikely !
     */
    public synchronized void Retransmit(long retransmit_no, Object sender,
					boolean mcast_to_group, boolean suspect) {
	if(suspect && sender != null) {
	    PassUp(new Event(Event.SUSPECT, sender));
	    return;
	}	

	Object   dest=mcast_to_group ? null : sender;
	Message  retransmit_req=new Message(dest, null, null); // message has no data, just 1 header

	retransmit_req.AddHeader(new MnakHeader(0, retransmit_no, sender));
	PassDown(new Event(Event.MSG, retransmit_req));	
    }
    

    /**
       Find message with sequence number <code>retransmit_no</code> in table msgs_sent
       if <code>sender</code> is null (that means, we sent it). Otherwise (<code>sender</code> 
       is not null), look up the message in <code>msgs_received_trash</code>. The second case happens
       when a receiver, after unsuccessful pt2pt retransmit request (maybe the original sender failed),
       mcasts the retransmit request to all members of the group.
       @param retransmit_no The sequence number of the missing message
       @param sender The original sender of the message. Together with <code>retransmit_no</code>, this
                     idntifies the message to be retransmitted uniquely.
       @param return_addr The address to which the message to be retransmitted should be sent
     */
    void RetransmitMessage(long retransmit_no, Object sender, Object return_addr) {
	Message      msg;
	MnakHeader   hdr;
	Vector       val;
	MsgEntry     entry;

	if(sender == null || sender.equals(local_addr)) {         // it was sent by us
	    msg=(Message)msgs_sent.get(new Long(retransmit_no));
	    if(msg == null) {
		System.err.println("MNAK.RetransmitMessage(): sequence number " + retransmit_no +
				   " not found !");
		return;
	    }

	    // Return address is already in the message as we're just resending it to the same dest !

	    PassDown(new Event(Event.MSG, msg));
	    return;	    
	}
	else {  // lookup in the trash ...
	    val=(Vector)msgs_received_trash.get(sender);  // lookup sender
	    if(val == null) {
		System.err.println("MNAK.RetransmitMessage(): sequence number " + retransmit_no +
				   " from sender '" + sender + "' not found !");
		return;
	    }
	    // we have a sender, now let's look for the sequence number...
	    for(int i=0; i < val.size(); i++) {
		entry=(MsgEntry)val.elementAt(i);
		if(entry.seq_no == retransmit_no) {
		    if(entry.msg != null) {			
			Message copy=entry.msg.Copy();
			copy.SetDest(return_addr);
			copy.SetSrc(sender);  // do as if 'sender' sent the message instead of us !
  			                      // UDP won't override this !
			copy.AddHeader(new MnakHeader(retransmit_no, 0, null));
			PassDown(new Event(Event.MSG, copy));			
			return;
		    }
		    break;
		}
	    }
	}
    }

    
    void AddToTrash(Object sender, long seq_no, Message msg) {
	MsgEntry  entry;
	Vector    val;

	val=(Vector)msgs_received_trash.get(sender);
	if(val == null) {
	    val=new Vector();
	    msgs_received_trash.put(sender, val);
	}
	val.addElement(new MsgEntry(seq_no, msg));
    }


    public String  GetName() {return "MNAK";}


    /** Setup the Protocol instance acording to the configuration string */
    public boolean SetProperties(Properties props) {
	String     str;
	this.props=props;
	str=props.getProperty("trace");
	if(str != null) {
	    trace=new Boolean(str).booleanValue();
	    props.remove("trace");
	}
	if(props.size() > 0) {
	    System.err.println("MNAK.SetProperties(): the following properties are not recognized:");
	    props.list(System.out);
	    return false;
	}
	return true;
    }



    /** Just remove if you don't need to reset any state */
    public void Reset() {
	Object         key;
	SlidingWindow  val;
    
	synchronized(msgs_sent) {
	    msgs_sent.clear();
	}

	synchronized(msgs_received) {
	    for(Enumeration e=msgs_received.keys(); e.hasMoreElements();) {
		key=e.nextElement();
		val=(SlidingWindow)msgs_received.get(key);
		if(val != null)
		    val.Stop();  // stop sliding window thread
	    }

	    msgs_received.clear();
	}

	synchronized(msgs_received_trash) {
	    msgs_received_trash.clear();
	}
	    


    }



    /**
       Remove the MnakHeader. The message might be a ucast message, in which case we just pass it
       up, or an mcast message. In the latter case, it could be a normal message, or a retransmit 
       request. In the first case, we stick it into a sliding window table (one per sender) and try
       to remove as many message as possible from it (until 'null' is returned). The sliding window
       table will call us when it requires a missing message (to fill a gap); we then have to
       send a retransmit request. Whenever a message is removed from the sliding window table, we 
       keep it in case someone asks us to retransmit it later on (we still have to write a STABLE 
       layer that discards 'old' messages). In the second case (retransmit request), we lookup the
       message either among the ones we sent ourself, or in the messages we received. If found, we
       just send it.
     */
    public void Up(Event evt) {
	MnakHeader     hdr;
	Message        msg, tmp_msg=null;
	SlidingWindow  win;
	Object         sender;


	if(evt.GetType() == Event.SET_LOCAL_ADDRESS) {
	    local_addr=evt.GetArg();
	    PassUp(evt);
	    return;
	}


	if(evt.GetType() == Event.MSG) {
	    msg=(Message)evt.GetArg();
	    hdr=(MnakHeader)msg.RemoveHeader();

	    if(hdr.seq_no == 0 && hdr.retransmit_no == 0) {  // ucast message
		PassUp(evt);
		return;
	    }

	    if(hdr.seq_no > 0) {                             // mcast message
		
		sender=msg.GetSrc();
		if(sender == null) {
		    System.err.println("MNAK.Up(): sender's address is null !");
		    return;
		}

		win=(SlidingWindow)msgs_received.get(sender);  // get the sliding window for the sender
		if(win == null) {                              // create one if not yet present
		    win=new SlidingWindow(sender, this);
		    msgs_received.put(sender, win);
		}
		win.Add(hdr.seq_no, msg);
		
		while((tmp_msg=win.Remove()) != null) {
		    AddToTrash(sender, hdr.seq_no, tmp_msg);
		    PassUp(new Event(Event.MSG, tmp_msg));
		}
		return;
		
	    }
	    else if(hdr.retransmit_no > 0) {                 // retransmit message
		RetransmitMessage(hdr.retransmit_no, hdr.sender, msg.GetSrc());
		return; // don't pass up further in the stack
	    }
	}

	PassUp(evt); // Pass up to the layer above us
    }






    /**
       Tag every message that is a multicast message (dest is null) with an MnakHeader, containing
       a sequence number.
     */
    public void Down(Event evt) {
	MnakHeader hdr;
	Message    msg;

	if(evt.GetType() == Event.STOP) {
	    Reset();
	    PassDown(evt);
	    return;
	}

	if(evt.GetType() == Event.MSG) {
	    msg=(Message)evt.GetArg();
	    if(msg.GetDest() == null) {                     // mcast message
		msgs_sent.put(new Long(my_seq_no), msg);    // store sent message (for retransmission)
		hdr=new MnakHeader(my_seq_no++, 0, null);
	    }
	    else                                            // ucast message (handled by NAK below)
		hdr=new MnakHeader(0, 0, null);
	    
	    msg.AddHeader(hdr);
	}

	PassDown(evt);  // Pass on to the layer below us
    }


}
