package JavaGroups;

import java.io.*;
import java.util.*;
import JavaGroups.JavaStack.*;


/**
   JChannel is a pure Java implementation of Channel.
 */
public class JChannel extends Channel {
    private String               channel_properties="UDP:PING:FD:GMS";
    private Object               my_addr=null;
    private String               group_addr=null;  // group name
    private View                 my_view=null;
    private Queue                mq=new Queue();
    private ProtocolStack        prot_stack=null;
    private Integer              flush_mutex=new Integer(0);
    private Integer              connect_mutex=new Integer(0);
    private Integer              disconnect_mutex=new Integer(0);
    private Integer              get_state_mutex=new Integer(0);
    private long                 timeout=5000;
    private boolean              receive_views=true;
    private boolean              receive_suspects=true;
    private boolean              receive_blocks=false;
    private boolean              receive_local_msgs=true;
    private boolean              receive_get_states=false;    
    private boolean              connected=false;
    private boolean              closed=false;      // Close has been called, channel is unusable
    private Serializable         state=null;


    private void CheckNotConnected() throws ChannelNotConnected {
	if(!connected)
	    throw new ChannelNotConnected();
    }

    private void CheckClosed() throws ChannelClosed {
	if(closed)
	    throw new ChannelClosed();
    }



    protected JChannel(boolean dummy) throws ChannelException {
	
    }


    public JChannel() throws ChannelException {
	this(null);
    }


    public JChannel(Object properties) throws ChannelException {
	if(properties != null) {
	    if(properties instanceof String)
		channel_properties=(String)properties;
	    else
		throw new ChannelException("JChannel: properties must be of type String");
	}
	prot_stack=new ProtocolStack(this, channel_properties);

	/* Setup protocol stack (create layers, queues between them */
	try {
	    prot_stack.Setup();
	}
	catch(Exception e) {
	    throw new ChannelException("JChannel(): " + e.getMessage());
	}
    }





    /** 
	Connects the channel to a group.
	@param group_address A <code>String</code> denoting the group name.
    */

    public synchronized void Connect(Object group_address) throws ChannelClosed {

	CheckClosed();

	if(connected) {
	    System.err.println("JChannel.Join(" + group_address + "): already connected !");
	    return;
	}

	if(!connected) {
	    if(group_address == null || !(group_address instanceof String)) {
		System.err.println("JChannel.Connect(): group address must be of type String !");
		return;
	    }	    
	    else
		this.group_addr=(String)group_address;

	    try {                      // starts all layers (e.g. sets local address) --> START
		prot_stack.Start();    // bottom layer (UDP) bounces back a START_OK
	    }
	    catch(Exception e) {
		System.err.println("JChannel.Connect(): " + e);
		return;
	    }
	    

 	    Vector t=new Vector();
 	    t.addElement(my_addr);
 	    my_view=new View(my_addr, 0, t);  // create a dummy view

	    Event connect_event=new Event(Event.CONNECT, group_address);
	    Down(connect_event);
	    
	    /* Wait for notification that the channel has been connected to the group */
	    synchronized(connect_mutex) {             // wait for CONNECT_OK event
		try {
		    // connect_mutex.wait(timeout);
		    connect_mutex.wait();
		}
		catch(Exception e) {}
	    }
	    connected=true;
	}
    }






    public synchronized void Disconnect() {
	if(closed) return;
	    
	if(connected) {
	    
	    /* Send down a DISCONNECT event. The DISCONNECT event travels down to the GMS, where a
	       DISCONNECT_OK response is generated and sent up the stack. JChannel blocks until a
	       DISCONNECT_OK has been received, or until timeout has elapsed.
	    */

	    Event disconnect_event=new Event(Event.DISCONNECT, my_addr);

	    synchronized(disconnect_mutex) {
		try {
		    Down(disconnect_event);          // DISCONNECT is handled by each layer
		    disconnect_mutex.wait(timeout);  // wait for DISCONNECT_OK event
		}
		catch(Exception e) {
		    System.err.println("JChannel.Disconnect(): " + e);
		}
	    }
	    connected=false;

	    try {
		prot_stack.Stop();  // sends down a STOP event, bottom layer (UDP) bounces back STOP_OK
	    }
	    catch(Exception e) {
		System.err.println("JChannel.Disconnect(): " + e);
	    }
	}
    }






    /** Destroys the channel. After this method has been called, the channel us unusable. */
    public synchronized void Close() {
	if(closed)
	    return;

	/* Send down a FLUSH event. The FLUSH event travels down to the bottom-most layer
	   which flushes outstanding messages and then sends a FLUSH_OK up the stack.  Each
	   layer has to send outstanding messages before passing the FLUSH event further
	   down.  
	*/

	Event flush_event=new Event(Event.FLUSH);

	synchronized(flush_mutex) {         // Flush the stack: all outstanding msgs will be sent
	    try {
		Down(flush_event);          // FLUSH will be handled by each layer
		flush_mutex.wait();         // wait for FLUSH_OK event (generated by bottom layer)
	    }
	    catch(Exception e) {
		System.err.println("JChannel.Close(): " + e);
	    }
	}

	try {
	    mq.Close(false);                // closes and removes all messages
	}
	catch(Exception e) {
	    System.err.println(e);
	}
	
	Disconnect();                       // leave group if connected
	
	if(prot_stack != null) {
	    try {
		prot_stack.Destroy();
	    }
	    catch(Exception e) {
		System.err.println("JChannel.Close(): " + e);
	    }
	    prot_stack=null;
	}
      	closed=true;
    }
    



    public synchronized void finalize() {
	try {
	    Close();
	}
	catch(Exception e) {
	    System.err.println("JChannel.finalize(): " + e);
	}
    }
    





    public void Send(Message msg) throws ChannelNotConnected, ChannelClosed {
	CheckClosed();
	CheckNotConnected();
	Down(new Event(Event.MSG, msg));
    }




	
	
    public Object Receive(long timeout) throws ChannelNotConnected, ChannelClosed, Timeout {
	Message     retval=null;
	Event       evt;

	CheckClosed();
	CheckNotConnected();

	try {
	    evt=(timeout <= 0)? (Event)mq.Remove() : (Event)mq.Remove(timeout);
	    return GetEvent(evt);
	}
	catch(QueueClosed queue_closed) {
	    throw new ChannelClosed();
	}
	catch(Timeout t) {
	    throw t;
	}
	catch(Exception e) {
	    System.err.println("JChannel.Receive(): " + e);
	    return null;
	}
    }




    /** Just peeks at the next message, view or block. Does <em>not</em> install new view if
	view is received */
    public Object Peek(long timeout) throws ChannelNotConnected, ChannelClosed, Timeout {
	Message    retval=null;
	Event      evt;

	CheckClosed();
	CheckNotConnected();

	try {
	    evt=(timeout <= 0)? (Event)mq.Peek() : (Event)mq.Peek(timeout);
	    return GetEvent(evt);
	}
	catch(QueueClosed queue_closed) {
	    System.err.println("JChannel.Peek(): " + queue_closed);
	    return null;
	}
	catch(Timeout t) {
	    return null;
	}
	catch(Exception e) {
	    System.err.println("JChannel.Peek(): " + e);
	    return null;
	}
    }



    private Object GetEvent(Event evt) {
	if(evt == null)
	    return null; // correct ?
	
	switch(evt.GetType()) {
	case Event.MSG:
	    return evt.GetArg();
	case Event.VIEW_CHANGE:
	    return evt.GetArg();
	case Event.SUSPECT:
	    return new SuspectEvent(evt.GetArg());
	case Event.BLOCK:
	    return new BlockEvent();
	case Event.GET_APPLSTATE:
	    return new GetStateEvent(evt.GetArg());
	case Event.STATE_RECEIVED:
	    return new SetStateEvent(evt.GetArg());
	default:
	    System.err.println("JChannel.ReturnEvent(): event is neither message " + 
			       "nor view nor block !");
	    return null;
	}
    }



    public View    GetView() {
	return closed || !connected ? null : my_view;
    }


    
    public Object  GetLocalAddress() {
	return closed ? null : my_addr;
    }


    public Object  GetGroupAddress()  {
	return closed ? null : !connected ? null : group_addr;
    }



    public void    SetOpt(int option, Object value) {
	if(closed) {
	    System.err.println("JChannel.SetOpt(): channel is closed; option not set !");
	    return;
	}

	switch(option) {
	case VIEW:
	    if(value instanceof Boolean)
		receive_views=((Boolean)value).booleanValue();
	    else
		System.err.println("JChannel.SetOpt(" + option + ", " + value + "): value has " +
				   "to be Boolean");
	    break;
	case SUSPECT:
	    if(value instanceof Boolean)
		receive_suspects=((Boolean)value).booleanValue();
	    else
		System.err.println("JChannel.SetOpt(" + option + ", " + value + "): value has " +
				   "to be Boolean");
	    break;
	case BLOCK:
	    if(value instanceof Boolean)
		receive_blocks=((Boolean)value).booleanValue();
	    else
		System.err.println("JChannel.SetOpt(" + option + ", " + value + "): value has " +
				   "to be Boolean");
	    if(receive_blocks)
		receive_views=true;
	    break;

	case GET_STATE_EVENTS:
	    if(value instanceof Boolean)
		receive_get_states=((Boolean)value).booleanValue();
	    else
		System.err.println("JChannel.SetOpt(" + option + ", " + value + "): value has " +
				   "to be Boolean");
	    break;


	case LOCAL:
	    if(value instanceof Boolean)
		receive_local_msgs=((Boolean)value).booleanValue();
	    else
		System.err.println("JChannel.SetOpt(" + option + ", " + value + "): value has " +
				   "to be Boolean");
	    break;
	default:
	    System.err.println("JChannel.SetOpt(" + option + ", " + value + "): option not known");
	    break;
	}
    }



    public Object  GetOpt(int option) {
	if(closed)
	    return null;

	switch(option) {
	case VIEW:
	    return new Boolean(receive_views);
	case BLOCK:
	    return new Boolean(receive_blocks);
	case SUSPECT:
	    return new Boolean(receive_suspects);
	case GET_STATE_EVENTS:
	    return new Boolean(receive_get_states);
	case LOCAL:
	    return new Boolean(receive_local_msgs);
	default:
	    System.err.println("JChannel.GetOpt(" + option + "): option not known");
	    return null;
	}
    }



    public void    BlockOk() {
	
    }


    /**
       Retrieves the current group state. Sends GET_STATE event down to STATE_TRANSFER layer.
       Blocks until STATE_TRANSFER sends up a GET_STATE_OK event or until <code>timeout</code>
       milliseconds have elapsed. The argument of GET_STATE_OK should be a single object.
     */
    public boolean GetState(long timeout) {
	return _GetState(new Event(Event.GET_STATE, new Integer(0)), timeout);
    }


    /**
       Retrieves the current group state. Sends GET_STATE event down to STATE_TRANSFER layer.
       Blocks until STATE_TRANSFER sends up a GET_STATE_OK event or until <code>timeout</code>
       milliseconds have elapsed. The argument of GET_STATE_OK should be a vector of objects.
    */
    public boolean GetAllStates(long timeout) {
	return _GetState(new Event(Event.GET_STATE, new Integer(1)), timeout);
    }



    public void ReturnState(Serializable state) {
	Down(new Event(Event.GET_APPLSTATE_OK, state));
    }




    private boolean _GetState(Event evt, long timeout) {
	synchronized(get_state_mutex) {
	    state=null;
	    Down(evt);
	    try {
		get_state_mutex.wait(timeout);  // notified by GET_STATE_OK event
	    }
	    catch(Exception e) {}

	    if(state != null)                   // 'state' set by GET_STATE_OK event
		return true;
	    else
		return false;
	}
    }








    private void HandleUpEvent(Event evt) {
	switch(evt.GetType()) {
	case Event.CONNECT_OK:
	    synchronized(connect_mutex) {
		connect_mutex.notify();
	    }
	    break;
	case Event.DISCONNECT_OK:
	    synchronized(disconnect_mutex) {
		disconnect_mutex.notify();
	    }
	    break;
	case Event.FLUSH_OK:
	    synchronized(flush_mutex) {
		flush_mutex.notify();
	    }
	    break;

	case Event.GET_STATE_OK:
	    try {
		mq.Add(new Event(Event.STATE_RECEIVED, evt.GetArg()));
	    }
	    catch(Exception e) {}

	    synchronized(get_state_mutex) {
		state=(Serializable)evt.GetArg();
		get_state_mutex.notify();
	    }
	    break;

	case Event.SET_LOCAL_ADDRESS:
	    my_addr=evt.GetArg();
	    break;

	default:
	    break;
	}
    }




    /**
       Called by the ProtocolStack when a message is received. It will be added to the message
       queue from which subsequent <code>Receive</code>s will dequeue it.
     */
    public void Up(Event evt) {
	int        type=evt.GetType();
	Message    msg;

	if(type != Event.MSG      && type != Event.VIEW_CHANGE && 
	   type != Event.SUSPECT  && type != Event.BLOCK &&
	   type != Event.GET_APPLSTATE) { // normal Event
	    HandleUpEvent(evt);
	    return;
	}

	if(mq == null) {
	    System.err.println("JChannel.Up(): message queue is null");
	    return;
	}

	switch(type) {

	case Event.MSG:
	    msg=(Message)evt.GetArg();
	    if(!receive_local_msgs) {  // discard local messages (sent by myself to me)
		if(my_addr != null && msg.GetSrc() != null)
		    if(my_addr.equals(msg.GetSrc()))
			return;
	    }
	    break;

	case Event.VIEW_CHANGE:
	    if(!receive_views)  // discard if client has not set receving views to on
		return;
	    if(connected == false)
		my_view=(View)evt.GetArg();
	    break;

	case Event.SUSPECT:
	    if(!receive_suspects)
		return;
	    break;

	case Event.GET_APPLSTATE:  // return the application's state
	    if(!receive_get_states) {  // if not set to handle state transfers, send null state
		Down(new Event(Event.GET_APPLSTATE_OK, null));
		return;
	    }
	    break;

	case Event.BLOCK:
	    if(!receive_blocks)  // discard if client has not set receving views to on
		return;
	    break;

	default:
	    System.err.println("JChannel.Up(): event type " + type + " not known");
	    return;
	} // end switch

	
	try {
	    mq.Add(evt);
	}
	catch(Exception e) {
	    System.err.println("JChannel.Up(): " + e);
	}
    }



    public void Down(Event evt) {
	if(prot_stack != null)
	    prot_stack.Down(evt);
	else
	    System.err.println("JChannel.Down(): no protocol stack available !");
    }







}
